subroutine uv_removes_clean(nv,visi,ouv,nc,mic,dcct,freq,first,last,subtract)
  use image_def
  use gbl_message
  use imager_interfaces, only : ompget_inner_threads
  !$ use omp_lib
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER -- Support for UV_RESTORE and similar
  !
  !     Compute visibilities for a UV data set according to a
  !     set of Clean Components , and remove them from the original
  !     UV table.
  !
  !   Slow version using machine  Sin and Cos function.
  !
  !   A semi-Slow version using interpolation from pre-tabulated
  !   Sin/Cos can be implemented, but would still need to be optimized
  !   It seems the Modulo function costs a lot, not just the Sin & Cos.
  !!
  !-----------------------------------------------------------------
  integer, intent(in) :: nv           !! Number of visibilities
  real, intent(in) :: visi(:,:)       !! UV data set
  integer, intent(in) :: nc           !! Number of channels
  integer, intent(in) :: mic(:)       !! Number of Clean Components
  real, intent(out) :: ouv(:,:)       !! Extracted UV data set
  real, intent(in) :: dcct(:,:,:)     !! Clean component
  real(8), intent(in) :: freq         !! Apparent observing frequency
  integer, intent(in) :: first        !! First channel
  integer, intent(in) :: last         !! Last channel
  logical, intent(in) :: subtract     !! Compute Diff or Model
  !
  ! Constants
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: clight = 299792458.d-6   ! in meter and MHz units
  real(8), parameter :: f_to_k=2.d0*pi/clight    !
  !
  ! Local ---
  real(8) :: cst
  real(8) :: pidsur            ! 2*pi/Lambda  to convert D (in m) to wave numbers
  real(8) :: phase, cp, sp
  real(4) :: rvis, ivis, value
  integer ii, ir, iw
  integer oi, or, ow
  integer ic,iv,jc,kc
  !
  real :: cc_size(20), cc_count(20), cc_flux(20)
  real :: gauss, q2
  logical :: do_size
  integer :: no, icmax, scc, ncc_size
  integer :: iplane, j, jsize
  integer :: nthread
  !
  ! Code ---
  scc = size(dcct,1)      
  pidsur = f_to_k * freq
  !
  if (scc.ne.3) then
    ncc_size = 0
    do iplane=1,nc  ! jc = first, last
      do ic=1,mic(iplane)
        if (dcct(3,ic,iplane).eq.0) exit
        jsize = 0
        do j=1,ncc_size
          if (dcct(4,ic,iplane).eq.cc_size(j)) then
            jsize = j
            exit
          endif
        enddo
        if (jsize.eq.0) then
          ncc_size = ncc_size+1
          cc_size(ncc_size) =  dcct(4,ic,iplane)
        endif
        if (ncc_size.gt.10) exit
      enddo
    enddo
    cc_count = 0
    cc_flux = 0.
  else
    ncc_size = 1
  endif
  !
  no = ubound(mic,1)
  if (no.lt.last-first+1 .or. no.gt.ubound(dcct,3)) then
    Print *,'Remove Clean Slow dimension error ',no,last-first+1,ubound(dcct,3)
  endif
  icmax = maxval(mic)
  if (icmax.gt.ubound(dcct,2))  then
    Print *,'Remove Clean Slow -- too many Clean Comp.',icmax,ubound(dcct,2)
  endif
  ! Print *,'CCT Bounds ',ubound(dcct),' FIRST ',first,' LAST ',last
  !
  ! Remove clean component from UV data set
  !
  ! G = flux * exp(-2i*pi(u*x0+v*y0)) * g(u,v)
  ! and for the 1-D Gaussian
  !     model = flux*4*ln2/(pi*b**2) * exp (-4*ln2*(r/b)**2)
  !     g(u,v) = exp(- pi**2/4/ln(2)*(b*q)**2)
  !     where q**2 = u**2+v**2
  !      and  b is the beam size in Radian (FWHM)
  !
  cst = pi**2/4d0/log(2D0) * (freq/clight)**2   ! Scale factor from u^2+v^2 in m^2 to radian^-2
  !
  do_size = scc.eq.4    ! Check size if needed
  !
  !$ nthread = ompget_inner_threads() 
  !$OMP PARALLEL DEFAULT(none) NUM_THREADS(nthread) &
  !$OMP    & SHARED(visi,ouv,dcct)  SHARED(pidsur,nv,first,last,mic) &
  !$OMP    & PRIVATE(iv,jc,ic,kc,ir,ii,iw,or,oi,ow,phase,rvis,ivis,cp,sp) &
  !$OMP    & PRIVATE(value, q2, gauss) SHARED(do_size, cst, subtract)
  !
  !$OMP DO SCHEDULE(DYNAMIC,32)
  do iv=1,nv            ! Visibilies
    ouv(1:7,iv) = visi(1:7,iv)
    if (do_size) then
      ! Load (U,V) in radian**-1
      q2 = cst * (ouv(1,iv)**2 + ouv(2,iv)**2)
    endif
    !
    do jc = first,last  ! Channels
      ir = 5+3*jc
      ii = 6+3*jc
      iw = 7+3*jc
      !
      kc = jc-first+1
      or = 5+3*kc      
      oi = 6+3*kc
      ow = 7+3*kc
      !
      ouv(or,iv) = visi(ir,iv)
      ouv(oi,iv) = visi(ii,iv)
      do ic = 1,mic(kc) ! Clean components
        if (dcct(3,ic,kc).ne.0) then
          value = dcct(3,ic,kc)
          !
          ! Apply the Gauss factor if any
          if (do_size) then
            if (dcct(4,ic,kc).ne.0) then
              gauss = q2 * dcct(4,ic,kc)**2
              if (gauss.lt.40) then
                gauss = exp(-gauss)
                value = value*gauss
              else
                cycle  ! Too faint, no signal
              endif
            endif
          endif
          !
          phase = (ouv(1,iv)*dcct(1,ic,kc) + ouv(2,iv)*dcct(2,ic,kc))*pidsur
          cp = cos(phase)
          sp = sin(phase)
          !!call cossin(phase,cp,sp)
          rvis = value*cp
          ivis = value*sp
          if (subtract) then
            ouv(or,iv) = ouv(or,iv) - rvis   ! Subtract
            ouv(oi,iv) = ouv(oi,iv) - ivis
          else
            ouv(or,iv) = rvis   ! Model
            ouv(oi,iv) = ivis
          endif
        else if (iv.eq.1) then
          Print *,'Premature end of work for channel ',jc, kc, mic(kc)
          exit ! End of work, jump to next channel
        endif
      enddo             ! Clean components
      ouv(ow,iv) = visi(iw,iv)
    enddo               ! Channels
  enddo                 ! Visibilities
  !$OMP END DO
  !$OMP END PARALLEL
end subroutine uv_removes_clean
!
subroutine uv_removeh_clean(method,hbeam,hclean, &
  & hcct,visi,ouv,nc,mic,fcou,freq,first,last,ifield,subtract)
  use image_def
  use clean_def
  use imager_interfaces, except_this => uv_removeh_clean
  use gbl_message
  !$ use omp_lib
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER   -- Support for UV_RESTORE
  !
  !     Compute visibilities for a UV data set according to a
  !     set of Clean Components , and remove them from the original
  !     UV table. Valid for Mosaics, treating just one Field at
  !     a time.
  !
  !   This version is for tranpose CCT data (3 or 4,ncomp,nchannels)
  !   and uses an intermediate FFT for speed.
  !!
  !-----------------------------------------------------------------
  type(clean_par), intent(in) :: method !! Cleaning Method
  type(gildas), intent(in) :: hbeam     !! header of Dirty beam
  type(gildas), intent(in) :: hclean    !! header of Clean cube
  type(gildas), intent(in) :: hcct      !! header of Clean Components Table
  integer, intent(in) :: nc             !! Number of channels
  integer, intent(in) :: mic(:)         !! Number of Clean Components
  real, intent(in) :: visi(:,:)          !! Input visibilities
  real, intent(out) :: ouv(:,:)         !! Output visibilities
  real, intent(in), target :: fcou(:,:,:)   !! Clean Components
  real(8), intent(in) :: freq           !! Apparent observing frequency
  integer, intent(in) :: first          !! First channel
  integer, intent(in) :: last           !! Last channel
  integer, intent(in) :: ifield         !! Field number
  logical, intent(in) :: subtract       !! Compute Diff or Model
  !
  ! Local ---
  integer :: jc,kc,oic,olc,j3,j4,ier
  real, allocatable :: dmap(:,:,:)
  real, pointer :: fctmp(:,:,:)
  integer :: nx,ny
  integer :: ibeam
  !
  oic = first
  olc = last
  !
  nx = (hcct%gil%convert(1,1)-1)*2   ! Reference pixel is at Mx/2+1
  ny = (hcct%gil%convert(1,3)-1)*2
  allocate(dmap(nx,ny,nc),stat=ier)  ! This is assumed not to fail - it is unsafe
  !
  ! Loop over channels to compute the Clean Component Cube
  fctmp => fcou(:,:,:) !! oic-ic+1:) ! A voir
  do jc=oic,olc
    kc = jc-oic+1
    call clean_make_cct(method,hclean,dmap(:,:,kc:kc),fctmp(:,:,kc:kc),method%gsize)
  enddo
  !
  ! This is approximate - a better solution would be to scan
  ! by Channels to figure out the most appropriate primary beam,
  ! identifying blocks of channels with the same primary beam.
  !
  ! TO BE DONE
  jc = (oic+olc)/2
  ibeam = beam_for_channel(jc,hclean,hbeam)
  if (hbeam%gil%faxi.eq.3) then
    j3 = ibeam
    j4 = ifield
  else
    j3 = ifield
    j4 = ibeam
  endif
  !
  call uv_removeg_clean(hcct,visi,ouv,olc-oic+1,mic(oic:olc), &
    & dmap,freq, oic, olc, hbeam%r4d(:,:,j3,j4),subtract)
end subroutine uv_removeh_clean
!
subroutine uv_removeg_clean(hcct,visi,ouv,nc,mic,dmap,freq,first,last,dbeam,subtract)
  use image_def
  use imager_interfaces, except_this => uv_removeg_clean
  use gbl_message
  !$ use omp_lib
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER  -- Support for UV_RESTORE  
  !
  !     Compute visibilities for a UV data set according to a
  !     set of Clean Components , and remove them from the original
  !     UV table
  !
  !   This version is for tranpose CCT data (3 or 4,ncomp,nchannels)
  !   and uses an intermediate FFT for speed
  !!
  !-----------------------------------------------------------------
  type(gildas), intent(in) :: hcct    !! header of Clean Components Table
  integer, intent(in) :: nc           !! Number of channels
  integer, intent(in) :: mic(:)       !! Number of Clean Components
  real, intent(in) :: visi(:,:)        !! Input visibilities
  real, intent(out) :: ouv(:,:)       !! Output visibilities
  real, intent(in) :: dmap(:,:,:)     !! Sky distribution of Clean components
  real(8), intent(in) :: freq         !! Apparent observing frequency
  integer, intent(in) :: first        !! First channel
  integer, intent(in) :: last         !! last channel
  real, intent(in) :: dbeam(:,:)      !! Primary Beam for current field
  logical, intent(in) :: subtract     !! Compute Diff or Model
  !
  ! Constants
  real(8), parameter :: pi=3.14159265358979323846d0
  integer, parameter :: mscales=20
  !
  ! Local ---
  integer :: nv    ! Number of visibilities
  integer :: mv    ! Size of a visibility
  integer :: nx,ny ! Large size for FFT
  integer :: mx,my ! Small size 
  integer :: kx,ky
  complex, allocatable :: cfft(:,:,:)
  real, allocatable :: work(:)
  real, allocatable :: lmap(:,:)
  integer :: iplane,iv,ier, dim(2)
  logical :: error
  real(8) :: xinc, yinc, xref, yref, xval, yval
  integer :: mthread, lx,ly
  !
  ! Code ---
  nv = ubound(visi,2)    ! Number of Visibilities  TO BE DONE
  mv = 7+3*nc
  !
  ! Image size - Twice the (reference_pixel-1) by convention
  mx = (hcct%gil%convert(1,1)-1)*2
  my = (hcct%gil%convert(1,3)-1)*2
  if ((mx.ne.size(dmap,1)).or.(my.ne.size(dmap,2))) then
    Print *,'MX MY ',mx,my
    Print *,'Sizes ',size(dmap,1),size(dmap,2)
    call map_message(seve%e,'RESTORE_G','Size mismatch ')
    error = .true.
    return
  endif
  !
  nx = mx
  ny = my
  ! No reinterpolation - Will come later if needed
  ! call cct_fft_size(mx,my,nx,ny)
  !
  ! Get Virtual Memory & compute the FFT
  allocate(cfft(nx,ny,nc),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,'RESTORE_G','uv_removeg_clean -- allocation error')
    error = .true.
    return
  endif
  !
  xref = hcct%gil%convert(1,1)
  xval = hcct%gil%convert(2,1)
  xinc = hcct%gil%convert(3,1)
  yref = hcct%gil%convert(1,3)
  yval = hcct%gil%convert(2,3)
  yinc = hcct%gil%convert(3,3)
  dim = [nx,ny]
  !
  !  call fourt_plan(ftbeam,dim,2,1,1)
  kx = mx/2+1
  lx = nx/2+1
  ky = my/2+1
  ly = ny/2+1
  !
  ! Must set the number of threads according to number of channels
  ! and available Thread nesting
  mthread = 1
  !$ mthread = ompget_inner_threads()
  mthread = min(nc,mthread)
  !
  !$OMP PARALLEL IF (nc.gt.1) DEFAULT(none) NUM_THREADS(mthread) SHARED(dmap,dbeam,cfft) &
  !$OMP    & SHARED(nc,nx,ny,mx,my,dim) &
  !$OMP    & PRIVATE(iplane,work,lmap,ier) 
  !
  allocate(work(2*max(nx,ny)),lmap(nx,ny),stat=ier)
  !!Print *,'IER ',ier, size(dbeam,1), size(dbeam,2)
  !!Print *,dbeam(1,1)
  !$OMP DO
  do iplane=1,nc
    ! Apply primary beam
    lmap(:,:) = dbeam * dmap(:,:,iplane)
    !TEST!Print *,'Done LMAP ',iplane
    cfft(:,:,iplane) = cmplx(lmap,0.0)
    !TEST!Print *,'Done CFFT ',iplane
    ! FOURT is now Thread-safe for the non-FFTW version.
    call fourt(cfft(:,:,iplane),dim,2,1,1,work)
    !TEST!Print *,'Done FOURT ',iplane
    call recent(nx,ny,cfft(:,:,iplane))
    !TEST!Print *,'Done RECENT ',iplane
  enddo
  !$OMP END DO
  deallocate(work)
  !$OMP END PARALLEL
  !
  ! Extract the visibility subset
  do iv = 1,nv
    ouv(1:7,iv) = visi(1:7,iv)
    ouv(8:mv,iv) = visi(5+3*first:7+3*last,iv)
  enddo
  !
  ! Interpolate and subtract the model visibilities
  !$ mthread = ompget_inner_threads()
  call do_smodel(ouv,mv,nv,cfft,nx,ny,nc,freq,xinc,yinc,1.0,mthread,subtract)
  !
  deallocate(cfft)
end subroutine uv_removeg_clean
!
subroutine uv_removef_clean(hcct,visi,ouv,nc,mic,dcct,freq,first,last,subtract)
  use image_def
  use imager_interfaces, except_this => uv_removef_clean
  use gbl_message
  !$ use omp_lib
  !-----------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER  -- Support for UV_RESTORE
  !
  !     Compute visibilities for a UV data set according to a
  !     set of Clean Components , and remove them from the original
  !     UV table.
  !   This version is for tranpose CCT data (3 or 4,ncomp,nchannels)
  !   and uses an intermediate FFT for speed.
  !!
  !-----------------------------------------------------------------
  type(gildas), intent(in) :: hcct    !! header of Clean Components Table
  integer, intent(in) :: nc           !! Number of channels
  integer, intent(in) :: mic(:)       !! Number of Clean Components
  real, intent(in) :: visi(:,:)        !! Input visibilities
  real, intent(out) :: ouv(:,:)       !! Output visibilities
  real, intent(in) :: dcct(:,:,:)     !! Clean components
  real(8), intent(in) :: freq         !! Apparent observing mean frequency of channel range
  integer, intent(in) :: first        !! First channel
  integer, intent(in) :: last         !! Last channel
  logical, intent(in) :: subtract     !! Compute Diff or Model
  !
  ! Constants
  real(8), parameter :: pi=3.14159265358979323846d0
  integer, parameter :: mscales=20    ! Maximum number of size scales
  !
  ! Local ---
  integer :: nv    ! Number of visibilities
  integer :: mv    ! Size of a visibility
  integer :: nx,ny
  integer :: mx,my
  integer :: kx,ky
  real :: dxy
  real :: value
  complex, allocatable :: cfft(:,:,:)
  complex, allocatable :: lfft(:,:,:)
  real, allocatable :: work(:)
  integer :: iplane,ic,ix,iy,iv,ier, dim(2)
  logical :: error
  real(8) :: xinc, yinc, xref, yref, xval, yval
  integer :: mthread, lx,ly, scc, ncc_size, jsize, j, is
  real :: asize, cc_size(mscales), fact(mscales), cc_count(mscales), cc_flux(mscales)
  !
  ! Code ---
  scc = size(dcct,1)    ! Size of a Clean Component (3 or 4)
  if (scc.ne.3) then
    ! Extended sources -- Find out number of sizes and size values
    ncc_size = 0
    do iplane=1,nc
      do ic=1,mic(iplane)
        if (dcct(3,ic,iplane).eq.0) exit
        jsize = 0
        do j=1,ncc_size
          if (dcct(4,ic,iplane).eq.cc_size(j)) then
            jsize = j
            exit
          endif
        enddo
        if (jsize.eq.0) then
          ncc_size = ncc_size+1
          cc_size(ncc_size) =  dcct(4,ic,iplane)
        endif
        if (ncc_size.gt.mscales) exit
      enddo
    enddo
    cc_count = 0
    cc_flux = 0.
  else
    ! Simple point sources
    ncc_size = 1
  endif
  !
  nv = ubound(visi,2)    ! Number of Visibilities TO BE DONE
  ! ! Print *,'Number of visibilities ',nv,' in removef_clean'
  mv = 7+3*nc
  !
  ! Image size - Twice the (reference_pixel-1) by convention
  mx = (hcct%gil%convert(1,1)-1)*2
  my = (hcct%gil%convert(1,3)-1)*2
  !
  call cct_fft_size(mx,my,nx,ny)
  !
  ! Get Virtual Memory & compute the FFT
  allocate(cfft(nx,ny,nc),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,'UV_RESTORE','uv_removef_clean -- allocation error')
    error = .true.
    return
  endif
  !
  xref = hcct%gil%convert(1,1)
  xval = hcct%gil%convert(2,1)
  xinc = hcct%gil%convert(3,1)
  yref = hcct%gil%convert(1,3)
  yval = hcct%gil%convert(2,3)
  yinc = hcct%gil%convert(3,3)
  dim = [nx,ny]
  !
  !  call fourt_plan(ftbeam,dim,2,1,1)
  kx = mx/2+1
  lx = nx/2+1
  ky = my/2+1
  ly = ny/2+1
  !
  ! Must set the number of threads according to number of channels
  ! and available Thread nesting
  mthread = 1
  !$ mthread = ompget_inner_threads()
  mthread = min(nc,mthread)
  !
  if (scc.eq.3) then
    ! Simple point sources
    !
    !$OMP PARALLEL IF (nc.gt.1) DEFAULT(none) NUM_THREADS(mthread) SHARED(dcct,cfft) &
    !$OMP    & SHARED(nc,mic,nx,ny,mx,my,dim,xinc,yinc,xref,yref,xval,yval) &
    !$OMP    & PRIVATE(iplane,ic,ix,iy,value,ier) PRIVATE(work) &
    !$OMP    & PRIVATE(dxy) SHARED(kx,lx,ky,ly)
    !
    allocate(work(2*max(nx,ny)),stat=ier)
    !$OMP DO
    do iplane=1,nc
      cfft(:,:,iplane) = 0.0
      do ic=1,mic(iplane)
        if (dcct(3,ic,iplane).eq.0) exit
        !
        value = dcct(3,ic,iplane)
        ! The NINT is required because of rounding errors
        ix = nint( (dcct(1,ic,iplane)-xval) / xinc + xref )
        iy = nint( (dcct(2,ic,iplane)-yval) / yinc + yref )
        cfft(ix-kx+lx,iy-ky+ly,iplane) = cfft(ix-kx+lx,iy-ky+ly,iplane) + &
          & cmplx(value,0.0)
      enddo
      !
      ! FOURT is now Thread-safe for the non-FFTW version.
      call fourt(cfft(:,:,iplane),dim,2,1,1,work)
      call recent(nx,ny,cfft(:,:,iplane))
    enddo
    !$OMP END DO
    deallocate(work)
    !$OMP END PARALLEL
    !
  else
    ! Variable size components
    !
    !$OMP PARALLEL IF (nc.gt.1) DEFAULT(none) NUM_THREADS(mthread) SHARED(dcct,cfft) &
    !$OMP    & SHARED(nc,mic,nx,ny,mx,my,dim,xinc,yinc,xref,yref,xval,yval, fact) &
    !$OMP    & PRIVATE(iplane,ic,ix,iy,is,value,ier) PRIVATE(work) &
    !$OMP    & PRIVATE(asize, lfft) SHARED(kx,lx,ky,ly,cc_size,ncc_size) &
    !$OMP    & PRIVATE(cc_count,cc_flux)
    !
    allocate(work(2*max(nx,ny)),lfft(nx,ny,ncc_size),stat=ier)
    !$OMP DO
    do iplane=1,nc
      cfft(:,:,iplane) = 0.0
      cc_count = 0
      cc_flux = 0.0
      lfft = cmplx(0.,0.)
      do ic=1,mic(iplane)
        if (dcct(3,ic,iplane).eq.0) exit
        !
        value = dcct(3,ic,iplane)
        ! The NINT is required because of rounding errors
        ix = nint( (dcct(1,ic,iplane)-xval) / xinc + xref )
        iy = nint( (dcct(2,ic,iplane)-yval) / yinc + yref )
        asize = dcct(4,ic,iplane)
        do is=1,ncc_size
          if (cc_size(is).eq.asize) then
            lfft(ix-kx+lx,iy-ky+ly,is) = lfft(ix-kx+lx,iy-ky+ly,is) + &
              & cmplx(value,0.0)
            cc_count(is) = cc_count(is)+1
            cc_flux(is) = cc_flux(is)+value
            exit
          endif
        enddo
      enddo
      !
      do is=1,ncc_size
        ! FOURT is now Thread-safe for the non-FFTW version.
        call fourt(lfft(:,:,is),dim,2,1,1,work)
        ! For flux normalisation
        ! fact = cmajor*cminor*pi/(4.0*log(2.0))   &
        !     &    /abs(xinc*yinc)/(nx*ny)
        ! For simple FFT normalization  1./(nx*ny)
        if (cc_size(is).ne.0) then
          ! Factor is just 1.0 here, since we work in total flux
          ! and directly in the Fourier Plane
          call mulgau(lfft(:,:,is),nx,ny,   &
               &    cc_size(is),cc_size(is),0.0,  &
               &    1.0,real(xinc),real(yinc),-1)
        endif
        !
        cfft(:,:,iplane) = cfft(:,:,iplane) + lfft(:,:,is)
      enddo
      !
      call recent(nx,ny,cfft(:,:,iplane))
    enddo
    !$OMP END DO
    deallocate(work)
    !$OMP END PARALLEL
    !
  endif
  !
  ! Extract the visibility subset
  do iv = 1,nv
    ouv(1:7,iv) = visi(1:7,iv)
    ouv(8:mv,iv) = visi(5+3*first:7+3*last,iv)
  enddo
  !
  ! Interpolate and subtract the model visibilities
  !$ mthread = ompget_inner_threads()
  call do_smodel(ouv,mv,nv,cfft,nx,ny,nc,freq,xinc,yinc,1.0,mthread,subtract)
  !
  deallocate(cfft)
end subroutine uv_removef_clean
!
subroutine cossin(phase,rcos,rsin)
  !-------------------------------------------------------
  !*
  ! Semi-Fast,  Semi-accurate Sin/Cos pair computation
  ! using (not yet clever) interpolation from a precise
  ! loop-up table
  !
  ! A solution using Taylor expansion and symmetries
  ! would be faster and more accurate. The speed is
  ! unfortunately controlled by the Modulo function.
  !    Might become faster if a simple test is made
  ! to avoid this Modulo.  To be developped as needed.
  !! 
  !-------------------------------------------------------
  real(8), intent(inout) :: phase  !! Phase value (radian)
  real(8), intent(out) :: rcos     !! Approximate Cosine
  real(8), intent(out) :: rsin     !! Approximate Sine
  !
  ! Constants
  real(8), parameter :: pi=3.14159265358979323846d0
  integer, parameter :: mcos=2048
  integer, save :: ncos = 0
  real(8), save :: cosine(mcos)
  real(8), save :: sine(mcos)
  ! Local ---
  real(8), save :: rstep
  real(8) :: rdeg
  integer :: i
  logical :: minus
  !
  ! Use accurate value
  rcos = cos(phase)
  rsin = sin(phase)
  !
  ! Approximate solution below
  if (.TRUE.) return
  !
  if (ncos.eq.0) then
    ncos = mcos
    rstep = 2.01d0*pi/mcos
    do i=1,ncos
      rdeg = (i-1)*rstep
      cosine(i) = cos(rdeg)
      sine(i) = sin(rdeg)
    enddo
  endif
  !
  ! Unclear why 0.5*step offset is added ? 
  if (phase.ge.0) then
    minus =.false.
    rdeg = modulo(phase,2.0d0*pi)+0.5d0*rstep
  else
    minus = .true.
    rdeg = modulo(-phase,2.0d0*pi)+0.5d0*rstep
  endif
  rdeg = rdeg/rstep
  i = int(rdeg)
  rdeg = rdeg - i
  i = i+1
  ! One could do better also by noting the Sin/Cos are
  ! the derivatives of Cos/Sin...
  rcos = (1.0-rdeg)*cosine(i) + rdeg*cosine(i+1)
  rsin = (1.0-rdeg)*sine(i) + rdeg*sine(i+1)
  if (minus) rsin = -rsin
end subroutine cossin
!
