#!/usr/bin/env python3

"""
This file is used for generating i16x8 related test cases
which inherites from the 'SimdCmpCase' class and overloads
with the 'get_test_cases' method.
"""

from simd_compare import SimdCmpCase


# Generate i16x8 test case
class Simdi16x8CmpCase(SimdCmpCase):

    LANE_TYPE = 'i16x8'

    BINARY_OPS = ['eq', 'ne', 'lt_s', 'lt_u', 'le_s', 'le_u', 'gt_s', 'gt_u', 'ge_s', 'ge_u']

    # Overloads base class method and sets test data for i16x8.
    def get_case_data(self):

        case_data = []

        # eq
        # i16x8.eq  (i16x8) (i16x8)
        case_data.append(['#', 'eq'])
        case_data.append(['#', 'i16x8.eq  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['eq', ['0xFFFF', '0xFFFF'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0x0000', '0x0000'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0xF0F0', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0x0F0F', '0x0F0F'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['eq', ['0xFFFF', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0xFFFF', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0x8080', '32896'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0x8080', '-32640'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['eq', ['-1', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0', '0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['65535', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['65535', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['65535', '0'], ['65535', '0']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0', '65535'], ['0', '65535']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['-32768', '65534', '-1', '-0', '0', '1', '2', '65535'],
                          ['32768', '-2', '-1', '-0', '0', '1', '2', '-1']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['eq', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['eq', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['eq', ['0x0F0F', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0x8000', '0x8001', '0x8002', '0x8003', '0x8004', '0x8005', '0x8006', '0x8007'],
                          ['32775', '32774', '32773', '32772', '32771', '32770', '32769', '32768']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['32768', '32769', '65534', '65535', '0', '-1', '-32767', '-32768'],
                          ['-32768', '-32767', '-1', '0', '65535', '65534', '32769', '32768']], ['-1', '-1', '0', '0', '0', '0', '-1', '-1'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.eq  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.eq  (i16x8) (i8x16)'])
        case_data.append(['eq', ['0xFFFF', '0xFF'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['eq', ['65535', '255'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['eq', ['0', '0'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['eq', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['eq', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['eq', [['-128', '-128', '0', '0', '1', '1', '255', '255'],
                          ['-128', '0', '1', '255']], ['0', '0', '-1', '-1', '0', '0', '0', '0'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['eq', ['0x5555', '0xAA'], '0', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.eq  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.eq  (i16x8) (i32x4)'])
        case_data.append(['eq', ['0xFFFF', '0xFFFFFFFF'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['eq', ['65535', '4294967295'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['eq', ['0', '0'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['eq', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['eq', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['eq', [['65535', '0', '1', '32768'], ['65535', '0', '1', '32768']], ['-1', '0', '-1', '-1', '-1', '0', '-1', '0'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['eq', ['0x5555', '0xAAAAAAAA'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['eq', ['012_345', '12345'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0x0_1234', '0x1234'], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # ne
        # i16x8.ne  (i16x8) (i16x8)
        case_data.append(['#', 'ne'])
        case_data.append(['#', 'i16x8.ne  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['ne', ['0xFFFF', '0xFFFF'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0x0000', '0x0000'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0xF0F0', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0x0F0F', '0x0F0F'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['ne', ['0xFFFF', '65535'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0xFFFF', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0x8080', '32896'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0x8080', '-32640'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['ne', ['-1', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0', '0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['255', '255'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['65535', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['255', '0'], ['255', '0']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0', '255'], ['0', '255']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['255', '32767', '-0', '0', '1', '2', '65534', '65535'],
                          ['255', '32767', '0', '0', '1', '2', '-2', '-1']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['ne', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['ne', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['ne', ['0x0F0F', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['0x8081', '0x8283', '0xFDFE', '0xFF00', '0x0001', '0x027F', '0x80FD', '0xFEFF'],
                          ['65279', '33021', '639', '1', '65280', '65022', '33411', '32897']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', [['128', '129', '130', '131', '-0', '255', '32766', '32767'],
                          ['32767', '32766', '255', '-0', '131', '130', '129', '28']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.ne  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.ne  (i16x8) (i8x16)'])
        case_data.append(['ne', ['0xFFFF', '0xFF'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ne', ['65535', '255'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ne', ['0', '0'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ne', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ne', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ne', [['-128', '-128', '0', '0', '1', '1', '255', '255'], ['-128', '0', '1', '255']],
                          ['-1', '-1', '0', '0', '-1', '-1', '-1', '-1'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ne', ['0x5555', '0xAA'], '-1', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.ne  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.ne  (i16x8) (i32x4)'])
        case_data.append(['ne', ['0xFFFF', '0xFFFFFFFF'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ne', ['65535', '4294967295'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ne', ['0', '0'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ne', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ne', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ne', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['0', '-1', '0', '0', '0', '-1', '0', '-1'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ne', ['0x5555', '0xAAAAAAAA'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ne', ['012_345', '12345'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ne', ['0x0_1234', '0x1234'], '0', ['i16x8', 'i16x8', 'i16x8']])

        # lt_s
        # i16x8.lt_s  (i16x8) (i16x8)
        case_data.append(['#', 'lt_s'])
        case_data.append(['#', 'i16x8.lt_s  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['lt_s', ['0xFFFF', '0xFFFF'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0x0000', '0x0000'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0xF0F0', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0x0F0F', '0x0F0F'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0xFFFF', '65535'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0xFFFF', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0x8080', '32896'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0x8080', '-32640'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['-1', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0', '0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['255', '255'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['65535', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['255', '0'], ['255', '0']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0', '255'], ['0', '255']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['255', '32767', '-0', '0', '1', '2', '65534', '65535'], ['255', '32767', '0', '0', '1', '2', '-2', '-1']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['lt_s', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['lt_s', ['0x0F0F', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['0', '0', '-1', '0', '0', '0', '0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['0x8081', '0x8283', '0xFDFE', '0xFF00', '0x0001', '0x027F', '0x80FD', '0xFEFF'],
                          ['65279', '33021', '639', '1', '65280', '65022', '33411', '32897']], ['-1', '0', '-1', '-1', '0', '0', '-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', [['128', '129', '130', '131', '-0', '255', '32766', '32767'],
                          ['32767', '32766', '255', '-0', '131', '130', '129', '28']], ['-1', '-1', '-1', '0', '-1', '0', '0', '0'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.lt_s  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.lt_s  (i16x8) (i8x16)'])
        case_data.append(['lt_s', ['0xFFFF', '0xFF'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_s', ['65535', '255'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_s', ['0', '0'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_s', [['-128', '-128', '0', '0', '1', '1', '255', '255'], ['-128', '0', '1', '255']],
                          ['0', '0', '0', '0', '-1', '-1', '0', '0'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_s', ['0x5555', '0xAA'], '0', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.lt_s  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.lt_s  (i16x8) (i32x4)'])
        case_data.append(['lt_s', ['0xFFFF', '0xFFFFFFFF'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_s', ['65535', '4294967295'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_s', ['0', '0'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_s', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['0', '-1', '0', '0', '0', '0', '0', '0'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_s', ['0x5555', '0xAAAAAAAA'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_s', ['012_345', '12345'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_s', ['0x0_1234', '0x1234'], '0', ['i16x8', 'i16x8', 'i16x8']])

        # lt_u
        # i16x8.lt_u  (i16x8) (i16x8)
        case_data.append(['#', 'lt_u'])
        case_data.append(['#', 'i16x8.lt_u  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['lt_u', ['0xFFFF', '0xFFFF'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0x0000', '0x0000'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0xF0F0', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0x0F0F', '0x0F0F'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['lt_u', ['0xFFFF', '65535'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0xFFFF', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0x8080', '32896'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0x8080', '-32640'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['lt_u', ['-1', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0', '0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['255', '255'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['65535', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['255', '0'], ['255', '0']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0', '255'], ['0', '255']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['255', '32767', '-0', '0', '1', '2', '65534', '65535'],
                          ['255', '32767', '0', '0', '1', '2', '-2', '-1']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['lt_u', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['lt_u', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['lt_u', ['0x0F0F', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['-1', '-1', '-1', '0', '0', '0', '0', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['0x8081', '0x8283', '0xFDFE', '0xFF00', '0x0001', '0x027F', '0x80FD', '0xFEFF'],
                          ['65279', '33021', '639', '1', '65280', '65022', '33411', '32897']], ['-1', '0', '0', '0', '-1', '-1', '-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', [['128', '129', '130', '131', '-0', '255', '32766', '32767'],
                          ['32767', '32766', '255', '-0', '131', '130', '129', '28']], ['-1', '-1', '-1', '0', '-1', '0', '0', '0'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.lt_u  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.lt_u  (i16x8) (i8x16)'])
        case_data.append(['lt_u', ['0xFFFF', '0xFF'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_u', ['65535', '255'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_u', ['0', '0'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_u', [['-128', '-128', '0', '0', '1', '1', '255', '255'], ['-128', '0', '1', '255']], ['0', '-1'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['lt_u', ['0x5555', '0xAA'], '-1', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.lt_u  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.lt_u  (i16x8) (i32x4)'])
        case_data.append(['lt_u', ['0xFFFF', '0xFFFFFFFF'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_u', ['65535', '4294967295'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_u', ['0', '0'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_u', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']],
                          ['0', '-1', '0', '0', '0', '0', '0', '0'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_u', ['0x5555', '0xAAAAAAAA'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['lt_u', ['012_345', '12345'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['lt_u', ['0x0_1234', '0x1234'], '0', ['i16x8', 'i16x8', 'i16x8']])

        # le_s
        # i16x8.le_s  (i16x8) (i16x8)
        case_data.append(['#', 'le_s'])
        case_data.append(['#', 'i16x8.le_s  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['le_s', ['0xFFFF', '0xFFFF'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0x0000', '0x0000'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0xF0F0', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0x0F0F', '0x0F0F'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['le_s', ['0xFFFF', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0xFFFF', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0x8080', '32896'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0x8080', '-32640'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['le_s', ['-1', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0', '0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['65535', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['65535', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['65535', '0'], ['65535', '0']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0', '65535'], ['0', '65535']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['-32768', '65534', '-1', '-0', '0', '1', '2', '65535'],
                          ['32768', '-2', '-1', '-0', '0', '1', '2', '-1']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['le_s', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['le_s', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['le_s', ['0x0F0F', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['0', '0', '-1', '0', '0', '0', '0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['0x8000', '0x8001', '0x8002', '0x8003', '0x8004', '0x8005', '0x8006', '0x8007'],
                          ['32775', '32774', '32773', '32772', '32771', '32770', '32769', '32768']], ['-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', [['32768', '32769', '65534', '65535', '0', '-1', '-32767', '-32768'],
                          ['-32768', '-32767', '-1', '0', '65535', '65534', '32769', '32768']], ['-1', '-1', '-1', '-1', '0', '0', '-1', '-1'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.le_s  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.le_s  (i16x8) (i8x16)'])
        case_data.append(['le_s', ['0xFFFF', '0xFF'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_s', ['65535', '255'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_s', ['0', '0'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_s', [['-128', '-128', '0', '0', '1', '1', '255', '255'], ['-128', '0', '1', '255']],
                          ['0', '0', '-1', '-1', '-1', '-1', '0', '0'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_s', ['0x5555', '0xAA'], '0', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.le_s  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.le_s  (i16x8) (i32x4)'])
        case_data.append(['le_s', ['0xFFFF', '0xFFFFFFFF'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_s', ['65535', '4294967295'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_s', ['0', '0'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_s', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['-1', '-1', '-1', '-1', '-1', '0', '-1', '0'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_s', ['0x5555', '0xAAAAAAAA'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_s', ['012_345', '12345'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_s', ['0x0_1234', '0x1234'], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # le_u
        # i16x8.le_u  (i16x8) (i16x8)
        case_data.append(['#', 'le_u'])
        case_data.append(['#', 'i16x8.le_u  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['le_u', ['0xFFFF', '0xFFFF'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0x0000', '0x0000'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0xF0F0', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0x0F0F', '0x0F0F'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['le_u', ['0xFFFF', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0xFFFF', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0x8080', '32896'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0x8080', '-32640'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['le_u', ['-1', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0', '0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['65535', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['65535', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['65535', '0'], ['65535', '0']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0', '65535'], ['0', '65535']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['-32768', '65534', '-1', '-0', '0', '1', '2', '65535'], ['32768', '-2', '-1', '-0', '0', '1', '2', '-1']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['le_u', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['le_u', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['le_u', ['0x0F0F', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['-1', '-1', '-1', '0', '0', '0', '0', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['0x8000', '0x8001', '0x8002', '0x8003', '0x8004', '0x8005', '0x8006', '0x8007'],
                          ['32775', '32774', '32773', '32772', '32771', '32770', '32769', '32768']], ['-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', [['32768', '32769', '65534', '65535', '0', '-1', '-32767', '-32768'],
                          ['-32768', '-32767', '-1', '0', '65535', '65534', '32769', '32768']], ['-1', '-1', '-1', '0', '-1', '0', '-1', '-1'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.le_u  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.le_u  (i16x8) (i8x16)'])
        case_data.append(['le_u', ['0xFFFF', '0xFF'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_u', ['65535', '255'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_u', ['0', '0'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_u', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['0', '0', '-1', '-1', '-1', '-1', '-1', '-1'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['le_u', ['0x5555', '0xAA'], '-1', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.le_u  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.le_u  (i16x8) (i32x4)'])
        case_data.append(['le_u', ['0xFFFF', '0xFFFFFFFF'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_u', ['65535', '4294967295'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_u', ['0', '0'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_u', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['-1', '-1', '-1', '-1', '-1', '0', '-1', '0'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_u', ['0x5555', '0xAAAAAAAA'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['le_u', ['012_345', '12345'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['le_u', ['0x0_edcb', '-0x1234'], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # gt_s
        # i16x8.gt_s  (i16x8) (i16x8)
        case_data.append(['#', 'gt_s'])
        case_data.append(['#', 'i16x8.gt_s  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['gt_s', ['0xFFFF', '0xFFFF'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0x0000', '0x0000'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0xF0F0', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0x0F0F', '0x0F0F'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['gt_s', ['0xFFFF', '65535'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0xFFFF', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0x8080', '32896'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0x8080', '-32640'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['gt_s', ['-1', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0', '0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['65535', '65535'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['65535', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['65535', '0'], ['65535', '0']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0', '65535'], ['0', '65535']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['-32768', '65534', '-1', '-0', '0', '1', '2', '65535'], ['32768', '-2', '-1', '-0', '0', '1', '2', '-1']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['gt_s', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['gt_s', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['gt_s', ['0x0F0F', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['-1', '-1', '0', '-1', '-1', '-1', '-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['0x8000', '0x8001', '0x8002', '0x8003', '0x8004', '0x8005', '0x8006', '0x8007'],
                          ['32775', '32774', '32773', '32772', '32771', '32770', '32769', '32768']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', [['32768', '32769', '65534', '65535', '0', '-1', '-32767', '-32768'],
                          ['-32768', '-32767', '-1', '0', '65535', '65534', '32769', '32768']], ['0', '0', '0', '0', '-1', '-1', '0', '0'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.gt_s  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.gt_s  (i16x8) (i8x16)'])
        case_data.append(['gt_s', ['0xFFFF', '0xFF'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_s', ['65535', '255'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_s', ['0', '0'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_s', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['-1', '0', '0', '-1'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_s', ['0x5555', '0xAA'], '-1', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.gt_s  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.gt_s  (i16x8) (i32x4)'])
        case_data.append(['gt_s', ['0xFFFF', '0xFFFFFFFF'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_s', ['65535', '4294967295'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_s', ['0', '0'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_s', [['65535', '0', '1', '32768'], ['65535', '0', '1', '32768']], ['0', '0', '0', '0', '0', '-1', '0', '0'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_s', ['0x5555', '0xAAAAAAAA'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_s', ['012_345', '12345'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_s', ['0x0_1234', '0x1234'], '0', ['i16x8', 'i16x8', 'i16x8']])

        # gt_u
        # i16x8.gt_u  (i16x8) (i16x8)
        case_data.append(['#', 'gt_u'])
        case_data.append(['#', 'i16x8.gt_u  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['gt_u', ['0xFFFF', '0xFFFF'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', ['0x0000', '0x0000'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', ['0xF0F0', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', ['0x0F0F', '0x0F0F'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['gt_u', ['0xFFFF', '65535'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', ['0xFFFF', '-1'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', ['0x8080', '32896'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', ['0x8080', '-32640'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '0', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['eq', ['-1', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['0', '0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['65535', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', ['65535', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['65535', '0'], ['65535', '0']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['0', '65535'], ['0', '65535']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['eq', [['-32768', '65534', '-1', '-0', '0', '1', '2', '65535'],
                          ['32768', '-2', '-1', '-0', '0', '1', '2', '-1']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['gt_u', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '0', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['gt_u', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '0', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['gt_u', ['0x0F0F', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['0', '0', '0', '-1', '-1', '-1', '-1', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['0x8000', '0x8001', '0x8002', '0x8003', '0x8004', '0x8005', '0x8006', '0x8007'],
                          ['32775', '32774', '32773', '32772', '32771', '32770', '32769', '32768']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', [['32768', '32769', '65534', '65535', '0', '-1', '-32767', '-32768'],
                          ['-32768', '-32767', '-1', '0', '65535', '65534', '32769', '32768']], ['0', '0', '0', '-1', '0', '-1', '0', '0'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.gt_u  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.gt_u  (i16x8) (i8x16)'])
        case_data.append(['gt_u', ['0xFFFF', '0xFF'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_u', ['65535', '255'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_u', ['0', '0'], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '0', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_u', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['-1', '0', '0', '0'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['gt_u', ['0x5555', '0xAA'], '0', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.gt_u  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.gt_u  (i16x8) (i32x4)'])
        case_data.append(['gt_u', ['0xFFFF', '0xFFFFFFFF'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_u', ['65535', '4294967295'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_u', ['0', '0'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_u', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['0', '0', '0', '0', '0', '-1', '0', '-1'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_u', ['0x5555', '0xAAAAAAAA'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['gt_u', ['012_345', '12345'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['gt_u', ['0x0_1234', '0x1234'], '0', ['i16x8', 'i16x8', 'i16x8']])

        # ge_s
        # i16x8.ge_s  (i16x8) (i16x8)
        case_data.append(['#', 'ge_s'])
        case_data.append(['#', 'i16x8.ge_s  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['ge_s', ['0xFFFF', '0xFFFF'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0x0000', '0x0000'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0xF0F0', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0x0F0F', '0x0F0F'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['ge_s', ['0xFFFF', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0xFFFF', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0x8080', '32896'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0x8080', '-32640'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['ge_s', ['-1', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0', '0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['65535', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['65535', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['65535', '0'], ['65535', '0']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0', '65535'], ['0', '65535']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['-32768', '65534', '-1', '-0', '0', '1', '2', '65535'], ['32768', '-2', '-1', '-0', '0', '1', '2', '-1']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['ge_s', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['ge_s', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['ge_s', ['0x0F0F', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['-1', '-1', '0', '-1', '-1', '-1', '-1', '0'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['0x8000', '0x8001', '0x8002', '0x8003', '0x8004', '0x8005', '0x8006', '0x8007'],
                          ['32775', '32774', '32773', '32772', '32771', '32770', '32769', '32768']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', [['32768', '32769', '65534', '65535', '0', '-1', '-32767', '-32768'],
                          ['-32768', '-32767', '-1', '0', '65535', '65534', '32769', '32768']], ['-1', '-1', '0', '0', '-1', '-1', '-1', '-1'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.ge_s  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.ge_s  (i16x8) (i8x16)'])
        case_data.append(['ge_s', ['0xFFFF', '0xFF'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_s', ['65535', '255'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_s', ['0', '0'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_s', [['-128', '0', '1', '255'], ['-128', '0', '1', '255']], ['-1', '-1', '-1', '-1', '0', '0', '-1', '-1'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_s', ['0xAAAA', '0x55'], '0', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.ge_s  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.ge_s  (i16x8) (i32x4)'])
        case_data.append(['ge_s', ['0xFFFF', '0xFFFFFFFF'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_s', ['65535', '4294967295'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_s', ['0', '0'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_s', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_s', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_s', [['65535', '0', '1', '32768'], ['65535', '0', '1', '32768']], ['-1', '0', '-1', '-1', '-1', '-1', '-1', '0'], ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_s', ['0x5555', '0xAAAAAAAA'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_s', ['012_345', '12345'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_s', ['0x0_1234', '0x1234'], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # ge_u
        # i16x8.ge_u  (i16x8) (i16x8)
        case_data.append(['#', 'ge_u'])
        case_data.append(['#', 'i16x8.ge_u  (i16x8) (i16x8)'])

        # hex vs hex
        case_data.append(['#', 'hex vs hex'])
        case_data.append(['ge_u', ['0xFFFF', '0xFFFF'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0x0000', '0x0000'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0xF0F0', '0xF0F0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0x0F0F', '0x0F0F'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0xFFFF', '0x0000'], ['0xFFFF', '0x0000']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0x0000', '0xFFFF'], ['0x0000', '0xFFFF']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB'],
                          ['0x0100', '0x0302', '0x0904', '0x1110', '0x0A12', '0x1A0B', '0xAA1B', '0xFFAB']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs dec
        case_data.append(['#', 'hex vs dec'])
        case_data.append(['ge_u', ['0xFFFF', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0xFFFF', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0x8080', '32896'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0x8080', '-32640'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0x8180', '0x8382', '0xFEFD', '0x00FF', '0x0100', '0x7F02', '0xFD80', '0xFFFE'],
                          ['33152', '33666', '65277', '255', '256', '32514', '64896', '65534']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # dec vs dec
        case_data.append(['#', 'dec vs dec'])
        case_data.append(['ge_u', ['-1', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0', '0'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['65535', '65535'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['65535', '-1'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['65535', '0'], ['65535', '0']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0', '65535'], ['0', '65535']], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['-32768', '65534', '-1', '-0', '0', '1', '2', '65535'], ['32768', '-2', '-1', '-0', '0', '1', '2', '-1']], '-1', ['i16x8', 'i16x8', 'i16x8']])

        # hex vs float
        case_data.append(['#', 'hex vs float'])
        case_data.append(['ge_u', [['0x0000', '0xc300', '0x0000', '0xc2fe', '0x0000', '0xbf80', '0x0000', '0x0000'],
                          ['-128.0', '-127.0', '-1.0', '0.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])
        case_data.append(['ge_u', [['0x0000', '0x3f80', '0x0000', '0x42fe', '0x0000', '0x4300', '0x0000', '0x437f'],
                          ['1.0', '127.0', '128.0', '255.0']], '-1', ['i16x8', 'f32x4', 'i16x8']])

        # not equal
        case_data.append(['#', 'not equal'])
        case_data.append(['ge_u', ['0x0F0F', '0xF0F0'], '0', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0x0000', '0xFFFF'], ['0xFFFF', '0x0000']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0x0001', '0x0203', '0x0409', '0x1011', '0x120A', '0x0B1A', '0x1BAA', '0xABFF'],
                          ['0xFFAB', '0xAA1B', '0x1A0B', '0x0A12', '0x1110', '0x0904', '0x0302', '0x0100']], ['0', '0', '0', '-1', '-1', '-1', '-1', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['0x8000', '0x8001', '0x8002', '0x8003', '0x8004', '0x8005', '0x8006', '0x8007'],
                          ['32775', '32774', '32773', '32772', '32771', '32770', '32769', '32768']], ['0', '-1'], ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', [['32768', '32769', '65534', '65535', '0', '-1', '-32767', '-32768'],
                          ['-32768', '-32767', '-1', '0', '65535', '65534', '32769', '32768']], ['-1', '-1', '0', '-1', '0', '-1', '-1', '-1'], ['i16x8', 'i16x8', 'i16x8']])

        # i16x8.ge_u  (i16x8) (i8x16)
        case_data.append(['#', 'i16x8.ge_u  (i16x8) (i8x16)'])
        case_data.append(['ge_u', ['0xFFFF', '0xFF'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_u', ['65535', '255'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_u', ['0', '0'], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x00', '0x01', '0x02', '0x03', '0x04', '0x05', '0x06', '0x07', '0x08', '0x09', '0x0A', '0x0B', '0x0C', '0x0D', '0x0E', '0x0F']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['-128', '-127', '-126', '-125', '-3', '-2', '-1', '0', '0', '1', '2', '127', '128', '253', '254', '255']], '-1', ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_u', [['-128', '-128', '0', '0', '1', '1', '255', '255'], ['-128', '0', '1', '255']], ['-1', '0'], ['i16x8', 'i8x16', 'i16x8']])
        case_data.append(['ge_u', ['0xAAAA', '0x55'], '-1', ['i16x8', 'i8x16', 'i16x8']])

        # i16x8.ge_u  (i16x8) (i32x4)
        case_data.append(['#', 'i16x8.ge_u  (i16x8) (i32x4)'])
        case_data.append(['ge_u', ['0xFFFF', '0xFFFFFFFF'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_u', ['65535', '4294967295'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_u', ['0', '0'], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_u', [['0x0100', '0x0302', '0x0504', '0x0706', '0x0908', '0x0B0A', '0x0D0C', '0x0F0E'],
                          ['0x03020100', '0x07060504', '0x0B0A0908', '0x0F0E0D0C']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_u', [['33152', '33666', '65277', '255', '256', '32514', '64896', '65534'],
                          ['2206368128', '16776957', '2130837760', '4294901120']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_u', [['65535', '0', '1', '32768'], ['-128', '0', '1', '255']], '-1', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_u', ['0x5555', '0xAAAAAAAA'], '0', ['i16x8', 'i32x4', 'i16x8']])
        case_data.append(['ge_u', ['012_345', '12345'], '-1', ['i16x8', 'i16x8', 'i16x8']])
        case_data.append(['ge_u', ['0x0_1234', '0x1234'], '-1', ['i16x8', 'i16x8', 'i16x8']])

        return case_data


def gen_test_cases():
    i16x8 = Simdi16x8CmpCase()
    i16x8.gen_test_cases()


if __name__ == '__main__':
    i16x8 = Simdi16x8CmpCase()
    i16x8.gen_test_cases()
