import type { SchemaYaml } from './types.js';
/**
 * Error thrown when loading a schema fails.
 */
export declare class SchemaLoadError extends Error {
    readonly schemaPath: string;
    readonly cause?: Error | undefined;
    constructor(message: string, schemaPath: string, cause?: Error | undefined);
}
/**
 * Gets the package's built-in schemas directory path.
 * Uses import.meta.url to resolve relative to the current module.
 */
export declare function getPackageSchemasDir(): string;
/**
 * Gets the user's schema override directory path.
 */
export declare function getUserSchemasDir(): string;
/**
 * Resolves a schema name to its directory path.
 *
 * Resolution order:
 * 1. User override: ${XDG_DATA_HOME}/openspec/schemas/<name>/schema.yaml
 * 2. Package built-in: <package>/schemas/<name>/schema.yaml
 *
 * @param name - Schema name (e.g., "spec-driven")
 * @returns The path to the schema directory, or null if not found
 */
export declare function getSchemaDir(name: string): string | null;
/**
 * Resolves a schema name to a SchemaYaml object.
 *
 * Resolution order:
 * 1. User override: ${XDG_DATA_HOME}/openspec/schemas/<name>/schema.yaml
 * 2. Package built-in: <package>/schemas/<name>/schema.yaml
 *
 * @param name - Schema name (e.g., "spec-driven")
 * @returns The resolved schema object
 * @throws Error if schema is not found in any location
 */
export declare function resolveSchema(name: string): SchemaYaml;
/**
 * Lists all available schema names.
 * Combines user override and package built-in schemas.
 */
export declare function listSchemas(): string[];
/**
 * Schema info with metadata (name, description, artifacts).
 */
export interface SchemaInfo {
    name: string;
    description: string;
    artifacts: string[];
    source: 'package' | 'user';
}
/**
 * Lists all available schemas with their descriptions and artifact lists.
 * Useful for agent skills to present schema selection to users.
 */
export declare function listSchemasWithInfo(): SchemaInfo[];
//# sourceMappingURL=resolver.d.ts.map