#!/bin/bash

# Semantics of return codes:
#
# 0  -- everything went well
# 1  -- uncategorized error
# 2  -- the input was invalid
# 10 -- the gp code failed during initial computation.
# 11 -- the C code failed
# 12 -- the gp code failed during check
# 20 -- the gp code was unable to check by itself
# 21 -- the gp code was unable to check by itself, but passed to magma,
# which failed.

if test -z "$CMH_BINDIR" ; then
   CMH_BINDIR=/opt/local/bin
fi
if test -z "$CMH_PKGDATADIR" ; then
   CMH_PKGDATADIR=/opt/local/share/cmh
fi
if test -z "$CMH_PKGLIBDIR" ; then
   CMH_PKGLIBDIR=/opt/local/lib/cmh
fi

if test -z "$GP_PROGRAM" ; then
   if test -d "" ; then
       GP_PROGRAM="/bin/gp"
   else
       GP_PROGRAM=gp
   fi
fi

GP_OPTS="-q --default colors=no,no,no,no,no,no,no"

SHIMURA_GP=$CMH_PKGDATADIR/shimura.gp
CHECKPOL_GP=$CMH_PKGDATADIR/checkpol.gp
CM2_BIN=$CMH_BINDIR/cm2

: ${wdir:=.}
: ${p0:=2^128}


usage() {
    echo "Usage: $0 [-o <data directory>] [-p] [-f] [-c] [--no-keep] <input specification>" >&2
    exit 2
}

# The different steps of the computation are:
#
# -p  : compute the period matrices
# -f  : compute the class polynomials
# -c  : compute a weil number and an example curve.

set_DAB_from_anything() {
    # Read the command line, and deduce either a D,A,B triple, or possibly
    # only A,B (or 0,A,B). Then D is unspecified, this implicitly means that
    # the discriminant should be computed by gp.
    set `echo $* | sed -e 's/[_,]/ /g' -e 's,\..*$,,' -e 's,^.*/,,'`
    if [ "$#" = 2 ] ; then
        D=0
    else
        D="$1"
        shift
    fi
    A="$1"
    B="$2"

    if [ "$A" -le 0 ] ; then
        echo "Error, $A must be >0" >&2
        exit 2
    fi

    nfacs="`echo "v=factor(z^4 + $A*z^2 + $B)[,2]; sum(i=1,#v,v[i])" | $GP_PROGRAM $GP_OPTS`"
    if [ "$nfacs" != 1 ] ; then
        echo "Error, X^4 + $A*X^2 + $B is not irreducible ($nfacs)" >&2
        exit 2
    fi
    if [ "$D" = 0 ] ; then
        D="`echo "nfinit (z^2 + $A*z + $B).disc" | $GP_PROGRAM $GP_OPTS`"
        if [ "$D" -le 0 ] ; then
            echo "Error, input arguments lead to bogus discriminant D=$D (one must have A^2-4*B>1 and squarefree)" >&2
            exit 2
        fi
    fi
    gal="`echo "g=galoisinit(nfinit(z^4 + $A*z^2 + $B));if(g==0,g,Vec(g.orders))" | $GP_PROGRAM $GP_OPTS`"
    case "$gal" in 0|"[4]") : ;;
        *) cat <<EOF >&2
Error, X^4 + $A*X^2 + $B has unsupported Galois group ($gal)
This program only supports Galois groups of the form D4 and C4. C2xC2
corresponds to the genus 1 case, and escapes the present scope.
EOF
            exit 2;;
    esac
    canonical="`echo "canonical_cm_parameters ($A, $B)" | $GP_PROGRAM $GP_OPTS $SHIMURA_GP`"
    if [ "$canonical" != "[$A, $B]" ] ; then
        echo "Error, parameters [$A, $B] not canonical. Use $canonical instead" >&2
        exit 2
    fi

    echo 'printf ("PARI/GP version: %s", version ())' | $GP_PROGRAM $GP_OPTS $SHIMURA_GP
    echo "Working with [D,A,B] = [$D,$A,$B]"
}

######################################################################

set -e

saved_args=("$@")

while [ "$#" -gt 0 ] ; do
    case "$1" in
        -o)             shift; wdir="$1";;
        -p)             dobase=1;;
        --full|-f)      docm2=1;;
        -c)             docheck=1;;
        -N|--no-keep)   nokeep=1;;
        -b)             shift; baseprec="$1";;
        -r)             allow_recursion=1;; # for batch checking only
        -*)     usage; echo "Unexpected argument $1" >&2; usage;;
        *) break;;
    esac
    shift
done

if ! [ "$dobase" ] && ! [ "$docm2" ] && ! [ "$docheck" ] ; then
    echo "Please give at least one of -p -f -c" >&2
    usage
    exit 2
fi

set_DAB_from_anything $*

if ! [ -d $wdir ] ; then
    mkdir -p $wdir
fi

######################################################################
# base computation

gp_code() {
    cat <<EOT
cm = 0;
cm = init_cmfield ($A, $B);
if(!cm,quit(1););
/* print_cmfield(cm); */
print_period_matrices (cm, 1);
EOT
}

if [ "$dobase" ] ; then
    OLDPWD=$PWD
    cd $wdir
    rm -f -- "${D}_${A}_${B}.in"
    export DATA_PREFIX=/tmp/
    set +e
    gp_code | $GP_PROGRAM $GP_OPTS $SHIMURA_GP > >(tee "${D}_${A}_${B}.gp.log") 2>&1
    rc=$?
    if [ $rc != 0 ] ; then
        echo "gp returned $rc" >&2
        exit 10
    fi
    set -e
    if test ${DATA_PREFIX} != `pwd`/ ; then
        mv -f "${DATA_PREFIX}${D}_${A}_${B}.in" "${D}_${A}_${B}.in"
    fi
    cd $OLDPWD
fi

######################################################################
# class polynomial computation

if [ "$docm2" ] ; then
    if ! [ -f "$wdir/${D}_${A}_${B}.in" ] ; then
        echo "$wdir/${D}_${A}_${B}.in is missing. Use -p first" >&2
        usage
    fi

    cd $wdir
    extra=
    if [ "$nokeep" ] ; then
        extra="--no-checkpoints"
    fi
    if [ "$baseprec" ] ; then
        extra="$extra -b $baseprec"
    fi
    set +e
    $CM2_BIN $extra -i ${D}_${A}_${B}.in -o ${D}_${A}_${B}.pol 2>&1 | tee ${D}_${A}_${B}.out
    rc=$?
    if [ $rc != 0 ] ; then
        echo "cm2 returned $rc" >&2
        exit 11
    fi
    set -e
fi

######################################################################

: ${cosetnum:=0}

gp_check_code() {
    # We must avoid primes which divide denominators.
    cat <<EOT
cm = 0;
cm = init_cmfield ($A, $B);
ok = checkpol(cm, $p0, 0, $cosetnum, "$wdir/");
if (type(ok)=="t_VEC", print("check: ok $A, $B");, print("check: FAIL $A, $B");quit(1););
EOT
}

recurse_if_relevant() {
    if [ "$allow_recursion" ] && ! [ "$baseprec" ] ; then
        echo "##### Trying with higher base precision ######"
        exec $0 -b 3000 "${saved_args[@]}"
    fi
}


if [ "$docheck" ] ; then
    if ! [ -f "$wdir/${D}_${A}_${B}.pol" ] ; then
        echo "$wdir/${D}_${A}_${B}.pol is missing. Use -p -f first" >&2
        usage
    fi

    set +e
    gp_check_code | CMH_PKGLIBDIR=$CMH_PKGLIBDIR $GP_PROGRAM $GP_OPTS $SHIMURA_GP $CHECKPOL_GP > >(tee "$wdir/${D}_${A}_${B}.check.log") 2>&1
    rc=$?
    if [ $rc != 0 ] ; then
        echo "return code from gp: $rc"
        if [ $rc = 2 ] ; then
            if [ "`type -p magma`" ] ; then
                echo "Attempting magma check";
                perl -ne '$x=1 if /magma check/; print if $x;' < "$wdir/${D}_${A}_${B}.check.log" | magma -b > >(tee -a "$wdir/${D}_${A}_${B}.check.log") 2>&1
                if grep -q "^check: ok" "$wdir/${D}_${A}_${B}.check.log" ; then
                    exit 0
                else
                    recurse_if_relevant
                    exit 21
                fi
            fi
            exit 20
        else
            recurse_if_relevant
            exit 12
        fi
    fi
    set -e
fi
