/*
 * See:
 *  https://developer.apple.com/legacy/library/technotes/tn/tn1150.html
 *  http://search.cpan.org/~wiml/Mac-Finder-DSStore/DSStoreFormat.pod
 *
 * "My [wiml] guess is that the string comparison follows the same rules as
 * HFS+ described in Apple's TN1150."
 *
 * TN1150 contains a FastUnicodeCompare routine defined as the "HFS Plus
 * case-insensitive string comparison algorithm".
 *
 * TN1150 specifies that "Unicode strings will be stored in fully decomposed
 * form, with composing characters stored in canonical order".
 *
 */

var unorm = require('unorm')

var lowerCaseMap = new Map([
  [0x0000, 0xffff],
  [0x0041, 0x0061],
  [0x0042, 0x0062],
  [0x0043, 0x0063],
  [0x0044, 0x0064],
  [0x0045, 0x0065],
  [0x0046, 0x0066],
  [0x0047, 0x0067],
  [0x0048, 0x0068],
  [0x0049, 0x0069],
  [0x004a, 0x006a],
  [0x004b, 0x006b],
  [0x004c, 0x006c],
  [0x004d, 0x006d],
  [0x004e, 0x006e],
  [0x004f, 0x006f],
  [0x0050, 0x0070],
  [0x0051, 0x0071],
  [0x0052, 0x0072],
  [0x0053, 0x0073],
  [0x0054, 0x0074],
  [0x0055, 0x0075],
  [0x0056, 0x0076],
  [0x0057, 0x0077],
  [0x0058, 0x0078],
  [0x0059, 0x0079],
  [0x005a, 0x007a],
  [0x00c6, 0x00e6],
  [0x00d0, 0x00f0],
  [0x00d8, 0x00f8],
  [0x00de, 0x00fe],
  [0x0110, 0x0111],
  [0x0126, 0x0127],
  [0x0132, 0x0133],
  [0x013f, 0x0140],
  [0x0141, 0x0142],
  [0x014a, 0x014b],
  [0x0152, 0x0153],
  [0x0166, 0x0167],
  [0x0181, 0x0253],
  [0x0182, 0x0183],
  [0x0184, 0x0185],
  [0x0186, 0x0254],
  [0x0187, 0x0188],
  [0x0189, 0x0256],
  [0x018a, 0x0257],
  [0x018b, 0x018c],
  [0x018e, 0x01dd],
  [0x018f, 0x0259],
  [0x0190, 0x025b],
  [0x0191, 0x0192],
  [0x0193, 0x0260],
  [0x0194, 0x0263],
  [0x0196, 0x0269],
  [0x0197, 0x0268],
  [0x0198, 0x0199],
  [0x019c, 0x026f],
  [0x019d, 0x0272],
  [0x019f, 0x0275],
  [0x01a2, 0x01a3],
  [0x01a4, 0x01a5],
  [0x01a7, 0x01a8],
  [0x01a9, 0x0283],
  [0x01ac, 0x01ad],
  [0x01ae, 0x0288],
  [0x01b1, 0x028a],
  [0x01b2, 0x028b],
  [0x01b3, 0x01b4],
  [0x01b5, 0x01b6],
  [0x01b7, 0x0292],
  [0x01b8, 0x01b9],
  [0x01bc, 0x01bd],
  [0x01c4, 0x01c6],
  [0x01c5, 0x01c6],
  [0x01c7, 0x01c9],
  [0x01c8, 0x01c9],
  [0x01ca, 0x01cc],
  [0x01cb, 0x01cc],
  [0x01e4, 0x01e5],
  [0x01f1, 0x01f3],
  [0x01f2, 0x01f3],
  [0x0391, 0x03b1],
  [0x0392, 0x03b2],
  [0x0393, 0x03b3],
  [0x0394, 0x03b4],
  [0x0395, 0x03b5],
  [0x0396, 0x03b6],
  [0x0397, 0x03b7],
  [0x0398, 0x03b8],
  [0x0399, 0x03b9],
  [0x039a, 0x03ba],
  [0x039b, 0x03bb],
  [0x039c, 0x03bc],
  [0x039d, 0x03bd],
  [0x039e, 0x03be],
  [0x039f, 0x03bf],
  [0x03a0, 0x03c0],
  [0x03a1, 0x03c1],
  [0x03a3, 0x03c3],
  [0x03a4, 0x03c4],
  [0x03a5, 0x03c5],
  [0x03a6, 0x03c6],
  [0x03a7, 0x03c7],
  [0x03a8, 0x03c8],
  [0x03a9, 0x03c9],
  [0x03e2, 0x03e3],
  [0x03e4, 0x03e5],
  [0x03e6, 0x03e7],
  [0x03e8, 0x03e9],
  [0x03ea, 0x03eb],
  [0x03ec, 0x03ed],
  [0x03ee, 0x03ef],
  [0x0402, 0x0452],
  [0x0404, 0x0454],
  [0x0405, 0x0455],
  [0x0406, 0x0456],
  [0x0408, 0x0458],
  [0x0409, 0x0459],
  [0x040a, 0x045a],
  [0x040b, 0x045b],
  [0x040f, 0x045f],
  [0x0410, 0x0430],
  [0x0411, 0x0431],
  [0x0412, 0x0432],
  [0x0413, 0x0433],
  [0x0414, 0x0434],
  [0x0415, 0x0435],
  [0x0416, 0x0436],
  [0x0417, 0x0437],
  [0x0418, 0x0438],
  [0x041a, 0x043a],
  [0x041b, 0x043b],
  [0x041c, 0x043c],
  [0x041d, 0x043d],
  [0x041e, 0x043e],
  [0x041f, 0x043f],
  [0x0420, 0x0440],
  [0x0421, 0x0441],
  [0x0422, 0x0442],
  [0x0423, 0x0443],
  [0x0424, 0x0444],
  [0x0425, 0x0445],
  [0x0426, 0x0446],
  [0x0427, 0x0447],
  [0x0428, 0x0448],
  [0x0429, 0x0449],
  [0x042a, 0x044a],
  [0x042b, 0x044b],
  [0x042c, 0x044c],
  [0x042d, 0x044d],
  [0x042e, 0x044e],
  [0x042f, 0x044f],
  [0x0460, 0x0461],
  [0x0462, 0x0463],
  [0x0464, 0x0465],
  [0x0466, 0x0467],
  [0x0468, 0x0469],
  [0x046a, 0x046b],
  [0x046c, 0x046d],
  [0x046e, 0x046f],
  [0x0470, 0x0471],
  [0x0472, 0x0473],
  [0x0474, 0x0475],
  [0x0478, 0x0479],
  [0x047a, 0x047b],
  [0x047c, 0x047d],
  [0x047e, 0x047f],
  [0x0480, 0x0481],
  [0x0490, 0x0491],
  [0x0492, 0x0493],
  [0x0494, 0x0495],
  [0x0496, 0x0497],
  [0x0498, 0x0499],
  [0x049a, 0x049b],
  [0x049c, 0x049d],
  [0x049e, 0x049f],
  [0x04a0, 0x04a1],
  [0x04a2, 0x04a3],
  [0x04a4, 0x04a5],
  [0x04a6, 0x04a7],
  [0x04a8, 0x04a9],
  [0x04aa, 0x04ab],
  [0x04ac, 0x04ad],
  [0x04ae, 0x04af],
  [0x04b0, 0x04b1],
  [0x04b2, 0x04b3],
  [0x04b4, 0x04b5],
  [0x04b6, 0x04b7],
  [0x04b8, 0x04b9],
  [0x04ba, 0x04bb],
  [0x04bc, 0x04bd],
  [0x04be, 0x04bf],
  [0x04c3, 0x04c4],
  [0x04c7, 0x04c8],
  [0x04cb, 0x04cc],
  [0x0531, 0x0561],
  [0x0532, 0x0562],
  [0x0533, 0x0563],
  [0x0534, 0x0564],
  [0x0535, 0x0565],
  [0x0536, 0x0566],
  [0x0537, 0x0567],
  [0x0538, 0x0568],
  [0x0539, 0x0569],
  [0x053a, 0x056a],
  [0x053b, 0x056b],
  [0x053c, 0x056c],
  [0x053d, 0x056d],
  [0x053e, 0x056e],
  [0x053f, 0x056f],
  [0x0540, 0x0570],
  [0x0541, 0x0571],
  [0x0542, 0x0572],
  [0x0543, 0x0573],
  [0x0544, 0x0574],
  [0x0545, 0x0575],
  [0x0546, 0x0576],
  [0x0547, 0x0577],
  [0x0548, 0x0578],
  [0x0549, 0x0579],
  [0x054a, 0x057a],
  [0x054b, 0x057b],
  [0x054c, 0x057c],
  [0x054d, 0x057d],
  [0x054e, 0x057e],
  [0x054f, 0x057f],
  [0x0550, 0x0580],
  [0x0551, 0x0581],
  [0x0552, 0x0582],
  [0x0553, 0x0583],
  [0x0554, 0x0584],
  [0x0555, 0x0585],
  [0x0556, 0x0586],
  [0x10a0, 0x10d0],
  [0x10a1, 0x10d1],
  [0x10a2, 0x10d2],
  [0x10a3, 0x10d3],
  [0x10a4, 0x10d4],
  [0x10a5, 0x10d5],
  [0x10a6, 0x10d6],
  [0x10a7, 0x10d7],
  [0x10a8, 0x10d8],
  [0x10a9, 0x10d9],
  [0x10aa, 0x10da],
  [0x10ab, 0x10db],
  [0x10ac, 0x10dc],
  [0x10ad, 0x10dd],
  [0x10ae, 0x10de],
  [0x10af, 0x10df],
  [0x10b0, 0x10e0],
  [0x10b1, 0x10e1],
  [0x10b2, 0x10e2],
  [0x10b3, 0x10e3],
  [0x10b4, 0x10e4],
  [0x10b5, 0x10e5],
  [0x10b6, 0x10e6],
  [0x10b7, 0x10e7],
  [0x10b8, 0x10e8],
  [0x10b9, 0x10e9],
  [0x10ba, 0x10ea],
  [0x10bb, 0x10eb],
  [0x10bc, 0x10ec],
  [0x10bd, 0x10ed],
  [0x10be, 0x10ee],
  [0x10bf, 0x10ef],
  [0x10c0, 0x10f0],
  [0x10c1, 0x10f1],
  [0x10c2, 0x10f2],
  [0x10c3, 0x10f3],
  [0x10c4, 0x10f4],
  [0x10c5, 0x10f5],
  [0x200c, 0x0000],
  [0x200d, 0x0000],
  [0x200e, 0x0000],
  [0x200f, 0x0000],
  [0x202a, 0x0000],
  [0x202b, 0x0000],
  [0x202c, 0x0000],
  [0x202d, 0x0000],
  [0x202e, 0x0000],
  [0x206a, 0x0000],
  [0x206b, 0x0000],
  [0x206c, 0x0000],
  [0x206d, 0x0000],
  [0x206e, 0x0000],
  [0x206f, 0x0000],
  [0x2160, 0x0010],
  [0x2161, 0x0010],
  [0x2162, 0x0010],
  [0x2163, 0x0010],
  [0x2164, 0x0010],
  [0x2165, 0x0010],
  [0x2166, 0x0010],
  [0x2167, 0x0010],
  [0x2168, 0x0010],
  [0x2169, 0x0010],
  [0x216a, 0x0010],
  [0x216b, 0x0010],
  [0x216c, 0x0010],
  [0x216d, 0x0010],
  [0x216e, 0x0010],
  [0x216f, 0x0010],
  [0xfeff, 0x0000],
  [0xff21, 0x0020],
  [0xff22, 0x0020],
  [0xff23, 0x0020],
  [0xff24, 0x0020],
  [0xff25, 0x0020],
  [0xff26, 0x0020],
  [0xff27, 0x0020],
  [0xff28, 0x0020],
  [0xff29, 0x0020],
  [0xff2a, 0x0020],
  [0xff2b, 0x0020],
  [0xff2c, 0x0020],
  [0xff2d, 0x0020],
  [0xff2e, 0x0020],
  [0xff2f, 0x0020],
  [0xff30, 0x0020],
  [0xff31, 0x0020],
  [0xff32, 0x0020],
  [0xff33, 0x0020],
  [0xff34, 0x0020],
  [0xff35, 0x0020],
  [0xff36, 0x0020],
  [0xff37, 0x0020],
  [0xff38, 0x0020],
  [0xff39, 0x0020],
  [0xff3a, 0x0020]
])

function HFSPlusFastUnicodeCompare (str1, str2) {
  var i, c1, c2

  for (i = 0; i < Math.min(str1.length, str2.length); i++) {
    c1 = str1.charCodeAt(i)
    c2 = str2.charCodeAt(i)

    if (lowerCaseMap.has(c1)) {
      c1 = lowerCaseMap.get(c1)
    }

    if (lowerCaseMap.has(c2)) {
      c2 = lowerCaseMap.get(c2)
    }

    if (c1 !== c2) {
      return c1 - c2
    }
  }

  if (str1.length !== str2.length) {
    return str1.length - str2.length
  }

  return 0
}

exports.compare = HFSPlusFastUnicodeCompare
exports.normalize = unorm.nfd
