![stl-fortran](media/logo.png)
============

Derivative free 1D function minimizer in modern Fortran

### Status

[![Language](https://img.shields.io/badge/-Fortran-734f96?logo=fortran&logoColor=white)](https://github.com/topics/fortran)
[![GitHub release](https://img.shields.io/github/release/jacobwilliams/fmin.svg)](https://github.com/jacobwilliams/fmin/releases/latest)
[![Build Status](https://github.com/jacobwilliams/fmin/actions/workflows/CI.yml/badge.svg)](https://github.com/jacobwilliams/fmin/actions)
[![codecov](https://codecov.io/gh/jacobwilliams/fmin/branch/master/graph/badge.svg)](https://codecov.io/gh/jacobwilliams/fmin)
[![last-commit](https://img.shields.io/github/last-commit/jacobwilliams/fmin)](https://github.com/jacobwilliams/fmin/commits/master)

### Compiling

A [Fortran Package Manager](https://github.com/fortran-lang/fpm) manifest file is included, so that the library and test cases can be compiled with FPM. For example:

```
fpm build --profile release
fpm test --profile release
```

By default, the library is built with double precision (`real64`) real values. Explicitly specifying the real kind can be done using the following preprocessor flags:

Preprocessor flag | Kind  | Number of bytes
----------------- | ----- | ---------------
`REAL32`  | `real(kind=real32)`  | 4
`REAL64`  | `real(kind=real64)`  | 8
`REAL128` | `real(kind=real128)` | 16

For example, to build a single precision version of the library, use:

```
fpm build --profile release --flag "-DREAL32"
```

To use `fmin` within your fpm project, add the following to your `fpm.toml` file:
```toml
[dependencies]
fmin = { git="https://github.com/jacobwilliams/fmin.git" }
```

### Example

```fortran
program test

use fmin_module
use iso_fortran_env, only: wp => real64 ! double precision

real(wp) :: xmin, xerr

real(wp),parameter :: ax  = -4.0_wp       ! lower bound
real(wp),parameter :: bx  = 0.0_wp        ! upper bound
real(wp),parameter :: tol = 1.0e-8_wp     ! tolerance
real(wp),parameter :: pi  = acos(-1.0_wp) ! pi
real(wp),parameter :: x  = -pi/2.0_wp     ! true answer

xmin = fmin(func,ax,bx,tol) ! compute the minimum

xerr = xmin - x  ! difference from true value

write(*,*) 'xmin       = ', xmin
write(*,*) 'xmin exact = ', x
write(*,*) 'xmin error = ', xerr

contains

    function func(x) result(f)

    implicit none

    real(wp),intent(in) :: x  !! indep. variable
    real(wp)            :: f  !! function value `f(x)`

    f = sin(x)

    end function func

end program test
```

The output is:

```text
 xmin       =   -1.5707963254967554
 xmin exact =   -1.5707963267948966
 xmin error =    1.2981411501300499E-009
```

### Documentation

 * The API documentation for the current ```master``` branch can be found [here](https://jacobwilliams.github.io/fmin/).  This is generated by processing the source files with [FORD](https://github.com/Fortran-FOSS-Programmers/ford).

### License

 * The Fmin source code and related files and documentation are distributed under a permissive free software [license](https://github.com/jacobwilliams/fmin/blob/master/LICENSE) (BSD-3).

### See also

  * Richard Brent, "[Algorithms for Minimization Without Derivatives](https://maths-people.anu.edu.au/~brent/pub/pub011.html)",
    Prentice - Hall, Inc. (1973)
  * [fmin from Netlib](http://www.netlib.org/fmm/fmin.f)
  * [roots-fortran](https://github.com/jacobwilliams/roots-fortran) (1D derivative-free roots solvers)
