import {
  A2AError
} from "./chunk-UHZEIZLS.js";
import {
  Extensions
} from "./chunk-ZX6KNMCP.js";

// src/server/context.ts
var ServerCallContext = class {
  _requestedExtensions;
  _user;
  _activatedExtensions;
  constructor(requestedExtensions, user) {
    this._requestedExtensions = requestedExtensions;
    this._user = user;
  }
  get user() {
    return this._user;
  }
  get activatedExtensions() {
    return this._activatedExtensions;
  }
  get requestedExtensions() {
    return this._requestedExtensions;
  }
  addActivatedExtension(uri) {
    this._activatedExtensions = Extensions.createFrom(this._activatedExtensions, uri);
  }
};

// src/server/transports/jsonrpc/jsonrpc_transport_handler.ts
var JsonRpcTransportHandler = class {
  requestHandler;
  constructor(requestHandler) {
    this.requestHandler = requestHandler;
  }
  /**
   * Handles an incoming JSON-RPC request.
   * For streaming methods, it returns an AsyncGenerator of JSONRPCResult.
   * For non-streaming methods, it returns a Promise of a single JSONRPCMessage (Result or ErrorResponse).
   */
  async handle(requestBody, context) {
    let rpcRequest;
    try {
      if (typeof requestBody === "string") {
        rpcRequest = JSON.parse(requestBody);
      } else if (typeof requestBody === "object" && requestBody !== null) {
        rpcRequest = requestBody;
      } else {
        throw A2AError.parseError("Invalid request body type.");
      }
      if (!this.isRequestValid(rpcRequest)) {
        throw A2AError.invalidRequest("Invalid JSON-RPC Request.");
      }
    } catch (error) {
      const a2aError = error instanceof A2AError ? error : A2AError.parseError(
        error instanceof SyntaxError && error.message || "Failed to parse JSON request."
      );
      return {
        jsonrpc: "2.0",
        id: rpcRequest?.id !== void 0 ? rpcRequest.id : null,
        error: a2aError.toJSONRPCError()
      };
    }
    const { method, id: requestId = null } = rpcRequest;
    try {
      if (method !== "agent/getAuthenticatedExtendedCard" && !this.paramsAreValid(rpcRequest.params)) {
        throw A2AError.invalidParams(`Invalid method parameters.`);
      }
      if (method === "message/stream" || method === "tasks/resubscribe") {
        const params = rpcRequest.params;
        const agentCard = await this.requestHandler.getAgentCard();
        if (!agentCard.capabilities.streaming) {
          throw A2AError.unsupportedOperation(`Method ${method} requires streaming capability.`);
        }
        const agentEventStream = method === "message/stream" ? this.requestHandler.sendMessageStream(params, context) : this.requestHandler.resubscribe(params, context);
        return (async function* jsonRpcEventStream() {
          try {
            for await (const event of agentEventStream) {
              yield {
                jsonrpc: "2.0",
                id: requestId,
                // Use the original request ID for all streamed responses
                result: event
              };
            }
          } catch (streamError) {
            console.error(
              `Error in agent event stream for ${method} (request ${requestId}):`,
              streamError
            );
            throw streamError;
          }
        })();
      } else {
        let result;
        switch (method) {
          case "message/send":
            result = await this.requestHandler.sendMessage(rpcRequest.params, context);
            break;
          case "tasks/get":
            result = await this.requestHandler.getTask(rpcRequest.params, context);
            break;
          case "tasks/cancel":
            result = await this.requestHandler.cancelTask(rpcRequest.params, context);
            break;
          case "tasks/pushNotificationConfig/set":
            result = await this.requestHandler.setTaskPushNotificationConfig(
              rpcRequest.params,
              context
            );
            break;
          case "tasks/pushNotificationConfig/get":
            result = await this.requestHandler.getTaskPushNotificationConfig(
              rpcRequest.params,
              context
            );
            break;
          case "tasks/pushNotificationConfig/delete":
            await this.requestHandler.deleteTaskPushNotificationConfig(rpcRequest.params, context);
            result = null;
            break;
          case "tasks/pushNotificationConfig/list":
            result = await this.requestHandler.listTaskPushNotificationConfigs(
              rpcRequest.params,
              context
            );
            break;
          case "agent/getAuthenticatedExtendedCard":
            result = await this.requestHandler.getAuthenticatedExtendedAgentCard(context);
            break;
          default:
            throw A2AError.methodNotFound(method);
        }
        return {
          jsonrpc: "2.0",
          id: requestId,
          result
        };
      }
    } catch (error) {
      let a2aError;
      if (error instanceof A2AError) {
        a2aError = error;
      } else {
        a2aError = A2AError.internalError(
          error instanceof Error && error.message || "An unexpected error occurred."
        );
      }
      return {
        jsonrpc: "2.0",
        id: requestId,
        error: a2aError.toJSONRPCError()
      };
    }
  }
  // Validates the basic structure of a JSON-RPC request
  isRequestValid(rpcRequest) {
    if (rpcRequest.jsonrpc !== "2.0") {
      return false;
    }
    if ("id" in rpcRequest) {
      const id = rpcRequest.id;
      const isString = typeof id === "string";
      const isInteger = typeof id === "number" && Number.isInteger(id);
      const isNull = id === null;
      if (!isString && !isInteger && !isNull) {
        return false;
      }
    }
    if (!rpcRequest.method || typeof rpcRequest.method !== "string") {
      return false;
    }
    return true;
  }
  // Validates that params is an object with non-empty string keys
  paramsAreValid(params) {
    if (typeof params !== "object" || params === null || Array.isArray(params)) {
      return false;
    }
    for (const key of Object.keys(params)) {
      if (key === "") {
        return false;
      }
    }
    return true;
  }
};

// src/server/authentication/user.ts
var UnauthenticatedUser = class {
  get isAuthenticated() {
    return false;
  }
  get userName() {
    return "";
  }
};

export {
  ServerCallContext,
  JsonRpcTransportHandler,
  UnauthenticatedUser
};
