/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { resolveColor, interpolateColor, getThemeTypeFromBackgroundColor, } from './color-utils.js';
export const lightTheme = {
    type: 'light',
    Background: '#FAFAFA',
    Foreground: '',
    LightBlue: '#89BDCD',
    AccentBlue: '#3B82F6',
    AccentPurple: '#8B5CF6',
    AccentCyan: '#06B6D4',
    AccentGreen: '#3CA84B',
    AccentYellow: '#D5A40A',
    AccentRed: '#DD4C4C',
    DiffAdded: '#C6EAD8',
    DiffRemoved: '#FFCCCC',
    Comment: '#008000',
    Gray: '#97a0b0',
    DarkGray: interpolateColor('#97a0b0', '#FAFAFA', 0.5),
    GradientColors: ['#4796E4', '#847ACE', '#C3677F'],
};
export const darkTheme = {
    type: 'dark',
    Background: '#1E1E2E',
    Foreground: '',
    LightBlue: '#ADD8E6',
    AccentBlue: '#89B4FA',
    AccentPurple: '#CBA6F7',
    AccentCyan: '#89DCEB',
    AccentGreen: '#A6E3A1',
    AccentYellow: '#F9E2AF',
    AccentRed: '#F38BA8',
    DiffAdded: '#28350B',
    DiffRemoved: '#430000',
    Comment: '#6C7086',
    Gray: '#6C7086',
    DarkGray: interpolateColor('#6C7086', '#1E1E2E', 0.5),
    GradientColors: ['#4796E4', '#847ACE', '#C3677F'],
};
export const ansiTheme = {
    type: 'ansi',
    Background: 'black',
    Foreground: '',
    LightBlue: 'blue',
    AccentBlue: 'blue',
    AccentPurple: 'magenta',
    AccentCyan: 'cyan',
    AccentGreen: 'green',
    AccentYellow: 'yellow',
    AccentRed: 'red',
    DiffAdded: 'green',
    DiffRemoved: 'red',
    Comment: 'gray',
    Gray: 'gray',
    DarkGray: 'gray',
};
export class Theme {
    name;
    type;
    colors;
    /**
     * The default foreground color for text when no specific highlight rule applies.
     * This is an Ink-compatible color string (hex or name).
     */
    defaultColor;
    /**
     * Stores the mapping from highlight.js class names (e.g., 'hljs-keyword')
     * to Ink-compatible color strings (hex or name).
     */
    _colorMap;
    semanticColors;
    /**
     * Creates a new Theme instance.
     * @param name The name of the theme.
     * @param rawMappings The raw CSSProperties mappings from a react-syntax-highlighter theme object.
     */
    constructor(name, type, rawMappings, colors, semanticColors) {
        this.name = name;
        this.type = type;
        this.colors = colors;
        this.semanticColors = semanticColors ?? {
            text: {
                primary: this.colors.Foreground,
                secondary: this.colors.Gray,
                link: this.colors.AccentBlue,
                accent: this.colors.AccentPurple,
                response: this.colors.Foreground,
            },
            background: {
                primary: this.colors.Background,
                diff: {
                    added: this.colors.DiffAdded,
                    removed: this.colors.DiffRemoved,
                },
            },
            border: {
                default: this.colors.Gray,
                focused: this.colors.AccentBlue,
            },
            ui: {
                comment: this.colors.Gray,
                symbol: this.colors.AccentCyan,
                dark: this.colors.DarkGray,
                gradient: this.colors.GradientColors,
            },
            status: {
                error: this.colors.AccentRed,
                success: this.colors.AccentGreen,
                warning: this.colors.AccentYellow,
            },
        };
        this._colorMap = Object.freeze(this._buildColorMap(rawMappings)); // Build and freeze the map
        // Determine the default foreground color
        const rawDefaultColor = rawMappings['hljs']?.color;
        this.defaultColor =
            (rawDefaultColor ? Theme._resolveColor(rawDefaultColor) : undefined) ??
                ''; // Default to empty string if not found or resolvable
    }
    /**
     * Gets the Ink-compatible color string for a given highlight.js class name.
     * @param hljsClass The highlight.js class name (e.g., 'hljs-keyword', 'hljs-string').
     * @returns The corresponding Ink color string (hex or name) if it exists.
     */
    getInkColor(hljsClass) {
        return this._colorMap[hljsClass];
    }
    /**
     * Resolves a CSS color value (name or hex) into an Ink-compatible color string.
     * @param colorValue The raw color string (e.g., 'blue', '#ff0000', 'darkkhaki').
     * @returns An Ink-compatible color string (hex or name), or undefined if not resolvable.
     */
    static _resolveColor(colorValue) {
        return resolveColor(colorValue);
    }
    /**
     * Builds the internal map from highlight.js class names to Ink-compatible color strings.
     * This method is protected and primarily intended for use by the constructor.
     * @param hljsTheme The raw CSSProperties mappings from a react-syntax-highlighter theme object.
     * @returns An Ink-compatible theme map (Record<string, string>).
     */
    _buildColorMap(hljsTheme) {
        const inkTheme = {};
        for (const key in hljsTheme) {
            // Ensure the key starts with 'hljs-' or is 'hljs' for the base style
            if (!key.startsWith('hljs-') && key !== 'hljs') {
                continue; // Skip keys not related to highlighting classes
            }
            const style = hljsTheme[key];
            if (style?.color) {
                const resolvedColor = Theme._resolveColor(style.color);
                if (resolvedColor !== undefined) {
                    // Use the original key from the hljsTheme (e.g., 'hljs-keyword')
                    inkTheme[key] = resolvedColor;
                }
                // If color is not resolvable, it's omitted from the map,
                // this enables falling back to the default foreground color.
            }
            // We currently only care about the 'color' property for Ink rendering.
            // Other properties like background, fontStyle, etc., are ignored.
        }
        return inkTheme;
    }
}
/**
 * Creates a Theme instance from a custom theme configuration.
 * @param customTheme The custom theme configuration.
 * @returns A new Theme instance.
 */
export function createCustomTheme(customTheme) {
    const colors = {
        type: 'custom',
        Background: customTheme.background?.primary ?? customTheme.Background ?? '',
        Foreground: customTheme.text?.primary ?? customTheme.Foreground ?? '',
        LightBlue: customTheme.text?.link ?? customTheme.LightBlue ?? '',
        AccentBlue: customTheme.text?.link ?? customTheme.AccentBlue ?? '',
        AccentPurple: customTheme.text?.accent ?? customTheme.AccentPurple ?? '',
        AccentCyan: customTheme.text?.link ?? customTheme.AccentCyan ?? '',
        AccentGreen: customTheme.status?.success ?? customTheme.AccentGreen ?? '',
        AccentYellow: customTheme.status?.warning ?? customTheme.AccentYellow ?? '',
        AccentRed: customTheme.status?.error ?? customTheme.AccentRed ?? '',
        DiffAdded: customTheme.background?.diff?.added ?? customTheme.DiffAdded ?? '',
        DiffRemoved: customTheme.background?.diff?.removed ?? customTheme.DiffRemoved ?? '',
        Comment: customTheme.ui?.comment ?? customTheme.Comment ?? '',
        Gray: customTheme.text?.secondary ?? customTheme.Gray ?? '',
        DarkGray: customTheme.DarkGray ??
            interpolateColor(customTheme.text?.secondary ?? customTheme.Gray ?? '', customTheme.background?.primary ?? customTheme.Background ?? '', 0.5),
        GradientColors: customTheme.ui?.gradient ?? customTheme.GradientColors,
    };
    // Generate CSS properties mappings based on the custom theme colors
    const rawMappings = {
        hljs: {
            display: 'block',
            overflowX: 'auto',
            padding: '0.5em',
            background: colors.Background,
            color: colors.Foreground,
        },
        'hljs-keyword': {
            color: colors.AccentBlue,
        },
        'hljs-literal': {
            color: colors.AccentBlue,
        },
        'hljs-symbol': {
            color: colors.AccentBlue,
        },
        'hljs-name': {
            color: colors.AccentBlue,
        },
        'hljs-link': {
            color: colors.AccentBlue,
            textDecoration: 'underline',
        },
        'hljs-built_in': {
            color: colors.AccentCyan,
        },
        'hljs-type': {
            color: colors.AccentCyan,
        },
        'hljs-number': {
            color: colors.AccentGreen,
        },
        'hljs-class': {
            color: colors.AccentGreen,
        },
        'hljs-string': {
            color: colors.AccentYellow,
        },
        'hljs-meta-string': {
            color: colors.AccentYellow,
        },
        'hljs-regexp': {
            color: colors.AccentRed,
        },
        'hljs-template-tag': {
            color: colors.AccentRed,
        },
        'hljs-subst': {
            color: colors.Foreground,
        },
        'hljs-function': {
            color: colors.Foreground,
        },
        'hljs-title': {
            color: colors.Foreground,
        },
        'hljs-params': {
            color: colors.Foreground,
        },
        'hljs-formula': {
            color: colors.Foreground,
        },
        'hljs-comment': {
            color: colors.Comment,
            fontStyle: 'italic',
        },
        'hljs-quote': {
            color: colors.Comment,
            fontStyle: 'italic',
        },
        'hljs-doctag': {
            color: colors.Comment,
        },
        'hljs-meta': {
            color: colors.Gray,
        },
        'hljs-meta-keyword': {
            color: colors.Gray,
        },
        'hljs-tag': {
            color: colors.Gray,
        },
        'hljs-variable': {
            color: colors.AccentPurple,
        },
        'hljs-template-variable': {
            color: colors.AccentPurple,
        },
        'hljs-attr': {
            color: colors.LightBlue,
        },
        'hljs-attribute': {
            color: colors.LightBlue,
        },
        'hljs-builtin-name': {
            color: colors.LightBlue,
        },
        'hljs-section': {
            color: colors.AccentYellow,
        },
        'hljs-emphasis': {
            fontStyle: 'italic',
        },
        'hljs-strong': {
            fontWeight: 'bold',
        },
        'hljs-bullet': {
            color: colors.AccentYellow,
        },
        'hljs-selector-tag': {
            color: colors.AccentYellow,
        },
        'hljs-selector-id': {
            color: colors.AccentYellow,
        },
        'hljs-selector-class': {
            color: colors.AccentYellow,
        },
        'hljs-selector-attr': {
            color: colors.AccentYellow,
        },
        'hljs-selector-pseudo': {
            color: colors.AccentYellow,
        },
        'hljs-addition': {
            backgroundColor: colors.AccentGreen,
            display: 'inline-block',
            width: '100%',
        },
        'hljs-deletion': {
            backgroundColor: colors.AccentRed,
            display: 'inline-block',
            width: '100%',
        },
    };
    const semanticColors = {
        text: {
            primary: customTheme.text?.primary ?? colors.Foreground,
            secondary: customTheme.text?.secondary ?? colors.Gray,
            link: customTheme.text?.link ?? colors.AccentBlue,
            accent: customTheme.text?.accent ?? colors.AccentPurple,
            response: customTheme.text?.response ??
                customTheme.text?.primary ??
                colors.Foreground,
        },
        background: {
            primary: customTheme.background?.primary ?? colors.Background,
            diff: {
                added: customTheme.background?.diff?.added ?? colors.DiffAdded,
                removed: customTheme.background?.diff?.removed ?? colors.DiffRemoved,
            },
        },
        border: {
            default: customTheme.border?.default ?? colors.Gray,
            focused: customTheme.border?.focused ?? colors.AccentBlue,
        },
        ui: {
            comment: customTheme.ui?.comment ?? colors.Comment,
            symbol: customTheme.ui?.symbol ?? colors.Gray,
            dark: colors.DarkGray,
            gradient: customTheme.ui?.gradient ?? colors.GradientColors,
        },
        status: {
            error: customTheme.status?.error ?? colors.AccentRed,
            success: customTheme.status?.success ?? colors.AccentGreen,
            warning: customTheme.status?.warning ?? colors.AccentYellow,
        },
    };
    return new Theme(customTheme.name, 'custom', rawMappings, colors, semanticColors);
}
/**
 * Validates a custom theme configuration.
 * @param customTheme The custom theme to validate.
 * @returns An object with isValid boolean and error message if invalid.
 */
export function validateCustomTheme(customTheme) {
    // Since all fields are optional, we only need to validate the name.
    if (customTheme.name && !isValidThemeName(customTheme.name)) {
        return {
            isValid: false,
            error: `Invalid theme name: ${customTheme.name}`,
        };
    }
    return {
        isValid: true,
    };
}
/**
 * Checks if a theme name is valid.
 * @param name The theme name to validate.
 * @returns True if the theme name is valid.
 */
function isValidThemeName(name) {
    // Theme name should be non-empty and not contain invalid characters
    return name.trim().length > 0 && name.trim().length <= 50;
}
/**
 * Picks a default theme name based on terminal background color.
 * It first tries to find a theme with an exact background color match.
 * If no match is found, it falls back to a light or dark theme based on the
 * luminance of the background color.
 * @param terminalBackground The hex color string of the terminal background.
 * @param availableThemes A list of available themes to search through.
 * @param defaultDarkThemeName The name of the fallback dark theme.
 * @param defaultLightThemeName The name of the fallback light theme.
 * @returns The name of the chosen theme.
 */
export function pickDefaultThemeName(terminalBackground, availableThemes, defaultDarkThemeName, defaultLightThemeName) {
    if (terminalBackground) {
        const lowerTerminalBackground = terminalBackground.toLowerCase();
        for (const theme of availableThemes) {
            if (!theme.colors.Background)
                continue;
            // resolveColor can return undefined
            const themeBg = resolveColor(theme.colors.Background)?.toLowerCase();
            if (themeBg === lowerTerminalBackground) {
                return theme.name;
            }
        }
    }
    const themeType = getThemeTypeFromBackgroundColor(terminalBackground);
    if (themeType === 'light') {
        return defaultLightThemeName;
    }
    return defaultDarkThemeName;
}
//# sourceMappingURL=theme.js.map