#!/usr/bin/env node
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { main } from './src/gemini.js';
import { FatalError, writeToStderr } from '@google/gemini-cli-core';
import { runExitCleanup } from './src/utils/cleanup.js';
// --- Global Entry Point ---
// Suppress known race condition error in node-pty on Windows
// Tracking bug: https://github.com/microsoft/node-pty/issues/827
process.on('uncaughtException', (error) => {
    if (process.platform === 'win32' &&
        error instanceof Error &&
        error.message === 'Cannot resize a pty that has already exited') {
        // This error happens on Windows with node-pty when resizing a pty that has just exited.
        // It is a race condition in node-pty that we cannot prevent, so we silence it.
        return;
    }
    // For other errors, we rely on the default behavior, but since we attached a listener,
    // we must manually replicate it.
    if (error instanceof Error) {
        writeToStderr(error.stack + '\n');
    }
    else {
        writeToStderr(String(error) + '\n');
    }
    process.exit(1);
});
main().catch(async (error) => {
    await runExitCleanup();
    if (error instanceof FatalError) {
        let errorMessage = error.message;
        if (!process.env['NO_COLOR']) {
            errorMessage = `\x1b[31m${errorMessage}\x1b[0m`;
        }
        writeToStderr(errorMessage + '\n');
        process.exit(error.exitCode);
    }
    writeToStderr('An unexpected critical error occurred:');
    if (error instanceof Error) {
        writeToStderr(error.stack + '\n');
    }
    else {
        writeToStderr(String(error) + '\n');
    }
    process.exit(1);
});
//# sourceMappingURL=index.js.map