/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
export const PREVIEW_GEMINI_MODEL = 'gemini-3-pro-preview';
export const PREVIEW_GEMINI_FLASH_MODEL = 'gemini-3-flash-preview';
export const DEFAULT_GEMINI_MODEL = 'gemini-2.5-pro';
export const DEFAULT_GEMINI_FLASH_MODEL = 'gemini-2.5-flash';
export const DEFAULT_GEMINI_FLASH_LITE_MODEL = 'gemini-2.5-flash-lite';
export const VALID_GEMINI_MODELS = new Set([
    PREVIEW_GEMINI_MODEL,
    PREVIEW_GEMINI_FLASH_MODEL,
    DEFAULT_GEMINI_MODEL,
    DEFAULT_GEMINI_FLASH_MODEL,
    DEFAULT_GEMINI_FLASH_LITE_MODEL,
]);
export const PREVIEW_GEMINI_MODEL_AUTO = 'auto-gemini-3';
export const DEFAULT_GEMINI_MODEL_AUTO = 'auto-gemini-2.5';
// Model aliases for user convenience.
export const GEMINI_MODEL_ALIAS_AUTO = 'auto';
export const GEMINI_MODEL_ALIAS_PRO = 'pro';
export const GEMINI_MODEL_ALIAS_FLASH = 'flash';
export const GEMINI_MODEL_ALIAS_FLASH_LITE = 'flash-lite';
export const DEFAULT_GEMINI_EMBEDDING_MODEL = 'gemini-embedding-001';
// Cap the thinking at 8192 to prevent run-away thinking loops.
export const DEFAULT_THINKING_MODE = 8192;
/**
 * Resolves the requested model alias (e.g., 'auto-gemini-3', 'pro', 'flash', 'flash-lite')
 * to a concrete model name, considering preview features.
 *
 * @param requestedModel The model alias or concrete model name requested by the user.
 * @param previewFeaturesEnabled A boolean indicating if preview features are enabled.
 * @returns The resolved concrete model name.
 */
export function resolveModel(requestedModel, previewFeaturesEnabled = false) {
    switch (requestedModel) {
        case PREVIEW_GEMINI_MODEL_AUTO: {
            return PREVIEW_GEMINI_MODEL;
        }
        case DEFAULT_GEMINI_MODEL_AUTO: {
            return DEFAULT_GEMINI_MODEL;
        }
        case GEMINI_MODEL_ALIAS_AUTO:
        case GEMINI_MODEL_ALIAS_PRO: {
            return previewFeaturesEnabled
                ? PREVIEW_GEMINI_MODEL
                : DEFAULT_GEMINI_MODEL;
        }
        case GEMINI_MODEL_ALIAS_FLASH: {
            return previewFeaturesEnabled
                ? PREVIEW_GEMINI_FLASH_MODEL
                : DEFAULT_GEMINI_FLASH_MODEL;
        }
        case GEMINI_MODEL_ALIAS_FLASH_LITE: {
            return DEFAULT_GEMINI_FLASH_LITE_MODEL;
        }
        default: {
            return requestedModel;
        }
    }
}
/**
 * Resolves the appropriate model based on the classifier's decision.
 *
 * @param requestedModel The current requested model (e.g. auto-gemini-2.5).
 * @param modelAlias The alias selected by the classifier ('flash' or 'pro').
 * @param previewFeaturesEnabled Whether preview features are enabled.
 * @returns The resolved concrete model name.
 */
export function resolveClassifierModel(requestedModel, modelAlias, previewFeaturesEnabled = false) {
    if (modelAlias === GEMINI_MODEL_ALIAS_FLASH) {
        if (requestedModel === DEFAULT_GEMINI_MODEL_AUTO ||
            requestedModel === DEFAULT_GEMINI_MODEL) {
            return DEFAULT_GEMINI_FLASH_MODEL;
        }
        if (requestedModel === PREVIEW_GEMINI_MODEL_AUTO ||
            requestedModel === PREVIEW_GEMINI_MODEL) {
            return PREVIEW_GEMINI_FLASH_MODEL;
        }
        return resolveModel(GEMINI_MODEL_ALIAS_FLASH, previewFeaturesEnabled);
    }
    return resolveModel(requestedModel, previewFeaturesEnabled);
}
export function getDisplayString(model, previewFeaturesEnabled = false) {
    switch (model) {
        case PREVIEW_GEMINI_MODEL_AUTO:
            return 'Auto (Gemini 3)';
        case DEFAULT_GEMINI_MODEL_AUTO:
            return 'Auto (Gemini 2.5)';
        case GEMINI_MODEL_ALIAS_PRO:
            return previewFeaturesEnabled
                ? PREVIEW_GEMINI_MODEL
                : DEFAULT_GEMINI_MODEL;
        case GEMINI_MODEL_ALIAS_FLASH:
            return previewFeaturesEnabled
                ? PREVIEW_GEMINI_FLASH_MODEL
                : DEFAULT_GEMINI_FLASH_MODEL;
        default:
            return model;
    }
}
/**
 * Checks if the model is a preview model.
 *
 * @param model The model name to check.
 * @returns True if the model is a preview model.
 */
export function isPreviewModel(model) {
    return (model === PREVIEW_GEMINI_MODEL ||
        model === PREVIEW_GEMINI_FLASH_MODEL ||
        model === PREVIEW_GEMINI_MODEL_AUTO);
}
/**
 * Checks if the model is a Gemini 2.x model.
 *
 * @param model The model name to check.
 * @returns True if the model is a Gemini-2.x model.
 */
export function isGemini2Model(model) {
    return /^gemini-2(\.|$)/.test(model);
}
/**
 * Checks if the model is an auto model.
 *
 * @param model The model name to check.
 * @returns True if the model is an auto model.
 */
export function isAutoModel(model) {
    return (model === GEMINI_MODEL_ALIAS_AUTO ||
        model === PREVIEW_GEMINI_MODEL_AUTO ||
        model === DEFAULT_GEMINI_MODEL_AUTO);
}
/**
 * Checks if the model supports multimodal function responses (multimodal data nested within function response).
 * This is supported in Gemini 3.
 *
 * @param model The model name to check.
 * @returns True if the model supports multimodal function responses.
 */
export function supportsMultimodalFunctionResponse(model) {
    return model.startsWith('gemini-3-');
}
//# sourceMappingURL=models.js.map