/*
 * Library to support cross-platform C split string functions
 *
 * Copyright (C) 2008-2022, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#if !defined( _LIBCSPLIT_H )
#define _LIBCSPLIT_H

#include <libcsplit/definitions.h>
#include <libcsplit/error.h>
#include <libcsplit/extern.h>
#include <libcsplit/features.h>
#include <libcsplit/types.h>

#include <stdio.h>

#if defined( __cplusplus )
extern "C" {
#endif

/* -------------------------------------------------------------------------
 * Support functions
 * ------------------------------------------------------------------------- */

/* Returns the library version as a string
 */
LIBCSPLIT_EXTERN \
const char *libcsplit_get_version(
             void );

/* -------------------------------------------------------------------------
 * Error functions
 * ------------------------------------------------------------------------- */

/* Frees an error
 */
LIBCSPLIT_EXTERN \
void libcsplit_error_free(
      libcsplit_error_t **error );

/* Prints a descriptive string of the error to the stream
 * Returns the number of printed characters if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_error_fprint(
     libcsplit_error_t *error,
     FILE *stream );

/* Prints a descriptive string of the error to the string
 * Returns the number of printed characters if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_error_sprint(
     libcsplit_error_t *error,
     char *string,
     size_t size );

/* Prints a backtrace of the error to the stream
 * Returns the number of printed characters if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_error_backtrace_fprint(
     libcsplit_error_t *error,
     FILE *stream );

/* Prints a backtrace of the error to the string
 * Returns the number of printed characters if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_error_backtrace_sprint(
     libcsplit_error_t *error,
     char *string,
     size_t size );

/* -------------------------------------------------------------------------
 * Narrow string functions
 * ------------------------------------------------------------------------- */

/* Splits a narrow character string
 * Make sure the value split_string is referencing, is set to NULL
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_narrow_string_split(
     const char *string,
     size_t string_size,
     char delimiter,
     libcsplit_narrow_split_string_t **split_string,
     libcsplit_error_t **error );

/* -------------------------------------------------------------------------
 * Narrow split string functions
 * ------------------------------------------------------------------------- */

/* Frees a split string
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_narrow_split_string_free(
     libcsplit_narrow_split_string_t **split_string,
     libcsplit_error_t **error );

/* Retrieves the string
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_narrow_split_string_get_string(
     libcsplit_narrow_split_string_t *split_string,
     char **string,
     size_t *string_size,
     libcsplit_error_t **error );

/* Retrieves the number of segments
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_narrow_split_string_get_number_of_segments(
     libcsplit_narrow_split_string_t *split_string,
     int *number_of_segments,
     libcsplit_error_t **error );

/* Retrieves a specific segment
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_narrow_split_string_get_segment_by_index(
     libcsplit_narrow_split_string_t *split_string,
     int segment_index,
     char **string_segment,
     size_t *string_segment_size,
     libcsplit_error_t **error );

/* Sets a specific segment
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_narrow_split_string_set_segment_by_index(
     libcsplit_narrow_split_string_t *split_string,
     int segment_index,
     char *string_segment,
     size_t string_segment_size,
     libcsplit_error_t **error );

#if defined( LIBCSPLIT_HAVE_WIDE_CHARACTER_TYPE )

/* -------------------------------------------------------------------------
 * Wide string functions
 * ------------------------------------------------------------------------- */

/* Splits a wide character string
 * Make sure the value split_string is referencing, is set to NULL
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_wide_string_split(
     const wchar_t *string,
     size_t string_size,
     wchar_t delimiter,
     libcsplit_wide_split_string_t **split_string,
     libcsplit_error_t **error );

/* -------------------------------------------------------------------------
 * Wide split string functions
 * ------------------------------------------------------------------------- */

/* Frees a split string
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_wide_split_string_free(
     libcsplit_wide_split_string_t **split_string,
     libcsplit_error_t **error );

/* Retrieves the string
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_wide_split_string_get_string(
     libcsplit_wide_split_string_t *split_string,
     wchar_t **string,
     size_t *string_size,
     libcsplit_error_t **error );

/* Retrieves the number of segments
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_wide_split_string_get_number_of_segments(
     libcsplit_wide_split_string_t *split_string,
     int *number_of_segments,
     libcsplit_error_t **error );

/* Retrieves a specific segment
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_wide_split_string_get_segment_by_index(
     libcsplit_wide_split_string_t *split_string,
     int segment_index,
     wchar_t **string_segment,
     size_t *string_segment_size,
     libcsplit_error_t **error );

/* Sets a specific segment
 * Returns 1 if successful or -1 on error
 */
LIBCSPLIT_EXTERN \
int libcsplit_wide_split_string_set_segment_by_index(
     libcsplit_wide_split_string_t *split_string,
     int segment_index,
     wchar_t *string_segment,
     size_t string_segment_size,
     libcsplit_error_t **error );

#endif /* defined( LIBCSPLIT_HAVE_WIDE_CHARACTER_TYPE ) */

#if defined( __cplusplus )
}
#endif

#endif /* !defined( _LIBCSPLIT_H ) */

