'use strict';

Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const react = require('react');
const remoteReceived = require('./hooks/remote-received.cjs');
const propsForElement = require('./hooks/props-for-element.cjs');
const constants = require('./constants.cjs');
const jsxRuntime = require('react/jsx-runtime');

/**
 * Takes a React component, and returns a new component that can be used to render that
 * component from a remote element. This wrapper will handle subscribing to changes in that
 * element, and will normalize the properties on that remote element before passing them
 * to your React component.
 */
function createRemoteComponentRenderer(Component, {
  name,
  eventProps
} = {}) {
  const RemoteComponentRenderer = /*#__PURE__*/react.memo(function RemoteComponentRenderer({
    element,
    receiver,
    components
  }) {
    const internalsRef = react.useRef();
    const attachedElement = remoteReceived.useRemoteReceived(element, receiver);
    const resolvedElement = attachedElement ?? element;
    const resolvedId = resolvedElement.id;
    const props = propsForElement.usePropsForRemoteElement(resolvedElement, {
      receiver,
      components,
      eventProps
    });
    props[constants.REMOTE_ELEMENT_PROP] = resolvedElement;
    props[constants.REMOTE_ELEMENT_ATTACHED_PROP] = attachedElement != null;
    if (internalsRef.current == null) {
      const internals = {
        id: resolvedId,
        receiver
      };
      internals.instanceRef = createImplementationRef(internals);
      internalsRef.current = internals;
    }
    internalsRef.current.id = resolvedId;
    internalsRef.current.receiver = receiver;
    react.useEffect(() => {
      const node = {
        id: resolvedId
      };
      receiver.implement(node, internalsRef.current?.instanceRef.current);
      return () => {
        receiver.implement(node, null);
      };
    }, [resolvedId, receiver]);
    return /*#__PURE__*/jsxRuntime.jsx(Component, {
      ref: internalsRef.current.instanceRef,
      ...props
    });
  });
  RemoteComponentRenderer.displayName = name ?? `RemoteComponentRenderer(${Component.displayName ?? Component.name ?? 'Component'})`;
  return RemoteComponentRenderer;
}
function createImplementationRef(internals) {
  let current = null;
  return {
    get current() {
      return current;
    },
    set current(implementation) {
      current = implementation;
      internals.receiver.implement(internals, implementation);
    }
  };
}

exports.createRemoteComponentRenderer = createRemoteComponentRenderer;
