export declare const RETAIN_METHOD: unique symbol;
export declare const RELEASE_METHOD: unique symbol;
export declare const RETAINED_BY: unique symbol;
/**
 * A mapped object type that takes an object with methods, and converts it into the
 * an object with the same methods that can be called over a thread.
 */
/**
 * An object that can retain a reference to a `MemoryManageable` object.
 */
export interface MemoryRetainer {
    add(manageable: MemoryManageable): void;
}
/**
 * An object transferred between threads that must have its memory manually managed,
 * in order to release the reference to a corresponding object on the original thread.
 */
export interface MemoryManageable {
    readonly [RETAINED_BY]: Set<MemoryRetainer>;
    [RETAIN_METHOD](): void;
    [RELEASE_METHOD](): void;
}
/**
 * A simple representation of a called function. This object allows this library to
 * release references to functions immediately when the function call that transferred
 * them into this thread is completed.
 */
export declare class StackFrame {
    private readonly memoryManaged;
    add(memoryManageable: MemoryManageable): void;
    release(): void;
}
/**
 * Indicates that a value is being manually memory-managed across threads by this library.
 */
export declare function isMemoryManageable(value: unknown): value is MemoryManageable;
/**
 * Marks a value as being used so it will not be automatically released. Calling `retain` will,
 * by default, deeply retain the value — that is, it will traverse into nested array elements
 * and object properties, and retain every `retain`-able thing it finds.
 *
 * You will typically use this alongside also storing that value in a variable that lives outside
 * the context of the function where that value was received.
 *
 * @example
 * import {retain} from '@quilted/threads';
 *
 * const allUsers = new Set<User>();
 *
 * async function sayHello(user: User) {
 *   allUsers.add(user);
 *   retain(user);
 *   return `Hey, ${await user.fullName()}!`;
 * }
 */
export declare function retain(value: any, { deep }?: {
    deep?: boolean | undefined;
}): boolean;
/**
 * Once you are no longer using the a `retain`-ed value, you can use this function to mark it as
 * being unused. Like `retain()`, this function will apply to all nested array elements and object
 * properties.
 *
 * @example
 * import {retain} from '@quilted/threads';
 *
 * const allUsers = new Set<User>();
 *
 * function removeUser(user: User) {
 *   allUsers.delete(user);
 *   release(user);
 * }
 */
export declare function release(value: any, { deep }?: {
    deep?: boolean | undefined;
}): boolean;
export declare function isBasicObject(value: unknown): boolean;
//# sourceMappingURL=memory.d.ts.map