import { NestedAbortController } from '@quilted/events';
import { ThreadClosedError } from './errors.mjs';
import { nanoid } from './nanoid.mjs';
import { MESSAGE_FUNCTION_RELEASE, MESSAGE_FUNCTION_RESULT, MESSAGE_CALL_RESULT, MESSAGE_FUNCTION_CALL, MESSAGE_CALL } from './constants.mjs';
import { ThreadFunctionsAutomatic } from './functions/ThreadFunctionsAutomatic.mjs';
import { ThreadSerializationStructuredClone } from './serialization/ThreadSerializationStructuredClone.mjs';

/**
 * An object that can serialize and deserialize values communicated between two threads.
 */

/**
 * An object that can serialize and deserialize values communicated between two threads.
 */

/**
 * Options to customize the creation of a `ThreadSerialization` instance.
 */

/**
 * Options to customize the creation of a `Thread` instance.
 */

/**
 * An object backing a `Thread` that provides the message-passing interface
 * that allows communication to flow between environments. This message-passing
 * interface is based on the [`postMessage` interface](https://developer.mozilla.org/en-US/docs/Web/API/Window/postMessage),
 * which is easily adaptable to many JavaScript objects and environments.
 */

/**
 * A map of messages that can be sent between threads.
 */

/**
 * The possible data payloads that can be transferred between threads.
 */

/**
 * A helper type that extracts all callable methods that can be safely proxied
 * between threads; that is, ones which return a promise or async iterator.
 */

/**
 * A thread represents a target JavaScript environment that exposes a set
 * of callable, asynchronous methods. The thread takes care of automatically
 * encoding and decoding its arguments and return values, so you can interact
 * with it as if its methods were implemented in the same environment as your
 * own code.
 */
class Thread {
  /**
   * An object that exposes the methods that can be called on the paired thread.
   * This object will automatically encode and decode arguments and return values
   * as necessary.
   */

  /**
   * An object that exposes the methods that can be called on this thread by the
   * paired thread. To set these methods, pass the `exports` option when creating
   * a new `Thread`.
   */

  /**
   * An object that provides the message-passing interface that allows communication
   * to flow between environments.
   */

  /**
   * An object that manages how functions are proxied between threads.
   */

  /**
   * An object that manages how values are serialized and deserialized between threads.
   */

  /**
   * An `AbortSignal` that indicates whether the communication channel is still open.
   */
  get signal() {
    return this.#abort.signal;
  }

  /**
   * A boolean indicating whether the communication channel is still open.
   */
  get closed() {
    return this.#abort.signal.aborted;
  }
  #abort;
  #idsToResolver = (() => new Map())();
  constructor(messages, {
    imports,
    exports,
    functions = new ThreadFunctionsAutomatic(),
    serialization = new ThreadSerializationStructuredClone(),
    signal
  } = {}) {
    this.messages = messages;
    this.#abort = signal ? new NestedAbortController(signal) : new AbortController();
    this.exports = exports ?? {};
    this.imports = createThreadImports(this.#handlerForCall.bind(this), imports);
    this.functions = functions;
    this.serialization = serialization;
    this.functions.start?.(this);
    this.serialization.start?.(this);
    this.signal.addEventListener('abort', () => {
      for (const id of this.#idsToResolver.keys()) {
        this.#resolveCall(id, undefined, new ThreadClosedError());
      }
      this.#idsToResolver.clear();
    }, {
      once: true
    });
    messages.listen(async rawData => {
      const isThreadMessageData = Array.isArray(rawData) && typeof rawData[0] === 'number';
      if (!isThreadMessageData) {
        return;
      }
      const data = rawData;
      switch (data[0]) {
        case MESSAGE_CALL:
          {
            const [, id, property, args] = data;
            const func = this.exports[property] ?? (() => {
              throw new Error(`No '${property}' method is exported from this thread`);
            });
            await this.#callLocal(func, args, (value, error, transferable) => {
              this.messages.send([MESSAGE_CALL_RESULT, id, value, error], transferable);
            });
            break;
          }
        case MESSAGE_FUNCTION_CALL:
          {
            const [, callID, funcID, args] = data;
            const func = this.functions.get(funcID, this) ?? missingThreadFunction;
            await this.#callLocal(func, args, (value, error, transferable) => {
              this.messages.send([MESSAGE_FUNCTION_RESULT, callID, value, error], transferable);
            });
            break;
          }
        case MESSAGE_CALL_RESULT:
        case MESSAGE_FUNCTION_RESULT:
          {
            this.#resolveCall(...data.slice(1));
            break;
          }
        case MESSAGE_FUNCTION_RELEASE:
          {
            const id = data[1];
            this.functions.release(id, this);
            break;
          }
      }
    }, {
      signal: this.signal
    });
  }

  /**
   * Closes the communication channel between the two threads. This will prevent
   * any further communication between the threads, and will clean up any memory
   * associated with in-progress communication. It will also reject any inflight
   * function calls between threads with a `ThreadClosedError`.
   */
  close() {
    this.#abort.abort();
  }

  /**
   * Requests that the thread provide the context needed to make a function
   * call between threads. You provide this method a function to call and the
   * unserialized arguments you wish to call it with, and the thread will call
   * the function you provided with a serialized call ID, the serialized arguments,
   * and any transferable objects that need to be passed between threads.
   */
  call(func, args) {
    if (this.closed) {
      return Promise.reject(new ThreadClosedError());
    }
    const transferable = [];
    const serialized = this.serialization.serialize(args, this, transferable);
    const id = nanoid();
    const done = this.#waitForResult(id);
    func(id, serialized, transferable);
    return done;
  }
  async #callLocal(func, args, withResult) {
    try {
      const result = this.functions.call ? await this.functions.call(func, args, this) : await func(...this.serialization.deserialize(args, this));
      const transferable = [];
      const serialized = this.serialization.serialize(result, this, transferable);
      withResult(serialized, undefined, transferable);
    } catch (error) {
      withResult(undefined, this.serialization.serialize(error, this));
    }
  }
  #handlerForCall(property) {
    return (...args) => {
      try {
        if (typeof property !== 'string' && typeof property !== 'number') {
          throw new Error(`Can’t call a symbol method on a thread: ${property.toString()}`);
        }
        return this.call((id, serializedArgs, transferable) => {
          this.messages.send([MESSAGE_CALL, id, property, serializedArgs], transferable);
        }, args);
      } catch (error) {
        return Promise.reject(error);
      }
    };
  }
  #resolveCall(...args) {
    const callID = args[0];
    const resolver = this.#idsToResolver.get(callID);
    if (resolver) {
      resolver(...args);
      this.#idsToResolver.delete(callID);
    }
  }
  #waitForResult(id) {
    const promise = new Promise((resolve, reject) => {
      this.#idsToResolver.set(id, (_, value, error) => {
        if (error == null) {
          resolve(this.serialization.deserialize(value, this));
        } else {
          reject(this.serialization.deserialize(error, this));
        }
      });
    });
    Object.defineProperty(promise, Symbol.asyncIterator, {
      async *value() {
        const result = await promise;
        Object.defineProperty(result, Symbol.asyncIterator, {
          value: () => result
        });
        yield* result;
      }
    });
    return promise;
  }
}
function createThreadImports(handlerForImport, imported) {
  let call;
  if (imported == null) {
    if (typeof Proxy !== 'function') {
      throw new Error(`You must pass an array of callable methods in environments without Proxies.`);
    }
    const cache = new Map();
    call = new Proxy({}, {
      get(_target, property) {
        if (cache.has(property)) {
          return cache.get(property);
        }
        const handler = handlerForImport(property);
        cache.set(property, handler);
        return handler;
      }
    });
  } else {
    call = {};
    for (const method of imported) {
      Object.defineProperty(call, method, {
        value: handlerForImport(method),
        writable: false,
        configurable: true,
        enumerable: true
      });
    }
  }
  return call;
}
function missingThreadFunction() {
  throw new Error(`You attempted to call a function that is not stored. It may have already been released.`);
}

export { Thread };
