'use strict';

Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const constants = require('./constants.cjs');
const ChildNode = require('./ChildNode.cjs');
const NodeList = require('./NodeList.cjs');
const selectors = require('./selectors.cjs');
const shared = require('./shared.cjs');

class ParentNode extends ChildNode.ChildNode {
  childNodes = (() => new NodeList.NodeList())();
  children = (() => new NodeList.NodeList())();
  appendChild(child) {
    this.insertInto(child, null);
  }
  insertBefore(child, ref) {
    this.insertInto(child, ref || null);
  }
  append(...nodes) {
    for (const child of nodes) {
      if (child == null) continue;
      this.appendChild(ChildNode.toNode(this, child));
    }
  }
  prepend(...nodes) {
    const before = this.firstChild;
    for (const child of nodes) {
      if (child == null) continue;
      this.insertBefore(ChildNode.toNode(this, child), before);
    }
  }
  replaceChildren(...nodes) {
    let child;
    while (child = this.firstChild) {
      this.removeChild(child);
    }
    this.append(...nodes);
  }
  removeChild(child) {
    if (child.parentNode !== this) throw Error(`not a child of this node`);
    const prev = child[constants.PREV];
    const next = child[constants.NEXT];
    if (prev) prev[constants.NEXT] = next;else this[constants.CHILD] = next;
    if (next) next[constants.PREV] = prev;
    const childNodes = this.childNodes;
    const childNodesIndex = childNodes.indexOf(child);
    childNodes.splice(childNodesIndex, 1);
    if (child.nodeType === 1) {
      const children = this.children;
      children.splice(children.indexOf(child), 1);
    }
    child[constants.PARENT] = null;
    child[constants.NEXT] = null;
    child[constants.PREV] = null;
    if (this[constants.IS_CONNECTED]) {
      for (const node of shared.selfAndDescendants(child)) {
        node[constants.IS_CONNECTED] = false;
        node.disconnectedCallback?.();
      }
    }
    if (this.nodeType === constants.NodeType.ELEMENT_NODE) {
      this[constants.HOOKS].removeChild?.(this, child, childNodesIndex);
    }
  }
  replaceChild(newChild, oldChild) {
    if (oldChild.parentNode !== this) {
      throw Error('reference node is not a child of this parent');
    }
    const next = oldChild[constants.NEXT];
    this.removeChild(oldChild);
    this.insertInto(newChild, next);
  }
  querySelectorAll(selector) {
    return selectors.querySelectorAll(this, selector);
  }
  querySelector(selector) {
    return selectors.querySelector(this, selector);
  }
  insertInto(child, before) {
    // append the children of a DocumentFragment:
    if (child.nodeType === constants.NodeType.DOCUMENT_FRAGMENT_NODE) {
      let node = child[constants.CHILD];
      while (node) {
        const next = node[constants.NEXT];
        this.insertInto(node, before);
        node = next;
      }
      return;
    }
    if (child.parentNode !== null) {
      child.parentNode.removeChild(child);
    }
    if (before) {
      if (before.parentNode !== this) {
        throw Error('reference node is not a child of this parent');
      }
      child[constants.NEXT] = before;
      child[constants.PREV] = before[constants.PREV];
      if (before[constants.PREV] === null) this[constants.CHILD] = child;
      before[constants.PREV] = child;
    } else {
      child[constants.NEXT] = null;
      let last = this[constants.CHILD];
      if (last) {
        let next;
        while (next = last[constants.NEXT]) last = next;
        last[constants.NEXT] = child;
        child[constants.PREV] = last;
      } else {
        this[constants.CHILD] = child;
        child[constants.PREV] = null;
      }
    }
    const ownerDocument = this[constants.OWNER_DOCUMENT];
    const isElement = child.nodeType === constants.NodeType.ELEMENT_NODE;
    child[constants.PARENT] = this;
    child[constants.OWNER_DOCUMENT] = ownerDocument;
    const childNodes = this.childNodes;
    let insertIndex;
    if (before) {
      insertIndex = childNodes.indexOf(before);
      childNodes.splice(insertIndex, 0, child);
      if (isElement) {
        const children = this.children;
        let ref = before;
        while (ref && ref.nodeType !== 1) ref = ref[constants.NEXT];
        if (ref) {
          children.splice(children.indexOf(ref), 0, child);
        } else {
          children.push(child);
        }
      }
    } else {
      insertIndex = childNodes.length;
      childNodes.push(child);
      if (isElement) this.children.push(child);
    }
    if (this[constants.IS_CONNECTED]) {
      for (const node of shared.selfAndDescendants(child)) {
        node[constants.IS_CONNECTED] = true;
        node.connectedCallback?.();
      }
    }
    if (this.nodeType === constants.NodeType.ELEMENT_NODE) {
      this[constants.HOOKS].insertChild?.(this, child, insertIndex);
    }
  }
}

exports.ParentNode = ParentNode;
