'use strict';

Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const constants = require('./constants.cjs');
const shared = require('./shared.cjs');

var Combinator = /*#__PURE__*/function (Combinator) {
  Combinator[Combinator["Descendant"] = 0] = "Descendant";
  Combinator[Combinator["Child"] = 1] = "Child";
  Combinator[Combinator["Sibling"] = 2] = "Sibling";
  Combinator[Combinator["Adjacent"] = 3] = "Adjacent";
  Combinator[Combinator["Inner"] = 4] = "Inner";
  return Combinator;
}(Combinator || {});
var MatcherType = /*#__PURE__*/function (MatcherType) {
  MatcherType[MatcherType["Unknown"] = 0] = "Unknown";
  MatcherType[MatcherType["Element"] = 1] = "Element";
  MatcherType[MatcherType["Id"] = 2] = "Id";
  MatcherType[MatcherType["Class"] = 3] = "Class";
  MatcherType[MatcherType["Attribute"] = 4] = "Attribute";
  MatcherType[MatcherType["Pseudo"] = 5] = "Pseudo";
  MatcherType[MatcherType["Function"] = 6] = "Function";
  return MatcherType;
}(MatcherType || {});
const ELEMENT_SELECTOR_TEST = /[a-z]/;
function querySelector(within, selector) {
  const parts = parseSelector(selector);
  let result = null;
  const child = within[constants.CHILD];
  if (child && parts[0].matchers.length) {
    walkNodesForSelector(child, parts, node => {
      result = node;
      return false;
    });
  }
  return result;
}
function querySelectorAll(within, selector) {
  const parts = parseSelector(selector);
  const results = [];
  const child = within[constants.CHILD];
  if (child && parts[0].matchers.length) {
    walkNodesForSelector(child, parts, node => {
      results.push(node);
    });
  }
  return results;
}
function parseSelector(selector) {
  let part = {
    combinator: Combinator.Inner,
    matchers: []
  };
  const parts = [part];
  const tokenizer = /\s*?([>\s+~]?)\s*?(?:(?:\[\s*([^\]=]+)(?:=(['"])(.*?)\3)?\s*\])|([#.]?)([^\s#.[>:+~]+)|:(\w+)(?:\((.*?)\))?)/gi;
  let token;
  while (token = tokenizer.exec(selector)) {
    // [1]: ancestor/parent/sibling/adjacent
    // [2]: attribute name
    // [4]: attribute value
    // [5]: id/class sigil
    // [6]: id/class name
    // [7]: :pseudo/:function() name
    // [8]: :function(argument) value
    if (token[1]) {
      // Update the combinator on the (now parent) Part:
      if (token[1] === '>') part.combinator = Combinator.Child;else if (token[1] === '+') part.combinator = Combinator.Adjacent;else if (token[1] === '~') part.combinator = Combinator.Sibling;else part.combinator = Combinator.Descendant;
      // Add a new Part for the next selector parts:
      part = {
        combinator: Combinator.Inner,
        matchers: []
      };
      parts.push(part);
    }
    let type = MatcherType.Unknown;
    if (token[2]) {
      type = MatcherType.Attribute;
    } else if (token[5]) {
      type = token[5] === '#' ? MatcherType.Id : MatcherType.Class;
    } else if (token[7]) {
      type = token[8] == null ? MatcherType.Pseudo : MatcherType.Function;
    } else if (token[6]) {
      if (token[6] === '*') {
        type = MatcherType.Unknown; // Universal selector matches all
      } else if (ELEMENT_SELECTOR_TEST.test(token[6])) {
        type = MatcherType.Element;
      }
    }
    part.matchers.push({
      type,
      name: token[2] || token[6] || token[7],
      value: token[4] ?? token[6] ?? token[8]
    });
  }
  return parts;
}
function matchesSelector(element, selector) {
  const parsed = parseSelector(selector);
  let part;
  while (part = parsed.pop()) {
    if (!matchesSelectorPart(element, part)) return false;
  }
  return true;
}
function walkNodesForSelector(node, parts, callback) {
  if (shared.isElementNode(node)) {
    if (matchesSelectorRecursive(node, parts)) {
      if (callback(node) === false) return false;
    }
    const child = node[constants.CHILD];
    if (child && walkNodesForSelector(child, parts, callback) === false) {
      return false;
    }
  }
  const next = node[constants.NEXT];
  if (next && walkNodesForSelector(next, parts, callback) === false) {
    return false;
  }
  return true;
}
function matchesSelectorRecursive(element, parts) {
  const {
    combinator,
    matchers
  } = parts[parts.length - 1];
  if (combinator === Combinator.Inner) {
    if (!matchesSelectorMatcher(element, matchers)) return false;
    const pp = parts.slice(0, -1);
    return pp.length === 0 || matchesSelectorRecursive(element, pp);
  }
  const link = combinator === Combinator.Child || combinator === Combinator.Descendant ? constants.PARENT : constants.PREV;
  let ref = element[link];
  if (!ref) return false;
  if (combinator === Combinator.Descendant || combinator === Combinator.Sibling) {
    // For descendant/sibling combinators, search through all ancestors/siblings
    while (ref) {
      if (shared.isElementNode(ref) && matchesSelectorMatcher(ref, matchers)) {
        const pp = parts.slice(0, -1);
        if (pp.length === 0) return true;
        if (matchesSelectorRecursive(element, pp)) return true;
      }
      ref = ref[link];
    }
    return false;
  } else {
    // For child/adjacent combinators, check only the immediate parent/sibling
    // For sibling combinators, skip non-element siblings
    if (combinator === Combinator.Adjacent && !shared.isElementNode(ref)) {
      // Skip to next element sibling
      while (ref && !shared.isElementNode(ref)) {
        ref = ref[link];
      }
      if (!ref) return false;
    }
    if (!shared.isElementNode(ref) || !matchesSelectorMatcher(ref, matchers)) {
      return false;
    }
    const pp = parts.slice(0, -1);
    return pp.length === 0 || matchesSelectorRecursive(element, pp);
  }
}
function matchesSelectorPart(element, {
  combinator,
  matchers
}) {
  if (combinator === Combinator.Inner) {
    return matchesSelectorMatcher(element, matchers);
  }
  const link = combinator === Combinator.Child || combinator === Combinator.Descendant ? constants.PARENT : constants.PREV;
  let ref = element[link];
  if (!ref) return false;

  // For sibling combinators, skip non-element siblings
  if (combinator === Combinator.Adjacent && !shared.isElementNode(ref)) {
    while (ref && !shared.isElementNode(ref)) {
      ref = ref[link];
    }
    if (!ref) return false;
  }
  if (!shared.isElementNode(ref) || !matchesSelectorMatcher(ref, matchers)) {
    return false;
  }
  if (combinator === Combinator.Descendant || combinator === Combinator.Sibling) {
    while (ref = ref[link]) {
      if (shared.isElementNode(ref) && matchesSelectorMatcher(ref, matchers)) return true;
    }
  }
  return true;
}
function matchesSelectorMatcher(element, matcher) {
  if (!element) return false;
  if (Array.isArray(matcher)) {
    for (const single of matcher) {
      if (matchesSelectorMatcher(element, single) === false) return false;
    }
    return true;
  }
  const {
    type,
    name,
    value
  } = matcher;
  switch (type) {
    case MatcherType.Unknown:
      return name === '*';
    // Universal selector
    case MatcherType.Element:
      return element.localName === name;
    case MatcherType.Id:
      return element.getAttribute('id') === name;
    case MatcherType.Class:
      const classAttr = element.getAttribute('class');
      if (!classAttr) return false;
      return classAttr.split(/\s+/).includes(name);
    case MatcherType.Attribute:
      return value == null ? element.hasAttribute(name) : element.getAttribute(name) === value;
    case MatcherType.Pseudo:
      switch (name) {
        default:
          throw Error(`Pseudo :${name} not implemented`);
      }
    case MatcherType.Function:
      switch (name) {
        case 'has':
          return matchesSelector(element, value || '');
        case 'not':
          return !matchesSelector(element, value || '');
        default:
          throw Error(`Function :${name}(${value}) not implemented`);
      }
  }
  return false;
}

exports.parseSelector = parseSelector;
exports.querySelector = querySelector;
exports.querySelectorAll = querySelectorAll;
