'use strict';

Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const constants = require('./constants.cjs');

function isCharacterData(node) {
  return constants.DATA in node;
}
function isTextNode(node) {
  return node.nodeType === constants.NodeType.TEXT_NODE;
}
function isCommentNode(node) {
  return node.nodeType === constants.NodeType.COMMENT_NODE;
}
function isElementNode(node) {
  return node.nodeType === constants.NodeType.ELEMENT_NODE;
}
function isDocumentFragmentNode(node) {
  return node.nodeType === constants.NodeType.DOCUMENT_FRAGMENT_NODE;
}
function isParentNode(node) {
  return 'appendChild' in node;
}
function cloneNode(node, deep, document = node.ownerDocument) {
  if (isTextNode(node)) {
    return document.createTextNode(node.data);
  } else if (isCommentNode(node)) {
    return document.createComment(node.data);
  } else if (isElementNode(node)) {
    const cloned = document.createElement(node.localName);
    if (node[constants.ATTRIBUTES]) {
      for (let i = 0; i < node[constants.ATTRIBUTES].length; i++) {
        const attribute = node[constants.ATTRIBUTES].item(i);
        cloned.setAttributeNS(attribute.namespaceURI, attribute.name, attribute.value);
      }
    }
    if (deep) {
      for (const child of node.childNodes) {
        cloned.appendChild(cloneNode(child, true, document));
      }
    }
    return cloned;
  } else if (isDocumentFragmentNode(node)) {
    const fragment = document.createDocumentFragment();
    if (deep) {
      for (const child of node.childNodes) {
        fragment.appendChild(cloneNode(child, true, document));
      }
    }
    return fragment;
  } else {
    const cloned = new node.constructor();
    cloned[constants.OWNER_DOCUMENT] = document;
    return cloned;
  }
}
function descendants(node) {
  const nodes = [];
  const walk = node => {
    nodes.push(node);
    const child = node[constants.CHILD];
    if (child) walk(child);
    const sibling = node[constants.NEXT];
    if (sibling) walk(sibling);
  };
  const child = node[constants.CHILD];
  if (child) walk(child);
  return nodes;
}
function selfAndDescendants(node) {
  const nodes = descendants(node);
  nodes.unshift(node);
  return nodes;
}

exports.cloneNode = cloneNode;
exports.descendants = descendants;
exports.isCharacterData = isCharacterData;
exports.isCommentNode = isCommentNode;
exports.isDocumentFragmentNode = isDocumentFragmentNode;
exports.isElementNode = isElementNode;
exports.isParentNode = isParentNode;
exports.isTextNode = isTextNode;
exports.selfAndDescendants = selfAndDescendants;
