/**
 * A representation of an `AbortSignal` that can be serialized between
 * two threads.
 */
export interface ThreadAbortSignalSerialization {
    /**
     * Whether the signal was already aborted at the time it was
     * sent to the sibling thread.
     */
    readonly aborted: boolean;
    /**
     * A function to connect the signal between the two threads. This
     * function should be called by the sibling thread when the abort
     * state changes (including changes since the thread-safe abort signal
     * was created).
     */
    start?(listener: (aborted: boolean) => void): void;
}
export interface ThreadAbortSignalOptions {
    /**
     * An optional AbortSignal that can cancel synchronizing the
     * (Preact) signal to its paired thread.
     */
    signal?: AbortSignal;
    /**
     * An optional function to call in order to manually retain the memory
     * associated with the `start` function of the serialized signal.
     * You only need to use this when using a strategy for serializing the
     * abort signal that requires manual memory management.
     */
    retain?(value: unknown): void;
    /**
     * An optional function to call in order to manually release the memory
     * associated with the `start` function of the serialized signal.
     * You only need to use this when using a strategy for serializing the
     * abort signal that requires manual memory management.
     */
    release?(value: unknown): void;
}
/**
 * Converts a serialized `AbortSignal` into a “live” one, which you can
 * use to cancel operations in the current environment. When the signal aborts,
 * all memory associated with the signal will be released automatically.
 */
export declare class ThreadAbortSignal implements AbortSignal {
    #private;
    get aborted(): boolean;
    get reason(): any;
    get onabort(): ((this: AbortSignal, ev: Event) => any) | null;
    set onabort(value: ((this: AbortSignal, ev: Event) => any) | null);
    constructor(signal: AbortSignal | ThreadAbortSignalSerialization | undefined, { signal: killswitchSignal, retain, release }?: ThreadAbortSignalOptions);
    addEventListener(...args: Parameters<AbortSignal['addEventListener']>): void;
    removeEventListener(...args: Parameters<AbortSignal['removeEventListener']>): void;
    dispatchEvent(...args: Parameters<AbortSignal['dispatchEvent']>): boolean;
    throwIfAborted(): void;
    /**
     * Converts an `AbortSignal` into a version of that signal that can
     * be transferred to a target `Thread`. The resulting object can be
     * serialized using the RPC utilities provided in this library, and
     * passed to `new ThreadAbortSignal()` to be converted into a “live”
     * `AbortSignal`.
     */
    static serialize(signal: Pick<AbortSignal, 'aborted' | 'addEventListener'>, { retain, release }?: ThreadAbortSignalOptions): ThreadAbortSignalSerialization;
    /**
     * Checks if a value is a serialized ThreadSignal.
     */
    static isSerialized(value: unknown): value is ThreadAbortSignalSerialization;
}
//# sourceMappingURL=ThreadAbortSignal.d.ts.map