## Copyright 2015 Oliver Heimlich
## Copyright 2018 Joel Dahne
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, see <http://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @documentencoding UTF-8
## @defmethod {@@infsup} linspace (@var{BASE}, @var{LIMIT})
## @defmethodx {@@infsup} linspace (@var{BASE}, @var{LIMIT}, @var{N})
##
## Return a row vector of @var{N} linearly spaced members between @var{BASE}
## and @var{LIMIT}.
##
## If @var{BASE} is greater than @var{LIMIT}, members are returned in
## decreasing order.  The default value for @var{N} is 100.
##
## If either @var{BASE} or @var{LIMIT} is not a scalar, they are
## converted to column vectors and the result is a matrix where the
## rows are the independent sequences.
##
## Accuracy: The result is an accurate enclosure.
##
## @example
## @group
## transpose (linspace (infsup (0), infsup (10), 4))
##   @result{} ans ⊂ 4×1 interval vector
##
##                     [0]
##        [3.3333, 3.3334]
##        [6.6666, 6.6667]
##                    [10]
## @end group
## @end example
## @seealso{linspace}
## @end defmethod

## Author: Oliver Heimlich
## Keywords: interval
## Created: 2015-07-19

function base = linspace (base, limit, n)

  if (nargin < 2 || nargin > 3)
    print_usage ();
    return
  endif
  if (not (isa (base, "infsup")))
    base = infsup (base);
  endif
  if (not (isa (limit, "infsup")))
    limit = infsup (limit);
  endif
  if (nargin < 3)
    n = 100;
  endif

  if (not (isvector (base)) || not (isvector (limit)))
    error ("linspace: BASE and LIMIT must be scalars or vectors");
  endif

  ## Convert base and limit to column vectors. In fact mpfr_linspace_d
  ## treats all arrays as row vectors but it is safer to not depend on
  ## this.
  base = reshape (base, [], 1);
  limit = reshape (limit, [], 1);

  l = mpfr_linspace_d (-inf, base.inf, limit.inf, n);
  u = mpfr_linspace_d (+inf, base.sup, limit.sup, n);

  empty = vec (isempty (base) | isempty (limit));
  l(empty, :) = inf;
  u(empty, :) = -inf;

  l(l == 0) = -0;

  base.inf = l;
  base.sup = u;

endfunction

%!assert (isequal (linspace (infsup (0), infsup (10), 9), infsup (linspace (0, 10, 9))));

%!# correct use of signed zeros
%!test
%! x = linspace (infsup (0), infsup (0));
%! assert (all (signbit (inf (x))));
%! assert (all (not (signbit (sup (x)))));

%!error <linspace: BASE and LIMIT must be scalars or vectors>
%!  linspace (infsup (zeros (2)), infsup (ones (2)), 3)
