package runtime

_INTEGER_DIGITS :: "0123456789abcdefghijklmnopqrstuvwxyz"

@(private="file")
_INTEGER_DIGITS_VAR := _INTEGER_DIGITS

when !ODIN_NO_RTTI {
	print_any_single :: #force_no_inline proc "contextless" (arg: any) {
		x := arg
		if x.data == nil {
			print_string("nil")
			return
		}

		if loc, ok := x.(Source_Code_Location); ok {
			print_caller_location(loc)
			return
		}
		x.id = typeid_base(x.id)
		switch v in x {
		case typeid:     print_typeid(v)
		case ^Type_Info: print_type(v)

		case string:  print_string(v)
		case cstring: print_string(string(v))
		case []byte:  print_string(string(v))

		case rune:  print_rune(v)

		case u8:    print_u64(u64(v))
		case u16:   print_u64(u64(v))
		case u16le: print_u64(u64(v))
		case u16be: print_u64(u64(v))
		case u32:   print_u64(u64(v))
		case u32le: print_u64(u64(v))
		case u32be: print_u64(u64(v))
		case u64:   print_u64(u64(v))
		case u64le: print_u64(u64(v))
		case u64be: print_u64(u64(v))

		case i8:    print_i64(i64(v))
		case i16:   print_i64(i64(v))
		case i16le: print_i64(i64(v))
		case i16be: print_i64(i64(v))
		case i32:   print_i64(i64(v))
		case i32le: print_i64(i64(v))
		case i32be: print_i64(i64(v))
		case i64:   print_i64(i64(v))
		case i64le: print_i64(i64(v))
		case i64be: print_i64(i64(v))

		case int:     print_int(v)
		case uint:    print_uint(v)
		case uintptr: print_uintptr(v)
		case rawptr:  print_uintptr(uintptr(v))

		case bool: print_string("true" if v else "false")
		case b8:   print_string("true" if v else "false")
		case b16:  print_string("true" if v else "false")
		case b32:  print_string("true" if v else "false")
		case b64:  print_string("true" if v else "false")

		case:
			ti := type_info_of(x.id)
			#partial switch v in ti.variant {
			case Type_Info_Pointer, Type_Info_Multi_Pointer:
				print_uintptr((^uintptr)(x.data)^)
				return
			}

			print_string("<invalid-value>")
		}
	}
	println_any :: #force_no_inline proc "contextless" (args: ..any) {
		context = default_context()
		loop: for arg, i in args {
			assert(arg.id != nil)
			if i != 0 {
				print_string(" ")
			}
			print_any_single(arg)
		}
		print_string("\n")
	}
}


encode_rune :: proc "contextless" (c: rune) -> ([4]u8, int) {
	r := c

	buf: [4]u8
	i := u32(r)
	mask :: u8(0x3f)
	if i <= 1<<7-1 {
		buf[0] = u8(r)
		return buf, 1
	}
	if i <= 1<<11-1 {
		buf[0] = 0xc0 | u8(r>>6)
		buf[1] = 0x80 | u8(r) & mask
		return buf, 2
	}

	// Invalid or Surrogate range
	if i > 0x0010ffff ||
	   (0xd800 <= i && i <= 0xdfff) {
		r = 0xfffd
	}

	if i <= 1<<16-1 {
		buf[0] = 0xe0 | u8(r>>12)
		buf[1] = 0x80 | u8(r>>6) & mask
		buf[2] = 0x80 | u8(r)    & mask
		return buf, 3
	}

	buf[0] = 0xf0 | u8(r>>18)
	buf[1] = 0x80 | u8(r>>12) & mask
	buf[2] = 0x80 | u8(r>>6)  & mask
	buf[3] = 0x80 | u8(r)     & mask
	return buf, 4
}

print_string :: #force_no_inline proc "contextless" (str: string) -> (n: int) {
	n, _ = stderr_write(transmute([]byte)str)
	return
}

print_strings :: #force_no_inline proc "contextless" (args: ..string) -> (n: int) {
	for str in args {
		m, err := stderr_write(transmute([]byte)str)
		n += m
		if err != 0 {
			break
		}
	}
	return
}

print_byte :: #force_no_inline proc "contextless" (b: byte) -> (n: int) {
	n, _ = stderr_write([]byte{b})
	return
}

print_encoded_rune :: #force_no_inline proc "contextless" (r: rune) {
	print_byte('\'')

	switch r {
	case '\a': print_string("\\a")
	case '\b': print_string("\\b")
	case '\e': print_string("\\e")
	case '\f': print_string("\\f")
	case '\n': print_string("\\n")
	case '\r': print_string("\\r")
	case '\t': print_string("\\t")
	case '\v': print_string("\\v")
	case:
		if r <= 0 {
			print_string("\\x00")
		} else if r < 32 {
			n0, n1 := u8(r) >> 4, u8(r) & 0xf
			print_string("\\x")
			print_byte(_INTEGER_DIGITS_VAR[n0])
			print_byte(_INTEGER_DIGITS_VAR[n1])
		} else {
			print_rune(r)
		}
	}
	print_byte('\'')
}

print_rune :: #force_no_inline proc "contextless" (r: rune) -> int #no_bounds_check {
	RUNE_SELF :: 0x80

	if r < RUNE_SELF {
		return print_byte(byte(r))
	}

	b, n := encode_rune(r)
	m, _ := stderr_write(b[:n])
	return m
}


print_u64 :: #force_no_inline proc "contextless" (x: u64) #no_bounds_check {
	b :: u64(10)
	u := x

	a: [129]byte
	i := len(a)
	for u >= b {
		i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]
		u /= b
	}
	i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]

	stderr_write(a[i:])
}


print_i64 :: #force_no_inline proc "contextless" (x: i64) #no_bounds_check {
	b :: u64(10)
	u := u64(abs(x))
	neg := x < 0

	a: [129]byte
	i := len(a)
	for u >= b {
		i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]
		u /= b
	}
	i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]
	if neg {
		i -= 1; a[i] = '-'
	}

	stderr_write(a[i:])
}

print_uint    :: proc "contextless" (x: uint)    { print_u64(u64(x)) }
print_uintptr :: proc "contextless" (x: uintptr) { print_u64(u64(x)) }
print_int     :: proc "contextless" (x: int)     { print_i64(i64(x)) }

print_caller_location :: #force_no_inline proc "contextless" (loc: Source_Code_Location) {
	print_string(loc.file_path)
	when ODIN_ERROR_POS_STYLE == .Default {
		print_byte('(')
		print_u64(u64(loc.line))
		if loc.column != 0 {
			print_byte(':')
			print_u64(u64(loc.column))
		}
		print_byte(')')
	} else when ODIN_ERROR_POS_STYLE == .Unix {
		print_byte(':')
		print_u64(u64(loc.line))
		if loc.column != 0 {
			print_byte(':')
			print_u64(u64(loc.column))
		}
		print_byte(':')
	} else {
		#panic("unhandled ODIN_ERROR_POS_STYLE")
	}
}
print_typeid :: #force_no_inline proc "contextless" (id: typeid) {
	when ODIN_NO_RTTI {
		if id == nil {
			print_string("nil")
		} else {
			print_string("<unknown type>")
		}
	} else {
		if id == nil {
			print_string("nil")
		} else {
			ti := type_info_of(id)
			print_type(ti)
		}
	}
}

@(optimization_mode="favor_size")
print_type :: #force_no_inline proc "contextless" (ti: ^Type_Info) {
	if ti == nil {
		print_string("nil")
		return
	}

	switch info in ti.variant {
	case Type_Info_Named:
		print_string(info.name)
	case Type_Info_Integer:
		switch ti.id {
		case int:     print_string("int")
		case uint:    print_string("uint")
		case uintptr: print_string("uintptr")
		case:
			print_byte('i' if info.signed else 'u')
			print_u64(u64(8*ti.size))
		}
		switch info.endianness {
		case .Platform: // nothing
		case .Little:   print_string("le")
		case .Big:      print_string("be")
		}

	case Type_Info_Rune:
		print_string("rune")
	case Type_Info_Float:
		print_byte('f')
		print_u64(u64(8*ti.size))
		switch info.endianness {
		case .Platform: // nothing
		case .Little:   print_string("le")
		case .Big:      print_string("be")
		}
	case Type_Info_Complex:
		print_string("complex")
		print_u64(u64(8*ti.size))
	case Type_Info_Quaternion:
		print_string("quaternion")
		print_u64(u64(8*ti.size))
	case Type_Info_String:
		if info.is_cstring {
			print_byte('c')
		}
		print_string("string")
		switch info.encoding {
		case .UTF_8:  /**/
		case .UTF_16: print_string("16")
		}
	case Type_Info_Boolean:
		switch ti.id {
		case bool: print_string("bool")
		case:
			print_byte('b')
			print_u64(u64(8*ti.size))
		}
	case Type_Info_Any:
		print_string("any")
	case Type_Info_Type_Id:
		print_string("typeid")

	case Type_Info_Pointer:
		if info.elem == nil {
			print_string("rawptr")
		} else {
			print_string("^")
			print_type(info.elem)
		}
	case Type_Info_Multi_Pointer:
		print_string("[^]")
		print_type(info.elem)
	case Type_Info_Soa_Pointer:
		print_string("#soa ^")
		print_type(info.elem)
	case Type_Info_Procedure:
		print_string("proc")
		if info.params == nil {
			print_string("()")
		} else {
			t := info.params.variant.(Type_Info_Parameters)
			print_byte('(')
			for t, i in t.types {
				if i > 0 { print_string(", ") }
				print_type(t)
			}
			print_string(")")
		}
		if info.results != nil {
			print_string(" -> ")
			print_type(info.results)
		}
	case Type_Info_Parameters:
		count := len(info.names)
		if count != 1 { print_byte('(') }
		for name, i in info.names {
			if i > 0 { print_string(", ") }

			t := info.types[i]

			if len(name) > 0 {
				print_string(name)
				print_string(": ")
			}
			print_type(t)
		}
		if count != 1 { print_string(")") }

	case Type_Info_Array:
		print_byte('[')
		print_u64(u64(info.count))
		print_byte(']')
		print_type(info.elem)

	case Type_Info_Enumerated_Array:
		if info.is_sparse {
			print_string("#sparse")
		}
		print_byte('[')
		print_type(info.index)
		print_byte(']')
		print_type(info.elem)


	case Type_Info_Dynamic_Array:
		print_string("[dynamic]")
		print_type(info.elem)
	case Type_Info_Slice:
		print_string("[]")
		print_type(info.elem)

	case Type_Info_Map:
		print_string("map[")
		print_type(info.key)
		print_byte(']')
		print_type(info.value)

	case Type_Info_Struct:
		switch info.soa_kind {
		case .None: // Ignore
		case .Fixed:
			print_string("#soa[")
			print_u64(u64(info.soa_len))
			print_byte(']')
			print_type(info.soa_base_type)
			return
		case .Slice:
			print_string("#soa[]")
			print_type(info.soa_base_type)
			return
		case .Dynamic:
			print_string("#soa[dynamic]")
			print_type(info.soa_base_type)
			return
		}

		print_string("struct ")
		if .packed      in info.flags { print_string("#packed ") }
		if .raw_union   in info.flags { print_string("#raw_union ") }
		if .all_or_none in info.flags { print_string("#all_or_none ") }
		if .simple      in info.flags { print_string("#simple ") }
		if .align in info.flags {
			print_string("#align(")
			print_u64(u64(ti.align))
			print_string(") ")
		}
		print_byte('{')
		for name, i in info.names[:info.field_count] {
			if i > 0 { print_string(", ") }
			print_string(name)
			print_string(": ")
			print_type(info.types[i])
		}
		print_byte('}')

	case Type_Info_Union:
		print_string("union ")
		if info.custom_align {
			print_string("#align(")
			print_u64(u64(ti.align))
			print_string(") ")
		}
		if info.no_nil {
			print_string("#no_nil ")
		}
		print_byte('{')
		for variant, i in info.variants {
			if i > 0 { print_string(", ") }
			print_type(variant)
		}
		print_string("}")

	case Type_Info_Enum:
		print_string("enum ")
		print_type(info.base)
		print_string(" {")
		for name, i in info.names {
			if i > 0 { print_string(", ") }
			print_string(name)
		}
		print_string("}")

	case Type_Info_Bit_Set:
		print_string("bit_set[")

		#partial switch elem in type_info_base(info.elem).variant {
		case Type_Info_Enum:
			print_type(info.elem)
		case Type_Info_Rune:
			print_encoded_rune(rune(info.lower))
			print_string("..")
			print_encoded_rune(rune(info.upper))
		case:
			print_i64(info.lower)
			print_string("..")
			print_i64(info.upper)
		}
		if info.underlying != nil {
			print_string("; ")
			print_type(info.underlying)
		}
		print_byte(']')

	case Type_Info_Bit_Field:
		print_string("bit_field ")
		print_type(info.backing_type)
		print_string(" {")
		for name, i in info.names[:info.field_count] {
			if i > 0 { print_string(", ") }
			print_string(name)
			print_string(": ")
			print_type(info.types[i])
			print_string(" | ")
			print_u64(u64(info.bit_sizes[i]))
		}
		print_byte('}')


	case Type_Info_Simd_Vector:
		print_string("#simd[")
		print_u64(u64(info.count))
		print_byte(']')
		print_type(info.elem)
		
	case Type_Info_Matrix:
		if info.layout == .Row_Major {
			print_string("#row_major ")
		}
		print_string("matrix[")
		print_u64(u64(info.row_count))
		print_string(", ")
		print_u64(u64(info.column_count))
		print_string("]")
		print_type(info.elem)
	}
}


@(require_results)
write_string :: proc "contextless" (i: ^int, dst: []byte, src: string) -> bool {
	if i^ < len(dst) {
		i^ += copy(dst[i^:], src)
		return true
	}
	return false
}


@(require_results)
write_byte :: proc "contextless" (i: ^int, dst: []byte, src: byte) -> bool {
	if i^ < len(dst) {
		dst[i^] = src
		i^ += 1
		return true
	}
	return false
}


@(require_results)
write_u64 :: proc "contextless" (j: ^int, dst: []byte, x: u64) -> bool {
	if j^ < len(dst) {
		b :: u64(10)
		u := x

		a: [129]byte
		i := len(a)
		for u >= b {
			i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]
			u /= b
		}
		i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]

		return write_string(j, dst, string(a[i:]))
	}
	return false
}

@(require_results)
write_i64 :: proc "contextless" (j: ^int, dst: []byte, x: i64) -> bool {
	if j^ < len(dst) {
		b :: u64(10)
		u := u64(abs(x))
		neg := x < 0

		a: [129]byte
		i := len(a)
		for u >= b {
			i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]
			u /= b
		}
		i -= 1; a[i] = _INTEGER_DIGITS_VAR[u % b]
		if neg {
			i -= 1; a[i] = '-'
		}

		return write_string(j, dst, string(a[i:]))
	}
	return false
}


@(require_results)
write_caller_location :: #force_no_inline proc "contextless" (i: ^int, buf: []byte, loc: Source_Code_Location) -> bool {
	write_string(i, buf, loc.file_path) or_return

	when ODIN_ERROR_POS_STYLE == .Default {
		write_byte(i, buf, '(')           or_return
		write_u64 (i, buf, u64(loc.line)) or_return
		if loc.column != 0 {
			write_byte(i, buf, ':')             or_return
			write_u64 (i, buf, u64(loc.column)) or_return
		}
		write_byte(i, buf, ')') or_return
		return true
	} else when ODIN_ERROR_POS_STYLE == .Unix {
		write_byte(i, buf, ':')           or_return
		write_u64 (i, buf, u64(loc.line)) or_return
		if loc.column != 0 {
			write_byte(i, buf, ':')             or_return
			write_u64 (i, buf, u64(loc.column)) or_return
		}
		write_byte(i, buf, ':') or_return
		return true
	} else {
		#panic("unhandled ODIN_ERROR_POS_STYLE")
	}
}

@(require_results)
write_typeid :: #force_no_inline proc "contextless" (i: ^int, buf: []byte, id: typeid) -> bool {
	when ODIN_NO_RTTI {
		if id == nil {
			write_string(i, buf, "nil") or_return
		} else {
			write_string(i, buf, "<unknown type>") or_return
		}
	} else {
		if id == nil {
			write_string(i, buf, "nil") or_return
		} else {
			ti := type_info_of(id)
			write_write_type(i, buf, ti) or_return
		}
	}
	return true
}


@(require_results)
write_rune :: #force_no_inline proc "contextless" (i: ^int, buf: []byte, r: rune) -> (written: int, ok: bool) #no_bounds_check {
	RUNE_SELF :: 0x80

	if r < RUNE_SELF {
		write_byte(i, buf,byte(r)) or_return
		return 1, true
	}

	b, n := encode_rune(r)
	prev := i^
	write_string(i, buf, string(b[:n])) or_return
	return i^ - prev, true
}

@(require_results)
write_encoded_rune :: #force_no_inline proc "contextless" (i: ^int, buf: []byte, r: rune) -> bool {
	write_byte(i, buf, '\'') or_return

	switch r {
	case '\a': write_string(i, buf, "\\a") or_return
	case '\b': write_string(i, buf, "\\b") or_return
	case '\e': write_string(i, buf, "\\e") or_return
	case '\f': write_string(i, buf, "\\f") or_return
	case '\n': write_string(i, buf, "\\n") or_return
	case '\r': write_string(i, buf, "\\r") or_return
	case '\t': write_string(i, buf, "\\t") or_return
	case '\v': write_string(i, buf, "\\v") or_return
	case:
		if r <= 0 {
			write_string(i, buf, "\\x00") or_return
		} else if r < 32 {
			n0, n1 := u8(r) >> 4, u8(r) & 0xf
			write_string(i, buf, "\\x")                   or_return
			write_byte  (i, buf, _INTEGER_DIGITS_VAR[n0]) or_return
			write_byte  (i, buf, _INTEGER_DIGITS_VAR[n1]) or_return
		} else {
			_ = write_rune(i, buf, r) or_return
		}
	}

	write_byte(i, buf, '\'') or_return
	return true
}

@(optimization_mode="favor_size")
write_write_type :: #force_no_inline proc "contextless" (i: ^int, buf: []byte, ti: ^Type_Info) -> bool {
	if ti == nil {
		write_string(i, buf, "nil") or_return
		return true
	}

	switch info in ti.variant {
	case Type_Info_Named:
		write_string(i, buf, info.name) or_return
	case Type_Info_Integer:
		switch ti.id {
		case int:     write_string(i, buf, "int")     or_return
		case uint:    write_string(i, buf, "uint")    or_return
		case uintptr: write_string(i, buf, "uintptr") or_return
		case:
			write_byte(i, buf, 'i' if info.signed else 'u') or_return
			write_u64 (i, buf, u64(8*ti.size))              or_return
		}
		switch info.endianness {
		case .Platform: // nothing
		case .Little:   write_string(i, buf, "le") or_return
		case .Big:      write_string(i, buf, "be") or_return
		}

	case Type_Info_Rune:
		write_string(i, buf, "rune") or_return
	case Type_Info_Float:
		write_byte(i, buf, 'f') or_return
		write_u64(i, buf, u64(8*ti.size)) or_return
		switch info.endianness {
		case .Platform: // nothing
		case .Little:   write_string(i, buf, "le") or_return
		case .Big:      write_string(i, buf, "be") or_return
		}

	case Type_Info_Complex:
		write_string(i, buf, "complex")      or_return
		write_u64   (i, buf, u64(8*ti.size)) or_return
	case Type_Info_Quaternion:
		write_string(i, buf, "quaternion")   or_return
		write_u64   (i, buf, u64(8*ti.size)) or_return
	case Type_Info_String:
		if info.is_cstring {
			write_byte(i, buf, 'c') or_return
		}
		write_string(i, buf, "string") or_return
		switch info.encoding {
		case .UTF_8:  /**/
		case .UTF_16: write_string(i, buf, "16") or_return
		}
	case Type_Info_Boolean:
		switch ti.id {
		case bool: write_string(i, buf, "bool") or_return
		case:
			write_byte(i, buf, 'b')            or_return
			write_u64 (i, buf, u64(8*ti.size)) or_return
		}
	case Type_Info_Any:
		write_string(i, buf, "any") or_return
	case Type_Info_Type_Id:
		write_string(i, buf, "typeid") or_return

	case Type_Info_Pointer:
		if info.elem == nil {
			write_string(i, buf, "rawptr") or_return
		} else {
			write_string    (i, buf, "^")       or_return
			write_write_type(i, buf, info.elem) or_return
		}
	case Type_Info_Multi_Pointer:
		write_string    (i, buf, "[^]")     or_return
		write_write_type(i, buf, info.elem) or_return
	case Type_Info_Soa_Pointer:
		write_string    (i, buf, "#soa ^")  or_return
		write_write_type(i, buf, info.elem) or_return
	case Type_Info_Procedure:
		write_string(i, buf, "proc") or_return
		if info.params == nil {
			write_string(i, buf, "()") or_return
		} else {
			t := info.params.variant.(Type_Info_Parameters)
			write_byte(i, buf, '(') or_return
			for t, j in t.types {
				if j > 0 { write_string(i, buf, ", ") or_return }
				write_write_type(i, buf, t) or_return
			}
			write_string(i, buf, ")") or_return
		}
		if info.results != nil {
			write_string    (i, buf, " -> ")       or_return
			write_write_type(i, buf, info.results) or_return
		}
	case Type_Info_Parameters:
		count := len(info.names)
		if count != 1 { write_byte(i, buf, '(') or_return }
		for name, j in info.names {
			if j > 0 { write_string(i, buf, ", ") or_return }

			t := info.types[j]

			if len(name) > 0 {
				write_string(i, buf, name) or_return
				write_string(i, buf, ": ") or_return
			}
			write_write_type(i, buf, t) or_return
		}
		if count != 1 { write_string(i, buf, ")") or_return }

	case Type_Info_Array:
		write_byte      (i, buf, '[')             or_return
		write_u64       (i, buf, u64(info.count)) or_return
		write_byte      (i, buf, ']')             or_return
		write_write_type(i, buf, info.elem)       or_return

	case Type_Info_Enumerated_Array:
		if info.is_sparse {
			write_string(i, buf, "#sparse") or_return
		}
		write_byte      (i, buf, '[')        or_return
		write_write_type(i, buf, info.index) or_return
		write_byte      (i, buf, ']')        or_return
		write_write_type(i, buf, info.elem)  or_return


	case Type_Info_Dynamic_Array:
		write_string    (i, buf, "[dynamic]") or_return
		write_write_type(i, buf, info.elem)   or_return
	case Type_Info_Slice:
		write_string    (i, buf, "[]")      or_return
		write_write_type(i, buf, info.elem) or_return

	case Type_Info_Map:
		write_string    (i, buf, "map[")     or_return
		write_write_type(i, buf, info.key)   or_return
		write_byte      (i, buf, ']')        or_return
		write_write_type(i, buf, info.value) or_return

	case Type_Info_Struct:
		switch info.soa_kind {
		case .None: // Ignore
		case .Fixed:
			write_string    (i, buf, "#soa[")            or_return
			write_u64       (i, buf, u64(info.soa_len))  or_return
			write_byte      (i, buf, ']')                or_return
			write_write_type(i, buf, info.soa_base_type) or_return
			return true
		case .Slice:
			write_string    (i, buf, "#soa[]")           or_return
			write_write_type(i, buf, info.soa_base_type) or_return
			return true
		case .Dynamic:
			write_string    (i, buf, "#soa[dynamic]")    or_return
			write_write_type(i, buf, info.soa_base_type) or_return
			return true
		}

		write_string(i, buf, "struct ") or_return
		if .packed      in info.flags { write_string(i, buf, "#packed ")      or_return }
		if .raw_union   in info.flags { write_string(i, buf, "#raw_union ")   or_return }
		if .all_or_none in info.flags { write_string(i, buf, "#all_or_none ") or_return }
		if .simple      in info.flags { write_string(i, buf, "#simple ")      or_return }
		if .align in info.flags {
			write_string(i, buf, "#align(")     or_return
			write_u64(i,    buf, u64(ti.align)) or_return
			write_string(i, buf, ") ")          or_return
		}
		write_byte(i, buf, '{') or_return
		for name, j in info.names[:info.field_count] {
			if j > 0 { write_string(i, buf, ", ") or_return }
			write_string    (i, buf, name)          or_return
			write_string    (i, buf, ": ")          or_return
			write_write_type(i, buf, info.types[j]) or_return
		}
		write_byte(i, buf, '}') or_return

	case Type_Info_Union:
		write_string(i, buf, "union ") or_return
		if info.custom_align {
			write_string(i, buf, "#align(")     or_return
			write_u64   (i, buf, u64(ti.align)) or_return
			write_string(i, buf, ") ")          or_return
		}
		if info.no_nil {
			write_string(i, buf, "#no_nil ") or_return
		}
		write_byte(i, buf, '{') or_return
		for variant, j in info.variants {
			if j > 0 { write_string(i, buf, ", ") or_return }
			write_write_type(i, buf, variant) or_return
		}
		write_string(i, buf, "}") or_return

	case Type_Info_Enum:
		write_string    (i, buf, "enum ")   or_return
		write_write_type(i, buf, info.base) or_return
		write_string    (i, buf, " {")      or_return
		for name, j in info.names {
			if j > 0 { write_string(i, buf, ", ") or_return }
			write_string(i, buf, name) or_return
		}
		write_string(i, buf, "}") or_return

	case Type_Info_Bit_Set:
		write_string(i, buf, "bit_set[") or_return

		#partial switch elem in type_info_base(info.elem).variant {
		case Type_Info_Enum:
			write_write_type(i, buf, info.elem) or_return
		case Type_Info_Rune:
			write_encoded_rune(i, buf, rune(info.lower)) or_return
			write_string      (i, buf, "..")             or_return
			write_encoded_rune(i, buf, rune(info.upper)) or_return
		case:
			write_i64   (i, buf, info.lower) or_return
			write_string(i, buf, "..")       or_return
			write_i64   (i, buf, info.upper) or_return
		}
		if info.underlying != nil {
			write_string    (i, buf, "; ")            or_return
			write_write_type(i, buf, info.underlying) or_return
		}
		write_byte(i, buf, ']') or_return

	case Type_Info_Bit_Field:
		write_string    (i, buf, "bit_field ")      or_return
		write_write_type(i, buf, info.backing_type) or_return
		write_string    (i, buf, " {")              or_return
		for name, j in info.names[:info.field_count] {
			if j > 0 { write_string(i, buf, ", ") or_return }
			write_string    (i, buf, name)                   or_return
			write_string    (i, buf, ": ")                   or_return
			write_write_type(i, buf, info.types[j])          or_return
			write_string    (i, buf, " | ")                  or_return
			write_u64       (i, buf, u64(info.bit_sizes[j])) or_return
		}
		write_byte(i, buf, '}') or_return


	case Type_Info_Simd_Vector:
		write_string    (i, buf, "#simd[")        or_return
		write_u64       (i, buf, u64(info.count)) or_return
		write_byte      (i, buf, ']')             or_return
		write_write_type(i, buf, info.elem)       or_return

	case Type_Info_Matrix:
		if info.layout == .Row_Major {
			write_string(i, buf, "#row_major ") or_return
		}
		write_string    (i, buf, "matrix[")              or_return
		write_u64       (i, buf, u64(info.row_count))    or_return
		write_string    (i, buf, ", ")                   or_return
		write_u64       (i, buf, u64(info.column_count)) or_return
		write_string    (i, buf, "]")                    or_return
		write_write_type(i, buf, info.elem)              or_return
	}
	return true
}