/* rbmake.h
 *
 * These routines are the high-level routines used to create new .rb
 * files.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#ifndef _RBMAKE_H
#define _RBMAKE_H

#include <stdio.h>
#include <string.h>
#include <rbmake/rbfile.h>

#define RB_TEXTCONV_NONE	0
#define RB_TEXTCONV_PRE		1
#define RB_TEXTCONV_SIMPLE_PARA 2
#define RB_TEXTCONV_OPT_CNT	3

#define RB_INCLUDE_NO		0
#define RB_INCLUDE_MATCH	1
#define RB_INCLUDE_YES		2
#define RB_INCLUDE_OPT_CNT 	3

#define RB_ENHANCE_DQUOTES	0x0002
#define RB_ENHANCE_SQUOTES	0x0004
#define RB_ENHANCE_EMDASHES	0x0008
#define RB_ENHANCE_ELLIPSES	0x0010

#define RB_ENHANCE_ALL		(RB_ENHANCE_DQUOTES|RB_ENHANCE_SQUOTES|RB_ENHANCE_EMDASHES|RB_ENHANCE_ELLIPSES)

#define RB_IMAGE_FULLPAGE_PORTRAIT	1
#define RB_IMAGE_FULLPAGE_LANDSCAPE	2

/* -------------------- Opaque objects -------------------- */

typedef struct RbMake RbMake;
typedef struct RbPage RbPage;
typedef struct RbImage RbImage;

typedef struct NameWithType NameWithType;
typedef struct HtmlPosition HtmlPosition;
typedef struct TagTree TagTree;

/* -------------------- Callback function typedefs -------------------- */

typedef bool (*RbMakeAllowUrlFunc)(RbPage *pg, const char *url, int type);
typedef void (*RbMakeScheduleUrlFunc)(RbMake *rb, RbPage *pg, const char *url,
				      int type);
typedef void (*RbMakeFinishInfoFunc)(RbPage *pg, RbInfoHash *infoHash);

typedef const char *(*GrabUrlAskForAuthInfoFunc)(const char *url,
						 const char *realm);

typedef void (*RbFetchStatusFunc)(RbMake *rb, const char *tocname,
				  const char *url, const char *errstr,
				  int joinedCnt);

typedef void (*RbImageFunc)(RbImage *img, const char *url);

/* -------------------- RbMake_* routines -------------------- */

void RbMake_init(void);
void RbMake_cleanup(void);
RbMake *RbMake_new(RbMakeAllowUrlFunc allowFunc,
		   RbMakeScheduleUrlFunc schedFunc,
		   RbMakeFinishInfoFunc finishInfoFunc);
char *RbMake_mungeBookName(RbMake *me, const char *bn);
bool RbMake_create(RbMake *me, const char *fn, int maxTocCnt);
void RbMake_finish(RbMake *me);
NameWithType *RbMake_findPageName(RbMake *me, const char *arg);
NameWithType *RbMake_addPageName(RbMake *me, const char *arg, int type);

int RbMake_getPageCount(RbMake *me);
int RbMake_getBinaryCount(RbMake *me);
int RbMake_getDoneCount(RbMake *me);
int RbMake_getHtmlDropCount(RbMake *me);
int RbMake_getBinaryDropCount(RbMake *me);

const char *RbMake_getFileName(RbMake *me);
const char *RbMake_getNewFileName(RbMake *me);
RbInfoHash *RbMake_getInfoHash(RbMake *me);

void RbMake_setAllowHRSize0PageBreaks(RbMake *me, bool trueOrFalse);
void RbMake_setBookParagraphDepth(RbMake *me, int depth);
void RbMake_setCoverImage(RbMake *me, const char *url);
void RbMake_setCreateHkeyFile(RbMake *me, bool trueOrFalse);
void RbMake_setEnhancePunctuation(RbMake *me, int enhanceFlags);
void RbMake_setFollowLinks(RbMake *me, int depth);
void RbMake_setGenerator(RbMake *me, const char *gen);
void RbMake_setIncludeAudio(RbMake *me, int incl);
void RbMake_setIncludeImages(RbMake *me, int incl);
void RbMake_setPageJoining(RbMake *me, int join);
void RbMake_setTextConversion(RbMake *me, int conv);
void RbMake_setUseBookParagraphs(RbMake *me, bool trueOrFalse);
void RbMake_setVerboseOutput(RbMake *me, bool trueOrFalse);

void RbMake_addMenuItem(RbMake *me, const char *text, const char *url);
const char *RbMake_addSubstRules(RbMake *me, MBuf *mb);

bool RbMake_getAllowHRSize0PageBreaks(RbMake *me);
int RbMake_getBookParagraphDepth(RbMake *me);
const char *RbMake_getCoverImage(RbMake *me);
bool RbMake_getCreateHkeyFile(RbMake *me);
int RbMake_getEnhancePunctuation(RbMake *me);
int RbMake_getFollowLinks(RbMake *me);
const char *RbMake_getGenerator(RbMake *me);
int RbMake_getIncludeAudio(RbMake *me);
int RbMake_getIncludeImages(RbMake *me);
MArray *RbMake_getMenuItems(RbMake *me);
int RbMake_getPageJoining(RbMake *me);
int RbMake_getTextConversion(RbMake *me);
bool RbMake_getUseBookParagraphs(RbMake *me);
bool RbMake_getVerboseOutput(RbMake *me);

char *RbMake_getTextConvOptName(int opt);
int RbMake_findTextConvOpt(const char *str, int len);
char *RbMake_getIncludeOptName(int opt);
int RbMake_findIncludeOpt(const char *str, int len);

/* -------------------- RbPage_* routines -------------------- */

RbPage *RbPage_new(RbMake *me, RbPage *fromPage, const char *url, int type);
void RbPage_delete(RbPage *me);
void RbPage_changeType(RbPage *me, int type);
const char *RbPage_appendContent(RbPage *me, const char *bp, int len);
void RbPage_finishContent(RbPage *me);
const char *RbPage_write(RbPage *me);
void RbPage_drop(RbPage *me);
char *RbPage_makeRbRef(RbPage *me, char *url);
void RbPage_noteHidxName(RbPage *me, const char *name, int32 hOff);
HtmlPosition *RbPage_usedHidxName(RbPage *me, const char *name, bool used);
void RbPage_noteHidxPara(RbPage *me, int32 off, TagTree *tt);
void RbPage_delLastHidxPara(RbPage *me, int elnum, int fudgePos);

RbMake *RbPage_getRbMake(RbPage *me);
const char *RbPage_getUrl(RbPage *me);
const char *RbPage_getTocName(RbPage *me);

MBuf *RbPage_getContent(RbPage *me);
bool RbPage_getConvertImage(RbPage *me);
bool RbPage_getParsePage(RbPage *me);
int RbPage_getType(RbPage *me);

void RbPage_setContent(RbPage *me, MBuf *mb);
void RbPage_setConvertImage(RbPage *me, bool trueOrFalse);
void RbPage_setParsePage(RbPage *me, bool trueOrFalse);

/* -------------------- GrabUrl_* routines -------------------- */

void GrabUrl_init(GrabUrlAskForAuthInfoFunc authFunc);
int GrabUrl_guessAuthHeader(const char *url);
const char *GrabUrl_askForAuthInfo(const char *url, void *ctxt);
const char *GrabUrl_setAuthInfo(const char *url, const char *realm,
				const char *b64str);
int GrabUrl_setHttpHeader(const char *header);
const char *GrabUrl_getHttpHeaders(void);
MBuf *GrabUrl_read(const char *url, MBuf *mb);

/* -------------------- RbFetch_* routines -------------------- */

void RbFetch_init(RbFetchStatusFunc statFunc);
void RbFetch_cleanup(void);
void RbFetch_loop(void);
void RbFetch_getURL(RbMake *rb, RbPage *pg, const char *url, int type);
RbFile *RbFetch_cachedOpenUrl(const char *url, int openFlags);
int RbFetch_prepareForRbContents(RbMake *rb, const char *url, bool unjoin,
				bool mergeInfo);
void RbFetch_getRbContents(RbMake *rb, const char *url, bool unjoin);
void RbFetch_getRbPage(RbMake *rb, RbFile *rbf, ToC *toc, const char *url);

/* -------------------- RbHtml_* routines -------------------- */

void RbHtml_init(void);
void RbHtml_cleanup(void);
void RbHtml_parsedPushFunc(void *userPtr, const char *bp, int len);
void RbHtml_parsedTextPushFunc(void *userPtr, const char *bp, int len);
void RbHtml_flushParsedPush(RbPage *po);

/* -------------------- RbImage_* routines -------------------- */

bool rbLibHasImageSupport(void);

void RbImage_init(RbImageFunc imageFunc);
MBuf *RbImage_turnMBufIntoPngMBuf(MBuf *mb, const char *url, int resizeIt);
void RbImage_setError(const char *msg);
const char *RbImage_lastErrorMessage(void);

void RbImage_toGray(RbImage *img);
void RbImage_toMono(RbImage *img);
void RbImage_resize(RbImage *img, int newcols, int newrows);
void RbImage_enhanceGray(RbImage *img, int n);
void RbImage_rotate(RbImage *img);

/* -------------------- URL-building routines -------------------- */

char *rbBuildURL(const char *url, const char *base);
void rbFreeURL(char *url);

#endif
