/*
 *  Copyright (C) 2019 KeePassXC Team <team@keepassxc.org>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 or (at your option)
 *  version 3 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "AttachmentRemove.h"

#include "Utils.h"
#include "core/Global.h"
#include "core/Group.h"

#include <QCommandLineParser>

AttachmentRemove::AttachmentRemove()
{
    name = QString("attachment-rm");
    description = QObject::tr("Remove an attachment of an entry.");
    positionalArguments.append(
        {QString("entry"), QObject::tr("Path of the entry with the target attachment."), QString("")});
    positionalArguments.append({QString("name"), QObject::tr("Name of the attachment to be removed."), QString("")});
}

int AttachmentRemove::executeWithDatabase(QSharedPointer<Database> database, QSharedPointer<QCommandLineParser> parser)
{
    auto& out = parser->isSet(Command::QuietOption) ? Utils::DEVNULL : Utils::STDOUT;
    auto& err = Utils::STDERR;

    auto args = parser->positionalArguments();
    auto entryPath = args.at(1);

    auto entry = database->rootGroup()->findEntryByPath(entryPath);
    if (!entry) {
        err << QObject::tr("Could not find entry with path %1.").arg(entryPath) << Qt::endl;
        return EXIT_FAILURE;
    }

    auto attachmentName = args.at(2);

    auto attachments = entry->attachments();
    if (!attachments->hasKey(attachmentName)) {
        err << QObject::tr("Could not find attachment with name %1.").arg(attachmentName) << Qt::endl;
        return EXIT_FAILURE;
    }

    entry->beginUpdate();
    attachments->remove(attachmentName);
    entry->endUpdate();

    QString errorMessage;
    if (!database->save(Database::Atomic, {}, &errorMessage)) {
        err << QObject::tr("Writing the database failed %1.").arg(errorMessage) << Qt::endl;
        return EXIT_FAILURE;
    }

    out << QObject::tr("Successfully removed attachment %1 from entry %2.").arg(attachmentName, entryPath) << Qt::endl;
    return EXIT_SUCCESS;
}
