"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultBackend = void 0;

var _performanceNow = _interopRequireDefault(require("performance-now"));

var _uuid = require("uuid");

var _query_helpers = require("../../common/helpers/notebooks/query_helpers");

var _sample_notebooks = require("../../common/helpers/notebooks/sample_notebooks");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class DefaultBackend {
  constructor() {
    _defineProperty(this, "backend", 'Default Backend');

    _defineProperty(this, "createNewNotebook", newNotebookName => {
      const noteObject = {
        dateCreated: new Date().toISOString(),
        name: newNotebookName,
        dateModified: new Date().toISOString(),
        backend: 'Default',
        paragraphs: []
      };
      return {
        object: noteObject
      };
    });

    _defineProperty(this, "indexNote", async function (client, body) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            notebook: body
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Doc Error:' + error);
      }
    });

    _defineProperty(this, "updateNote", async function (client, noteId, updateBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: noteId,
          body: {
            notebook: updateBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Doc Error:' + error);
      }
    });

    _defineProperty(this, "getNote", async function (client, noteId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: noteId
        });

        if (response.observabilityObjectList.length === 0) {
          throw new Error('notebook id not found');
        }

        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Doc Error:' + error);
      }
    });

    _defineProperty(this, "viewNotes", async function (client, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'notebook'
        });
        return response.observabilityObjectList.map(notebook => ({
          path: notebook.notebook.name,
          id: notebook.objectId,
          dateCreated: notebook.notebook.dateCreated,
          dateModified: notebook.notebook.dateModified
        }));
      } catch (error) {
        if (error.body.error.type === 'index_not_found_exception') {
          return [];
        } else throw new Error('View Notebooks Error:' + error);
      }
    });

    _defineProperty(this, "fetchNote", async function (client, noteId, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, noteId);
        return {
          path: noteObject.notebook.name,
          dateCreated: noteObject.notebook.dateCreated,
          dateModified: noteObject.notebook.dateModified,
          paragraphs: noteObject.notebook.paragraphs
        };
      } catch (error) {
        throw new Error('Fetching Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addNote", async function (client, params, _wreckOptions) {
      try {
        const newNotebook = this.createNewNotebook(params.name);
        const opensearchClientResponse = await this.indexNote(client, newNotebook.object);
        return {
          status: 'OK',
          message: opensearchClientResponse,
          body: opensearchClientResponse.objectId
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addSampleNotes", async function (client, visIds, _wreckOptions) {
      try {
        const notebooks = (0, _sample_notebooks.getSampleNotebooks)(visIds);
        const newNotebooks = [];

        for (let i = 0; i < notebooks.length; i++) {
          const notebook = notebooks[i];
          await this.indexNote(client, notebook.notebook).then(response => {
            newNotebooks.push({
              id: response.objectId,
              name: notebook.notebook.name,
              dateModified: notebook.dateModified,
              dateCreated: notebook.dateCreated
            });
          });
        }

        return {
          status: 'OK',
          message: '',
          body: newNotebooks
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });

    _defineProperty(this, "renameNote", async function (client, params, _wreckOptions) {
      try {
        const updateNotebook = {
          name: params.name,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          status: 'OK',
          message: opensearchClientResponse
        };
      } catch (error) {
        throw new Error('Renaming Notebook Error:' + error);
      }
    });

    _defineProperty(this, "cloneNote", async function (client, params, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, params.noteId);
        const newNotebook = this.createNewNotebook(params.name);
        const cloneNotebook = { ...newNotebook.object
        };
        cloneNotebook.paragraphs = noteObject.notebook.paragraphs;
        const opensearchClientIndexResponse = await this.indexNote(client, cloneNotebook);
        return {
          status: 'OK',
          body: { ...cloneNotebook,
            id: opensearchClientIndexResponse.objectId
          }
        };
      } catch (error) {
        throw new Error('Cloning Notebook Error:' + error);
      }
    });

    _defineProperty(this, "deleteNote", async function (client, noteList, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: noteList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Deleting Notebook Error:' + error);
      }
    });

    _defineProperty(this, "exportNote", async function (client, noteId, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, noteId);
        return {
          status: 'OK',
          body: opensearchClientGetResponse
        };
      } catch (error) {
        throw new Error('Export Notebook Error:' + error);
      }
    });

    _defineProperty(this, "importNote", async function (client, noteObj, _wreckOptions) {
      try {
        const newNoteObject = { ...noteObj
        };
        newNoteObject.id = 'note_' + (0, _uuid.v4)();
        newNoteObject.dateCreated = new Date().toISOString();
        newNoteObject.dateModified = new Date().toISOString();
        const opensearchClientIndexResponse = await this.indexNote(client, newNoteObject);
        return {
          status: 'OK',
          message: opensearchClientIndexResponse,
          body: opensearchClientIndexResponse.objectId
        };
      } catch (error) {
        throw new Error('Import Notebook Error:' + error);
      }
    });

    _defineProperty(this, "updateParagraph", function (paragraphs, paragraphId, paragraphInput, paragraphType) {
      try {
        const updatedParagraphs = [];
        paragraphs.map(paragraph => {
          const updatedParagraph = { ...paragraph
          };

          if (paragraph.id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            updatedParagraph.input.inputText = paragraphInput;

            if (paragraphType.length > 0) {
              updatedParagraph.input.inputType = paragraphType;
            }
          }

          updatedParagraphs.push(updatedParagraph);
        });
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Update Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "createParagraph", function (paragraphInput, inputType) {
      try {
        let paragraphType = 'MARKDOWN';

        if (inputType === 'VISUALIZATION') {
          paragraphType = 'VISUALIZATION';
        }

        if (inputType === 'OBSERVABILITY_VISUALIZATION') {
          paragraphType = 'OBSERVABILITY_VISUALIZATION';
        }

        if (paragraphInput.substring(0, 3) === '%sql' || paragraphInput.substring(0, 3) === '%ppl') {
          paragraphType = 'QUERY';
        }

        const inputObject = {
          inputType: paragraphType,
          inputText: paragraphInput
        };
        const outputObjects = [{
          outputType: paragraphType,
          result: '',
          execution_time: '0s'
        }];
        const newParagraph = {
          id: 'paragraph_' + (0, _uuid.v4)(),
          dateCreated: new Date().toISOString(),
          dateModified: new Date().toISOString(),
          input: inputObject,
          output: outputObjects
        };
        return newParagraph;
      } catch (error) {
        throw new Error('Create Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "runParagraph", async function (paragraphs, paragraphId, client) {
      try {
        const updatedParagraphs = [];
        let index = 0;

        for (index = 0; index < paragraphs.length; ++index) {
          const startTime = (0, _performanceNow.default)();
          const updatedParagraph = { ...paragraphs[index]
          };

          if (paragraphs[index].id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();

            if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'QUERY',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'OBSERVABILITY_VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'OBSERVABILITY_VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: 'Please select an input type (%sql, %ppl, or %md)',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            }
          }

          updatedParagraphs.push(updatedParagraph);
        }

        return updatedParagraphs;
      } catch (error) {
        throw new Error('Running Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateRunFetchParagraph", async function (client, request, _wreckOptions) {
      try {
        const scopedClient = client.asScoped(request);
        const params = request.body;
        const opensearchClientGetResponse = await this.getNote(scopedClient, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput, params.paragraphType);
        const updatedOutputParagraphs = await this.runParagraph(updatedInputParagraphs, params.paragraphId, client);
        const updateNotebook = {
          paragraphs: updatedOutputParagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(scopedClient, params.noteId, updateNotebook);
        let resultParagraph = {};
        let index = 0;

        for (index = 0; index < updatedOutputParagraphs.length; ++index) {
          if (params.paragraphId === updatedOutputParagraphs[index].id) {
            resultParagraph = updatedOutputParagraphs[index];
          }
        }

        return resultParagraph;
      } catch (error) {
        throw new Error('Update/Run Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateFetchParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput);
        const updateNotebook = {
          paragraphs: updatedInputParagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        let resultParagraph = {};
        updatedInputParagraphs.map(paragraph => {
          if (params.paragraphId === paragraph.id) {
            resultParagraph = paragraph;
          }
        });
        return resultParagraph;
      } catch (error) {
        throw new Error('Save Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "addFetchNewParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const paragraphs = opensearchClientGetResponse.notebook.paragraphs;
        const newParagraph = this.createParagraph(params.paragraphInput, params.inputType);
        paragraphs.splice(params.paragraphIndex, 0, newParagraph);
        const updateNotebook = {
          paragraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return newParagraph;
      } catch (error) {
        throw new Error('add/Fetch Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "deleteFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];

        if (params.paragraphId !== undefined) {
          opensearchClientGetResponse.notebook.paragraphs.map((paragraph, index) => {
            if (paragraph.id !== params.paragraphId) {
              updatedparagraphs.push(paragraph);
            }
          });
        }

        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        console.log('error', error);
        throw new Error('Delete Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "clearAllFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];
        opensearchClientGetResponse.notebook.paragraphs.map((paragraph, index) => {
          const updatedParagraph = { ...paragraph
          };
          updatedParagraph.output = [];
          updatedparagraphs.push(updatedParagraph);
        });
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        throw new Error('Clear Paragraph Error:' + error);
      }
    });
  } // Creates a new notebooks with sample markdown text
  // indexes a notebook with body provided
  // updates a notebook with updateBody provided as parameter
  // fetched a notebook by Id
  // gets first `FETCH_SIZE` notebooks available

  /* Fetches a notebook by id
   * Param: noteId -> Id of notebook to be fetched
   */

  /* Adds a notebook to storage
   * Param: name -> name of new notebook
   */

  /* Adds sample notebooks to storage
   * Param: name -> name of new notebook
   */

  /* Renames a notebook
   * Params: name -> new name for the notebook to be renamed
   *         noteId -> Id of notebook to be fetched
   */

  /* Clone a notebook
   * Params: name -> new name for the cloned notebook
   *         noteId -> Id for the notebook to be cloned
   */

  /* Delete a notebook
   * Param: noteId -> Id for the notebook to be deleted
   */

  /* Export a notebook
   * Param: noteId -> Id for the notebook to be exported
   */

  /* Import a notebook
   * Params: noteObj -> note Object to be imported
   */

  /* Updates input for required paragraphs
   * Params: paragraphs -> list of paragraphs
   *         paragraphId -> Id of paragraph to be updated
   *         paragraphInput -> Input to be added
   */
  // Creates new paragraph with the given input and input type

  /* Runs a paragraph
   * Currently only runs markdown by copying input.inputText to result
   * UI renders Markdown
   */

  /* --> Updates a Paragraph with input content
   * --> Runs it
   * --> Updates the notebook in index
   * --> Fetches the updated Paragraph (with new input content and output result)
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be updated
   *         paragraphInput -> paragraph input code
   */

  /* --> Updates a Paragraph with input content
   * --> Updates the notebook in index
   * --> Fetches the updated Paragraph (with new input content)
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be updated
   *         paragraphInput -> paragraph input code
   */

  /* --> Fetches the Paragraph
   * --> Adds a Paragraph with input content
   * --> Updates the notebook in index
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be fetched
   */

  /* --> Deletes a Paragraph with id
   * --> Fetches the all other Paragraphs as a list
   * --> Updates the notebook in index
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be deleted
   */

  /* --> Clears output for all the paragraphs
   * --> Fetches the all Paragraphs as a list (with cleared outputs)
   * --> Updates the notebook in index
   * Param: noteId -> Id of notebook to be cleared
   */


}

exports.DefaultBackend = DefaultBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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