'use strict';

var _classCallCheck2 = require('babel-runtime/helpers/classCallCheck');

var _classCallCheck3 = _interopRequireDefault(_classCallCheck2);

var _createClass2 = require('babel-runtime/helpers/createClass');

var _createClass3 = _interopRequireDefault(_createClass2);

var _possibleConstructorReturn2 = require('babel-runtime/helpers/possibleConstructorReturn');

var _possibleConstructorReturn3 = _interopRequireDefault(_possibleConstructorReturn2);

var _inherits2 = require('babel-runtime/helpers/inherits');

var _inherits3 = _interopRequireDefault(_inherits2);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var _require = require('../../core'),
    checkType = _require.util.checkType;

var MetricsAggregationBase = require('./metrics-aggregation-base');

/**
 * A multi-value metrics aggregation that calculates one or more percentiles
 * over numeric values extracted from the aggregated documents. These values can
 * be extracted either from specific numeric fields in the documents, or be
 * generated by a provided script.
 *
 * [Elasticsearch reference](https://www.elastic.co/guide/en/elasticsearch/reference/current/search-aggregations-metrics-percentile-aggregation.html)
 *
 * Aggregation that calculates one or more percentiles over numeric values
 * extracted from the aggregated documents.
 *
 * @example
 * const agg = esb.percentilesAggregation('load_time_outlier', 'load_time');
 *
 * @example
 * // Convert load time from mills to seconds on-the-fly using script
 * const agg = esb.percentilesAggregation('load_time_outlier').script(
 *     esb.script('inline', "doc['load_time'].value / params.timeUnit")
 *         .lang('painless')
 *         .params({ timeUnit: 1000 })
 * );
 *
 * @param {string} name The name which will be used to refer to this aggregation.
 * @param {string=} field The field to aggregate on
 *
 * @extends MetricsAggregationBase
 */

var PercentilesAggregation = function (_MetricsAggregationBa) {
    (0, _inherits3.default)(PercentilesAggregation, _MetricsAggregationBa);

    // eslint-disable-next-line require-jsdoc
    function PercentilesAggregation(name, field) {
        (0, _classCallCheck3.default)(this, PercentilesAggregation);
        return (0, _possibleConstructorReturn3.default)(this, (PercentilesAggregation.__proto__ || Object.getPrototypeOf(PercentilesAggregation)).call(this, name, 'percentiles', field));
    }

    /**
     * Enable the response to be returned as a keyed object where the key is the
     * bucket interval.
     *
     * @example
     * // Return the ranges as an array rather than a hash
     * const agg = esb.percentilesAggregation('balance_outlier', 'balance').keyed(
     *     false
     * );
     *
     * @param {boolean} keyed To enable keyed response or not. True by default
     * @returns {PercentilesAggregation} returns `this` so that calls can be chained
     */


    (0, _createClass3.default)(PercentilesAggregation, [{
        key: 'keyed',
        value: function keyed(_keyed) {
            this._aggsDef.keyed = _keyed;
            return this;
        }

        /**
         * Specifies the percents of interest.
         * Requested percentiles must be a value between 0-100 inclusive
         *
         * @example
         * // Specify particular percentiles to calculate
         * const agg = esb.percentilesAggregation(
         *     'load_time_outlier',
         *     'load_time'
         * ).percents([95, 99, 99.9]);
         *
         * @param {Array<number>} percents Parameter to specify particular percentiles to calculate
         * @returns {PercentilesAggregation} returns `this` so that calls can be chained
         * @throws {TypeError} If `percents` is not an instance of Array
         */

    }, {
        key: 'percents',
        value: function percents(_percents) {
            checkType(_percents, Array);
            this._aggsDef.percents = _percents;
            return this;
        }

        /**
         * Compression controls memory usage and approximation error. The compression
         * value limits the maximum number of nodes to 100 * compression. By
         * increasing the compression value, you can increase the accuracy of your
         * percentiles at the cost of more memory. Larger compression values also make
         * the algorithm slower since the underlying tree data structure grows in
         * size, resulting in more expensive operations. The default compression
         * value is 100.
         *
         * @example
         * const agg = esb.percentilesAggregation(
         *     'load_time_outlier',
         *     'load_time'
         * ).tdigest(200);
         *
         * @param {number} compression Parameter to balance memory utilization with estimation accuracy.
         * @returns {PercentilesAggregation} returns `this` so that calls can be chained
         */

    }, {
        key: 'tdigest',
        value: function tdigest(compression) {
            this._aggsDef.tdigest = { compression: compression };
            return this;
        }

        /**
         * Compression controls memory usage and approximation error. The compression
         * value limits the maximum number of nodes to 100 * compression. By
         * increasing the compression value, you can increase the accuracy of your
         * percentiles at the cost of more memory. Larger compression values also make
         * the algorithm slower since the underlying tree data structure grows in
         * size, resulting in more expensive operations. The default compression
         * value is 100.
         *
         * Alias for `tdigest`
         *
         * @example
         * const agg = esb.percentilesAggregation(
         *     'load_time_outlier',
         *     'load_time'
         * ).compression(200);
         *
         * @param {number} compression Parameter to balance memory utilization with estimation accuracy.
         * @returns {PercentilesAggregation} returns `this` so that calls can be chained
         */

    }, {
        key: 'compression',
        value: function compression(_compression) {
            this._aggsDef.tdigest = { compression: _compression };
            return this;
        }

        /**
         * HDR Histogram (High Dynamic Range Histogram) is an alternative implementation
         * that can be useful when calculating percentiles for latency measurements
         * as it can be faster than the t-digest implementation
         * with the trade-off of a larger memory footprint.
         *
         * The HDR Histogram can be used by specifying the method parameter in the request.
         *
         * @example
         * const agg = esb.percentilesAggregation('load_time_outlier', 'load_time')
         *     .percents([95, 99, 99.9])
         *     .hdr(3);
         *
         * @param {number} numberOfSigDigits The resolution of values
         * for the histogram in number of significant digits
         * @returns {PercentilesAggregation} returns `this` so that calls can be chained
         */

    }, {
        key: 'hdr',
        value: function hdr(numberOfSigDigits) {
            this._aggsDef.hdr = {
                number_of_significant_value_digits: numberOfSigDigits
            };
            return this;
        }
    }]);
    return PercentilesAggregation;
}(MetricsAggregationBase);

module.exports = PercentilesAggregation;