"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FlowFrameworkRoutesService = void 0;
exports.registerFlowFrameworkRoutes = registerFlowFrameworkRoutes;
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var _semver = _interopRequireDefault(require("semver"));
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helpers = require("../utils/helpers");
var _helpers2 = require("./helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
/**
 * Server-side routes to process flow-framework-related node API calls and execute the
 * corresponding API calls against the OpenSearch cluster.
 */
function registerFlowFrameworkRoutes(router, flowFrameworkRoutesService) {
  router.get({
    path: `${_common.GET_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflow);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflow);
  router.post({
    path: _common.SEARCH_WORKFLOWS_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.searchWorkflows);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/search`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.searchWorkflows);
  router.get({
    path: `${_common.GET_WORKFLOW_STATE_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflowState);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/state/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflowState);
  router.post({
    path: _common.CREATE_WORKFLOW_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.createWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/create`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.createWorkflow);
  router.put({
    path: `${_common.UPDATE_WORKFLOW_NODE_API_PATH}/{workflow_id}/{update_fields}/{reprovision}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        update_fields: _configSchema.schema.boolean(),
        reprovision: _configSchema.schema.boolean()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.updateWorkflow);
  router.put({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/update/{workflow_id}/{update_fields}/{reprovision}`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string(),
        workflow_id: _configSchema.schema.string(),
        update_fields: _configSchema.schema.boolean(),
        reprovision: _configSchema.schema.boolean()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.updateWorkflow);
  router.post({
    path: `${_common.PROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.provisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/provision/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.provisionWorkflow);
  router.post({
    path: `${_common.DEPROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/deprovision/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.DEPROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}/{resource_ids}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        resource_ids: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/deprovision/{workflow_id}/{resource_ids}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        resource_ids: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.delete({
    path: `${_common.DELETE_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deleteWorkflow);
  router.delete({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/delete/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deleteWorkflow);
  router.get({
    path: _common.GET_PRESET_WORKFLOWS_NODE_API_PATH,
    validate: {}
  }, flowFrameworkRoutesService.getPresetWorkflows);
}
class FlowFrameworkRoutesService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "client", void 0);
    // TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
    // https://github.com/opensearch-project/flow-framework/issues/171
    // Current implementation is making two calls and combining results via helper fn
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.getWorkflow', {
          workflow_id
        });
        const workflow = (0, _helpers2.toWorkflowObj)(response, workflow_id);
        const stateResponse = await callWithRequest('flowFramework.getWorkflowState', {
          workflow_id
        });
        const state = (0, _helpers2.getWorkflowStateFromResponse)(stateResponse.state);
        const resourcesCreated = (0, _helpers2.getResourcesCreatedFromResponse)(stateResponse.resources_created);
        const workflowWithState = {
          ...workflow,
          state,
          error: stateResponse.error,
          resourcesCreated
        };
        return res.ok({
          body: {
            workflow: workflowWithState
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    // TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
    // https://github.com/opensearch-project/flow-framework/issues/171
    // Current implementation is making two calls and combining results via helper fn
    _defineProperty(this, "searchWorkflows", async (context, req, res) => {
      const body = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const workflowsResponse = await callWithRequest('flowFramework.searchWorkflows', {
          body
        });
        const workflowHits = workflowsResponse.hits.hits;
        const workflowStatesResponse = await callWithRequest('flowFramework.searchWorkflowState', {
          body
        });
        const workflowStateHits = workflowStatesResponse.hits.hits;
        const workflowDict = (0, _helpers2.getWorkflowsFromResponses)(workflowHits, workflowStateHits);
        return res.ok({
          body: {
            workflows: workflowDict
          }
        });
      } catch (err) {
        if ((0, _helpers2.isIgnorableError)(err)) {
          return res.ok({
            body: {
              workflows: {}
            }
          });
        }
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getWorkflowState", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.getWorkflowState', {
          workflow_id
        });
        const state = (0, _helpers2.getWorkflowStateFromResponse)(response.state);
        const resourcesCreated = (0, _helpers2.getResourcesCreatedFromResponse)(response.resources_created);
        return res.ok({
          body: {
            workflowId: workflow_id,
            workflowState: state,
            resourcesCreated
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      const body = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.createWorkflow', {
          body
        });
        const workflowWithId = {
          ...body,
          id: response.workflow_id
        };
        return res.ok({
          body: {
            workflow: workflowWithId
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "updateWorkflow", async (context, req, res) => {
      const {
        workflow_id,
        update_fields,
        reprovision
      } = req.params;
      const workflowTemplate = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const {
          data_source_version
        } = req.query;
        const isPreV219 = data_source_version !== undefined ? _semver.default.lt(data_source_version, _common.MINIMUM_FULL_SUPPORTED_VERSION) : false;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        if (reprovision) {
          if (isPreV219) {
            await callWithRequest('flowFramework.updateAndReprovisionWorkflowAsync', {
              workflow_id,
              update_fields,
              body: workflowTemplate
            });
          } else {
            await callWithRequest('flowFramework.updateAndReprovisionWorkflow', {
              workflow_id,
              update_fields,
              body: workflowTemplate
            });
          }
        } else {
          await callWithRequest('flowFramework.updateWorkflow', {
            workflow_id,
            update_fields,
            body: workflowTemplate
          });
        }
        return res.ok({
          body: {
            workflowId: workflow_id,
            workflowTemplate
          }
        });
      } catch (err) {
        if ((0, _helpers2.isIgnorableError)(err)) {
          return res.ok({
            body: {
              workflowId: workflow_id,
              workflowTemplate
            }
          });
        }
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "provisionWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const {
          data_source_version
        } = req.query;
        const isPreV219 = data_source_version !== undefined ? _semver.default.lt(data_source_version, _common.MINIMUM_FULL_SUPPORTED_VERSION) : false;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        if (isPreV219) {
          await callWithRequest('flowFramework.provisionWorkflowAsync', {
            workflow_id
          });
        } else {
          await callWithRequest('flowFramework.provisionWorkflow', {
            workflow_id
          });
        }
        return res.ok();
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "deprovisionWorkflow", async (context, req, res) => {
      const {
        workflow_id,
        resource_ids
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        if (resource_ids !== undefined) {
          await callWithRequest('flowFramework.forceDeprovisionWorkflow', {
            workflow_id,
            resource_ids
          });
        } else {
          await callWithRequest('flowFramework.deprovisionWorkflow', {
            workflow_id
          });
        }
        return res.ok();
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.deleteWorkflow', {
          workflow_id
        });
        return res.ok({
          body: {
            id: response._id
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getPresetWorkflows", async (context, req, res) => {
      try {
        // In the future we may get these from backend via some API. For now we can
        // persist a set of working presets on server-side.
        const jsonTemplateDir = _path.default.resolve(__dirname, '../resources/templates');
        const jsonTemplates = _fs.default.readdirSync(jsonTemplateDir).filter(file => _path.default.extname(file) === '.json');
        const workflowTemplates = [];
        jsonTemplates.forEach(jsonTemplate => {
          const templateData = _fs.default.readFileSync(_path.default.join(jsonTemplateDir, jsonTemplate));
          const workflowTemplate = JSON.parse(templateData.toString());
          workflowTemplates.push(workflowTemplate);
        });
        return res.ok({
          body: {
            workflowTemplates
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.FlowFrameworkRoutesService = FlowFrameworkRoutesService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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