/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2003-10-01
 * Description : a kipi plugin to e-mailing images
 *
 * Copyright (C) 2003-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 * Copyright (C) 2006 by Tom Albers <tomalbers at kde dot nl>
 * Copyright (C) 2006 by Michael Hoechstetter <michael dot hoechstetter at gmx dot de>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqvbox.h>
#include <tqlayout.h>
#include <tqdir.h>
#include <tqwidget.h>
#include <tqfile.h>
#include <tqtextstream.h>
#include <tqlabel.h>
#include <tqgroupbox.h>
#include <tqwhatsthis.h>
#include <tqcombobox.h>
#include <tqcheckbox.h>
#include <tqprogressdialog.h>
#include <tqimage.h>
#include <tqevent.h>
#include <tqdragobject.h>
#include <tqstrlist.h>
#include <tqfileinfo.h>
#include <tqpushbutton.h>

// KDE includes.

#include <tdelocale.h>
#include <klineedit.h>
#include <kiconloader.h>
#include <tdefiledialog.h>
#include <tdemessagebox.h>
#include <knuminput.h>
#include <tdeinstance.h>
#include <tdeconfig.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include <kbuttonbox.h>
#include <ksqueezedtextlabel.h>
#include <tdeapplication.h>
#include <khelpmenu.h>
#include <tdepopupmenu.h>
#include <kurlrequester.h>

// LibKipi includes.

#include <libkipi/imagedialog.h>

// Local includes.

#include "kpaboutdata.h"
#include "pluginsversion.h"
#include "listimageserrordialog.h"
#include "sendimagesdialog.h"
#include "sendimagesdialog.moc"

namespace KIPISendimagesPlugin
{

class ImageItem : public TQListBoxText
{

public:
    ImageItem(TQListBox * parent, TQString const & comments, KURL const & url)
        : TQListBoxText(parent), _comments(comments), _url(url)
    {}

    TQString comments()                   { return _comments;                         }
    TQString name()                       { return _url.fileName();                   }
    KURL    url()                        { return _url;                              }
    TQString album()                      { return _url.directory().section('/', -1); }
    void setName(const TQString &newName) { setText(newName);                         }

private:

    TQString _comments;
    KURL    _url;
};

ListImageItems::ListImageItems(TQWidget *parent, const char *name)
              : TDEListBox(parent, name)
{
    setSelectionMode (TQListBox::Extended);
    setAcceptDrops(true);
}

void ListImageItems::dragEnterEvent(TQDragEnterEvent *e)
{
    e->accept(TQUriDrag::canDecode(e));
}

void ListImageItems::dropEvent(TQDropEvent *e)
{
    TQStrList strList;
    TQStringList FilesPath;

    if ( !TQUriDrag::decode(e, strList) ) return;

    TQStrList stringList;
    TQStrListIterator it(strList);
    char *str;

    while ( (str = it.current()) != 0 )
    {
        TQString filePath = TQUriDrag::uriToLocalFile(str);
        TQFileInfo fileInfo(filePath);

        if (fileInfo.isFile() && fileInfo.exists())
            FilesPath.append(fileInfo.filePath());
    
        ++it;
    }

    if (FilesPath.isEmpty() == false)
       emit addedDropItems(FilesPath);
}

SendImagesDialog::SendImagesDialog(TQWidget *parent, KIPI::Interface* interface,
                                   const KIPI::ImageCollection& images )
                : KDialogBase( IconList, i18n("Email Images Options"), Help|Ok|Cancel,
                  Ok, parent, "SendImagesDialog", false, true )
{
    m_interface = interface;
    m_thumbJob  = 0;

    setupImagesList();
    setupEmailOptions();
    readSettings();
    setImagesList( images.images() );
    page_setupImagesList->setFocus();
    m_ImagesFilesListBox->setSelected(0, true);
    slotImageSelected(m_ImagesFilesListBox->item(0));
    setNbItems();
    resize( 600, 400 );

    // About data and help button.

    m_about = new KIPIPlugins::KPAboutData(I18N_NOOP("Send Images"),
                                           0,
                                           TDEAboutData::License_GPL,
                                           I18N_NOOP("A Kipi plugin for emailing images"),
                                           "(c) 2003-2007, Gilles Caulier");

    m_about->addAuthor("Gilles Caulier", I18N_NOOP("Author and maintainer"),
                       "caulier dot gilles at gmail dot com");

    m_about->addAuthor("Michael Hoechstetter", I18N_NOOP("Developer"),
                       "michael dot hoechstetter at gmx dot de");

    m_about->addAuthor("Tom Albers", I18N_NOOP("Developer"),
                       "tomalbers at kde dot nl");

    m_helpButton        = actionButton( Help );
    KHelpMenu* helpMenu = new KHelpMenu(this, m_about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"), this, TQ_SLOT(slotHelp()), 0, -1, 0);
    m_helpButton->setPopup( helpMenu->menu() );

    slotMailAgentChanged(m_mailAgentName->currentItem());
}

SendImagesDialog::~SendImagesDialog()
{
    if ( m_thumbJob ) 
        delete m_thumbJob;

    delete m_about;
}

void SendImagesDialog::readSettings(void)
{
    // Read all settings from configuration file.
    
    TDEConfig config("kipirc");
    config.setGroup("SendImages Settings");

    TQString t = config.readEntry("MailAgentName", "Default");

    // The _old_ Kmail (mind the lowercase 'm') called the default mailer.
    // this is now renamed to 'Default'. It should not interfere with KMail, which
    // is now doing what you expect.
    
    if (t == "Kmail") t = "Default";
    m_mailAgentName->setCurrentText(t);

    m_ThunderbirdBinPath->setURL( config.readEntry("ThunderbirdBinPath", "/usr/bin/mozilla-thunderbird"));

    if (config.readEntry("ImagesChangeProp", "true") == "true")
        m_changeImagesProp->setChecked( true );
    else
        m_changeImagesProp->setChecked( false );

    m_imagesResize->setCurrentItem(config.readNumEntry("ImageResize", 2));  // Medium size used by default.
    m_imageCompression->setValue(config.readNumEntry("ImageCompression", 75));
    m_imagesFormat->setCurrentText(config.readEntry("ImageFormat", "JPEG"));
    m_attachmentlimit->setValue(config.readNumEntry("AttachmentLimit", 10));
        
    if (config.readEntry("AddComments", "true") == "true")
        m_addComments->setChecked( true );
    else
        m_addComments->setChecked( false );

    if (config.readEntry("Comment2ImageName", "true") == "true")
        m_comment2ImageName->setChecked( true );
    else
        m_comment2ImageName->setChecked( false );
}

void SendImagesDialog::writeSettings(void)
{
    // Write all settings in configuration file.

    TDEConfig config("kipirc");
    config.setGroup("SendImages Settings");
    config.writeEntry("MailAgentName", m_mailAgentName->currentText());
    config.writeEntry("ThunderbirdBinPath", m_ThunderbirdBinPath->url());
    config.writeEntry("AddComments", m_addComments->isChecked());
    config.writeEntry("ImagesChangeProp", m_changeImagesProp->isChecked());
    config.writeEntry("ImageResize", m_imagesResize->currentItem());
    config.writeEntry("ImageCompression", m_imageCompression->value());
    config.writeEntry("ImageFormat", m_imagesFormat->currentText());
    config.writeEntry("AttachmentLimit", m_attachmentlimit->value());
    config.writeEntry("Comment2ImageName", m_comment2ImageName->isChecked());
    config.sync();
}

void SendImagesDialog::setupImagesList(void)
{
    TQString whatsThis;

    page_setupImagesList = addPage(i18n("Images"),
                                   i18n("Images to EMail"),
                                   BarIcon("image-x-generic", TDEIcon::SizeMedium));

    TQVBoxLayout *vlay = new TQVBoxLayout( page_setupImagesList, 0, spacingHint() );

    //---------------------------------------------

    m_groupBoxImageList = new TQGroupBox( page_setupImagesList );
    m_groupBoxImageList->setFlat(false);
    TQGridLayout* grid   = new TQGridLayout( m_groupBoxImageList, 2, 2 , 20, 20);

    m_ImagesFilesListBox = new ListImageItems( m_groupBoxImageList, "ListImageItems" );
    TQWhatsThis::add( m_ImagesFilesListBox, i18n( "<p>This is the list of images  to email. "
                                                 "If you want to add some images click on the 'Add Images...' "
                                                 "button or use the drag-and-drop."));
    grid->addMultiCellWidget(m_ImagesFilesListBox, 0, 2, 0, 1);

    KButtonBox* imagesListButtonBox = new KButtonBox( m_groupBoxImageList,TQt::Vertical );
    TQPushButton* m_addImagesButton  = imagesListButtonBox->addButton ( i18n( "&Add ..." ) );
    TQWhatsThis::add( m_addImagesButton, i18n("<p>Add images to the list.") );
    TQPushButton* m_remImagesButton  = imagesListButtonBox->addButton ( i18n( "&Remove" ));
    TQWhatsThis::add( m_remImagesButton, i18n("<p>Remove selected images from the list.") );
    imagesListButtonBox->layout();
    grid->addMultiCellWidget(imagesListButtonBox, 0, 1, 2, 2);

    m_imageLabel = new TQLabel( m_groupBoxImageList );
    m_imageLabel->setFixedHeight( 120 );
    m_imageLabel->setAlignment( TQt::AlignHCenter | TQt::AlignVCenter );
    m_imageLabel->setSizePolicy( TQSizePolicy( TQSizePolicy::Preferred, TQSizePolicy::Preferred ) );
    TQWhatsThis::add( m_imageLabel, i18n( "<p>Preview of the currently selected image on the list." ) );
    grid->addMultiCellWidget(m_imageLabel, 2, 2, 2, 2);

    vlay->addWidget( m_groupBoxImageList );

    //---------------------------------------------

    TQGroupBox * groupBox2 = new TQGroupBox( i18n("Image Description"), page_setupImagesList );
    groupBox2->setColumnLayout(0, TQt::Vertical );
    groupBox2->layout()->setSpacing( 6 );
    groupBox2->layout()->setMargin( 11 );
    TQWhatsThis::add( groupBox2, i18n("<p>The description of the currently selected image on the list.") );

    TQVBoxLayout * groupBox2Layout = new TQVBoxLayout( groupBox2->layout() );
    groupBox2Layout->setAlignment( TQt::AlignTop );

    m_ImageComments = new KSqueezedTextLabel( groupBox2 );
    m_ImageComments->setAlignment( int( TQLabel::WordBreak | TQLabel::AlignVCenter ) );
    groupBox2Layout->addWidget( m_ImageComments );

    m_ImageAlbum = new KSqueezedTextLabel( groupBox2 );
    m_ImageAlbum->setAlignment( int( TQLabel::WordBreak | TQLabel::AlignVCenter ) );
    groupBox2Layout->addWidget( m_ImageAlbum );

    vlay->addWidget( groupBox2 );
    vlay->addStretch(1);

    //---------------------------------------------

    connect(m_addImagesButton, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotImagesFilesButtonAdd()));

    connect(m_remImagesButton, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotImagesFilesButtonRem()));

    connect(m_ImagesFilesListBox, TQ_SIGNAL( currentChanged( TQListBoxItem * ) ),
            this, TQ_SLOT( slotImageSelected( TQListBoxItem * )));

    connect(m_ImagesFilesListBox, TQ_SIGNAL( addedDropItems(TQStringList) ),
            this, TQ_SLOT( slotAddDropItems(TQStringList)));
}

void SendImagesDialog::setImagesList( const KURL::List& Files )
{
    if ( Files.count() == 0 ) return;

    for( KURL::List::ConstIterator it = Files.begin(); it != Files.end(); ++it )
    {
        KIPI::ImageInfo imageInfo = m_interface->info( *it );
        TQString comments = imageInfo.description();
    
        // Check if the new item already exist in the list.
    
        bool findItem = false;
    
        for (uint i = 0 ; i < m_ImagesFilesListBox->count() ; ++i)
        {
            ImageItem *pitem = static_cast<ImageItem*>( m_ImagesFilesListBox->item(i) );
    
            if (pitem->url() == (*it))
                findItem = true;
        }
    
        if (findItem == false)
        {
            ImageItem *item = new ImageItem(m_ImagesFilesListBox,
                                            comments,               // Image comments.
                                            *it                     // Complete url (path & file name).
                                            );
    
            item->setName( (*it).fileName() );
        }
    }

    m_ImagesFilesListBox->setCurrentItem( m_ImagesFilesListBox->count()-1) ;
    slotImageSelected(m_ImagesFilesListBox->item(m_ImagesFilesListBox->currentItem()));
    m_ImagesFilesListBox->centerCurrentItem();
}

void SendImagesDialog::setupEmailOptions(void)
{
    TQString whatsThis;

    page_setupEmailOptions = addPage(i18n("Mail"),
                                     i18n("Mail Options"),
                                     BarIcon("mail_generic", TDEIcon::SizeMedium));

    TQVBoxLayout *vlay = new TQVBoxLayout( page_setupEmailOptions, 0, spacingHint() );

    //---------------------------------------------

    TQHBoxLayout *hlay10  = new TQHBoxLayout( );
    vlay->addLayout( hlay10 );

    TQLabel *m_mailAgentLabel = new TQLabel( i18n("Mail agent:"), page_setupEmailOptions);

    m_mailAgentName = new TQComboBox( false, page_setupEmailOptions );
    m_mailAgentName->insertItem( i18n("Default") );
    m_mailAgentName->insertItem( "Balsa" );
    m_mailAgentName->insertItem( "Claws Mail" );
    m_mailAgentName->insertItem( "Evolution" );
    m_mailAgentName->insertItem( "GmailAgent" );
    m_mailAgentName->insertItem( "KMail" );
    m_mailAgentName->insertItem( "Mozilla" );
    m_mailAgentName->insertItem( "Netscape" );
    m_mailAgentName->insertItem( "Sylpheed" );
    m_mailAgentName->insertItem( "Sylpheed-Claws" );
    m_mailAgentName->insertItem( "Thunderbird" );
    m_mailAgentName->setCurrentText( i18n("Default") );
    TQWhatsThis::add( m_mailAgentName, i18n("<p>Select here your preferred external mail agent program."
                                           "These mail agent versions are supported:<p>"
                                           "<b>Balsa</b>: >= 2.x<p>"
					   "<b>Claws Mail</b>: >= 2.6.1<p>"
                                           "<b>Evolution</b>: >= 1.4<p>"
                                           "<b>GmailAgent</b>: >= 0.2<p>"
                                           "<b>KMail</b>: >= 1.3<p>"
                                           "<b>Mozilla</b>: >= 1.4<p>"
                                           "<b>Netscape</b>: >= 7.x<p>"
                                           "<b>Sylpheed</b>: >= 0.9<p>"
                                           "<b>Sylpheed-Claws</b>: >= 0.9<p>"
                                           "<b>Thunderbird</b>: >= 0.4<p>") );

    hlay10->addWidget( m_mailAgentLabel );
    hlay10->addStretch( 1 );
    hlay10->addWidget( m_mailAgentName );

    connect(m_mailAgentName, TQ_SIGNAL(activated(int)),
            this, TQ_SLOT(slotMailAgentChanged(int)));

    //---------------------------------------------

    m_labelThunderbirdBinPath = new TQLabel(i18n("&Thunderbird binary path:"), page_setupEmailOptions);
    vlay->addWidget( m_labelThunderbirdBinPath );

    m_ThunderbirdBinPath = new KURLRequester( "/usr/bin/thunderbird", page_setupEmailOptions);
    m_labelThunderbirdBinPath->setBuddy( m_ThunderbirdBinPath );
    vlay->addWidget(m_ThunderbirdBinPath);

    connect(m_ThunderbirdBinPath, TQ_SIGNAL(textChanged(const TQString&)),
            this, TQ_SLOT(slotThunderbirdBinPathChanged(const TQString&)));

    TQWhatsThis::add( m_ThunderbirdBinPath, i18n("<p>The path name to the Thunderbird binary program.") );

    //---------------------------------------------

    m_addComments = new TQCheckBox( i18n("Attach a file with caption and tags"), page_setupEmailOptions);
    TQWhatsThis::add( m_addComments, i18n("<p>If you enable this option, all image captions and tags "
                                         "will be added as an attached file.") );
    vlay->addWidget( m_addComments );
    m_comment2ImageName = new TQCheckBox( i18n("Generate new filenames from image captions (if available)"), page_setupEmailOptions);
    vlay->addWidget( m_comment2ImageName );

    //---------------------------------------------

    TQGroupBox * groupBox2 = new TQGroupBox( i18n("Image Properties"), page_setupEmailOptions );
    groupBox2->setColumnLayout(0, TQt::Vertical );
    groupBox2->layout()->setSpacing( 6 );
    groupBox2->layout()->setMargin( 11 );
    TQWhatsThis::add( groupBox2, i18n("<p>The properties of images to send.") );

    TQVBoxLayout * groupBox2Layout = new TQVBoxLayout( groupBox2->layout() );
    groupBox2Layout->setAlignment( TQt::AlignTop );

    m_changeImagesProp = new TQCheckBox(i18n("Adjust image properties"), groupBox2);
    TQWhatsThis::add( m_changeImagesProp, i18n("<p>If you enable this option, "
                     "all images to send can be resized and recompressed.") );
    m_changeImagesProp->setChecked( true );
    groupBox2Layout->addWidget( m_changeImagesProp );

    TQHBoxLayout *hlay12  = new TQHBoxLayout();
    groupBox2Layout->addLayout( hlay12 );

    m_imagesResize = new TQComboBox(false, groupBox2);
    m_imagesResize->insertItem(i18n("Very Small (320 pixels)"));
    m_imagesResize->insertItem(i18n("Small (640 pixels)"));
    m_imagesResize->insertItem(i18n("Medium (800 pixels)"));
    m_imagesResize->insertItem(i18n("Big (1024 pixels)"));
    m_imagesResize->insertItem(i18n("Very Big (1280 pixels)"));
    m_imagesResize->insertItem(i18n("Huge - for printing (1600 pixels)"));
    m_imagesResize->setCurrentText (i18n("Medium (800 pixels)"));
    whatsThis = i18n("<p>Select here the images size to send:<p>"
                     "<b>%1</b>: use this if you have a very slow internet "
                     "connection or if the target mailbox size is very limited.<p>"
                     "<b>%2</b>: use this if you have a slow internet connection "
                     "and if the target mailbox size is limited.<p>"
                     "<b>%3</b>: this is the default value for a medium internet connection "
                     "and a target mailbox size.<p>"
                     "<b>%4</b>: use this if you have a high-speed internet connection "
                     "and if the target mailbox size is not limited.<p>"
                     "<b>%5</b>: use this if you have no size or speed restrictions.<p>"
                     "<b>%6</b>: use this only for printing purpose.<p>")
                     .arg(i18n("very small (320 pixels)"))
                     .arg(i18n("small (640 pixels)"))
                     .arg(i18n("medium (800 pixels)"))
                     .arg(i18n("big (1024 pixels)"))
                     .arg(i18n("very big (1280 pixels)"))
		     .arg(i18n("huge - for printing (1600 pixels)"));
    TQWhatsThis::add( m_imagesResize, whatsThis );

    m_labelImageSize = new TQLabel( i18n("Sent image size:"), groupBox2);
    hlay12->addWidget( m_labelImageSize );
    m_labelImageSize->setBuddy( m_imagesResize );
    hlay12->addStretch( 1 );
    hlay12->addWidget(m_imagesResize);

    //---------------------------------------------

    m_imageCompression = new KIntNumInput(75, groupBox2);
    m_imageCompression->setRange(1, 100, 1, true );
    m_imageCompression->setLabel( i18n("Sent image quality level:") );
    groupBox2Layout->addWidget( m_imageCompression );
    whatsThis = i18n("<p>The new compression value of images to send:<p>");
    whatsThis = whatsThis + i18n("<b>1</b>: very high compression<p>"
                                 "<b>25</b>: high compression<p>"
                                 "<b>50</b>: medium compression<p>"
                                 "<b>75</b>: low compression (default value)<p>"
                                 "<b>100</b>: no compression");

    TQWhatsThis::add( m_imageCompression, whatsThis);

    //---------------------------------------------

    TQHBoxLayout *hlay13  = new TQHBoxLayout();
    groupBox2Layout->addLayout( hlay13 );

    m_imagesFormat = new TQComboBox(false, groupBox2);
    m_imagesFormat->insertItem("JPEG");
    m_imagesFormat->insertItem("PNG");
    m_imagesFormat->setCurrentText ("JPEG");
    whatsThis = i18n("<p>Select here the images files format to send.<p>");
    whatsThis = whatsThis + i18n("<b>JPEG</b>: The Joint Photographic Experts Group's file format "
                "is a good Web file format but it uses lossy compression.<p>"
                "<b>PNG</b>: the Portable Network Graphics format is an extensible file format for "
                "the lossless, portable, well-compressed storage of raster images. PNG provides a "
                "patent-free replacement for GIF and can also replace many common uses of TIFF. "
                "PNG is designed to work well in online viewing applications, such as the World Wide Web, "
                "so it is fully streamable with a progressive display option. Also, PNG can store gamma "
                "and chromaticity data for improved color matching on heterogeneous platforms.");
    TQWhatsThis::add( m_imagesFormat, whatsThis );

    m_labelImageFormat = new TQLabel(i18n("Image file format:"), groupBox2);
    hlay13->addWidget(m_labelImageFormat);
    m_labelImageFormat->setBuddy(m_imagesFormat);
    hlay13->addStretch(1);
    hlay13->addWidget(m_imagesFormat);

    vlay->addWidget(groupBox2);
    vlay->addStretch(1);

    m_attachmentlimit = new KIntNumInput(17, page_setupEmailOptions);
    m_attachmentlimit->setRange(1, 50, 1, true );
    m_attachmentlimit->setLabel( i18n("Maximum Email size limit:"));
    m_attachmentlimit->setSuffix(i18n("MB"));
    vlay->addWidget( m_attachmentlimit );
		    
    //---------------------------------------------

    connect(m_changeImagesProp, TQ_SIGNAL(toggled(bool)),
            m_labelImageSize, TQ_SLOT(setEnabled(bool)));

    connect(m_changeImagesProp, TQ_SIGNAL(toggled(bool)),
            m_imagesResize, TQ_SLOT(setEnabled(bool)));

    connect(m_changeImagesProp, TQ_SIGNAL(toggled(bool)),
            m_imageCompression, TQ_SLOT(setEnabled(bool)));

    connect(m_changeImagesProp, TQ_SIGNAL(toggled(bool)),
            m_labelImageFormat, TQ_SLOT(setEnabled(bool)));

    connect(m_changeImagesProp, TQ_SIGNAL(toggled(bool)),
            m_imagesFormat, TQ_SLOT(setEnabled(bool)));
}

void SendImagesDialog::slotHelp()
{
    TDEApplication::kApplication()->invokeHelp("sendimages", "kipi-plugins");
}

void SendImagesDialog::slotMailAgentChanged(int)
{
    if ( m_mailAgentName->currentText() == "Thunderbird" )
    {
        m_labelThunderbirdBinPath->setEnabled(true);
        m_ThunderbirdBinPath->setEnabled(true);
    }
    else
    {
       m_labelThunderbirdBinPath->setEnabled(false);
       m_ThunderbirdBinPath->setEnabled(false);
    }
}

void SendImagesDialog::slotThunderbirdBinPathChanged(const TQString &url )
{
    if ( m_mailAgentName->currentText() == "Thunderbird" )
       enableButtonOK( !url.isEmpty());
}

void SendImagesDialog::slotAddDropItems(TQStringList filesPath)
{
    setImagesList( KURL::List( filesPath) );
}

void SendImagesDialog::slotImagesFilesButtonAdd( void )
{
    KURL::List urls = KIPI::ImageDialog::getImageURLs( this, m_interface );

    if ( urls.isEmpty() ) return;

    setImagesList(urls);
    setNbItems();
}

void SendImagesDialog::slotImagesFilesButtonRem( void )
{
    for (uint i = 0 ; i < m_ImagesFilesListBox->count() ; ++i)
    {
        if (m_ImagesFilesListBox->isSelected(i))
        {
            m_ImagesFilesListBox->removeItem(i);
            m_ImagesFilesListBox->setCurrentItem(i);
            --i;
        }
    }

    m_ImagesFilesListBox->setSelected(m_ImagesFilesListBox->item(m_ImagesFilesListBox->currentItem()), true);
    slotImageSelected(m_ImagesFilesListBox->item(m_ImagesFilesListBox->currentItem()));
    setNbItems();
}

void SendImagesDialog::slotImageSelected( TQListBoxItem * item )
{
    if ( !item || m_ImagesFilesListBox->count() == 0 )
    {
        m_imageLabel->clear();
        return;
    }

    ImageItem *pitem = static_cast<ImageItem*>( item );

    if ( !pitem ) return;

    m_ImageComments->setText( i18n("Caption: %1").arg(pitem->comments()) );
    m_ImageAlbum->setText( i18n("Album: %1").arg(pitem->album()) );
    m_imageLabel->clear();

    if ( m_thumbJob ) 
        delete m_thumbJob;

    m_thumbJob = TDEIO::filePreview( pitem->url(), m_imageLabel->height() );

    connect(m_thumbJob, TQ_SIGNAL(gotPreview(const KFileItem*, const TQPixmap&)),
            TQ_SLOT(slotGotPreview(const KFileItem*, const TQPixmap&)));

    connect(m_thumbJob, TQ_SIGNAL(failed(const KFileItem*)),
            TQ_SLOT(slotFailedPreview(const KFileItem*)));
}

void SendImagesDialog::slotGotPreview(const KFileItem*, const TQPixmap &pixmap)
{
    m_imageLabel->setPixmap(pixmap);
    m_thumbJob = 0L;
}

void SendImagesDialog::slotFailedPreview(const KFileItem*)
{
    m_thumbJob = 0L;
}

void SendImagesDialog::slotOk()
{
    if ( m_ImagesFilesListBox->count() == 0 )
    {
        KMessageBox::error(this, i18n("You must add some images to send."));
        return;
    }

    if ( m_mailAgentName->currentText() == "Thunderbird" )
    {
        TQFile fileThunderbird(m_ThunderbirdBinPath->url());
    
        if (fileThunderbird.exists() == false)
        {
            KMessageBox::sorry(this, i18n("Thunderbird binary path is not valid. Please check it."));
            return;
        }
    }

    writeSettings();

    for (uint i = 0 ; i < m_ImagesFilesListBox->count() ; i++)
    {
        ImageItem *pitem = static_cast<ImageItem*>( m_ImagesFilesListBox->item(i) );
        m_images2send << pitem->url();
    }

    emit signalAccepted();
    accept();
}

void SendImagesDialog::setNbItems(void)
{
    if ( m_ImagesFilesListBox->count() == 0 ) m_groupBoxImageList->setTitle(i18n("Image List"));
    else
       m_groupBoxImageList->setTitle(i18n("Image List (1 item)", "Image List (%n items)",
                                     m_ImagesFilesListBox->count() ));
}

}  // NameSpace KIPISendimagesPlugin
