/***************************************************************************
 * This file is part of the KDE project
 * copyright (C) 2005 by Sebastian Sauer (mail@dipe.org)
 * copyright (C) 2005 by Tobi Krebs (tobi.krebs@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#ifndef KOMACRO_METAPARAMETER_H
#define KOMACRO_METAPARAMETER_H

#include <tqstring.h>
#include <tqvariant.h>
#include <tqobject.h>
#include <ksharedptr.h>

#include "komacro_export.h"

namespace KoMacro {

	// Forward declarations.
	class Variable;

	/**
	* Class to provide abstract methods for the undocumented 
	* TQt3 TQUObject-API functionality.
	*
	* The design tried to limit future porting to TQt4 by providing a
	* somewhat similar API to the TQt4 TQMeta* stuff.
	*/
	class KOMACRO_EXPORT MetaParameter : public TDEShared
	{

			/**
			* Property to get the type of the variable.
			*/
			TQ_PROPERTY(Type type READ type)

			/**
			* Property to get the type of the variable as string.
			*/
			TQ_PROPERTY(TQString typeName READ typeName)

		public:

			/**
			* List of @a MetaParameter instances.
			*/
			typedef TQValueList<TDESharedPtr <MetaParameter > > List;

			/**
			* Constructor.
			*
			* @param signatureargument The signatures argument
			* that will be used to determinate the arguments
			* type. This could be something like "const TQString&",
			* "int" or "TQMap &lt; TQString, TQVariant &gt; ".
			*/
			explicit MetaParameter(const TQString& signatureargument = TQString());

			/**
			* Destructor.
			*/
			~MetaParameter();

			/**
			* Possible types the @a MetaParameter could provide.
			*/
			enum Type {
				TypeNone = 0, /// None type, the @a MetaParameter is empty.
				TypeVariant, /// The @a MetaParameter is a TQVariant.
				TypeObject /// The @a MetaParameter is a TQObject.
			};

			/**
			* @return the @a MetaParameter::Type this variable has.
			*/
			Type type() const;

			/**
			* @return the @a MetaParameter::Type as string. The typename
			* could be "None", "Variant" or "Object".
			*/
			const TQString typeName() const;

			/**
			* Set the @a MetaParameter::Type this variable is.
			*/
			void setType(Type type);

			/**
			* @return the @a MetaParameter::Type this variable is.
			*/
			TQVariant::Type variantType() const;

			/**
			* Set the @a MetaParameter::Type this variable is.
			*/
			void setVariantType(TQVariant::Type varianttype);

			/**
			* @return true if the passed @a Variable @p variable is
			* valid for this @a MetaParameter . Valid means, that
			* the variable has a castable type.
			*/
			bool validVariable(TDESharedPtr<Variable> variable) const;

		protected:

			/**
			* @internal used method to set the signature argument. Those
			* argument will be used to determinate the arguments type.
			*/
			void setSignatureArgument(const TQString& signatureargument);

		private:
			/// @internal d-pointer class.
			class Private;
			/// @internal d-pointer instance.
			Private* const d;
	};

}

#endif
