/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>
   Copyright (C) 2005-2006 Thorsten Zachmann <zachmann@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <kprinter.h>
#include <tdeaccel.h>
#include <tdeglobalsettings.h>
#include <tqpainter.h>
#include <tqscrollbar.h>
#include <tqpopupmenu.h>
#include <tqcursor.h>
#include <tqfileinfo.h>
#include <tqtextstream.h>
#include <assert.h>
#include <tqtoolbutton.h>
#include <tqtooltip.h>
#include <tqdir.h>
#include <tqclipboard.h>
#include <tqradiobutton.h>
#include <tqdragobject.h>
#include <tqfile.h>

#include "KPrPage.h"
#include "KPrBackDia.h"
#include "autoformEdit/AFChoose.h"
#include "KPrPropertyEditor.h"
#include "KPrPgConfDia.h"
#include "KPrEffectDia.h"
#include "KPrRotationDialogImpl.h"
#include "KPrShadowDialogImpl.h"
#include "KPrImageEffectDia.h"
#include "KPrBackground.h"
#include "KPrPenStyleWidget.h"
#include "KPrObjectProperties.h"

#include <KoAutoFormat.h>

#include "KPrSlideTransitionDia.h"

#include "KPrPresDurationDia.h"
#include "KPrPartObject.h"
#include "KPrSideBar.h"
#include "KPrNoteBar.h"
#include "insertpagedia.h"
#include "KPrCustomSlideShowDia.h"

#include <KoPictureFilePreview.h>
#include <KoCreateStyleDia.h>

#include <dcopclient.h>
#include <tdefiledialog.h>
#include <tdemessagebox.h>
#include <kstdaction.h>
#include <tdeapplication.h>
#include <tdeio/netaccess.h>

#include "KPrView.h"
#include "KPrDocument.h"
#include "KPrWebPresentation.h"
#include "KPrMSPresentationSetup.h"
#include "KPrTextObject.h"

#include <tdelocale.h>
#include <kcolordialog.h>
#include <tdeconfig.h>
#include <tdefontdialog.h>
#include <tdeglobal.h>
#include <kimageio.h>
#include <tdeparts/event.h>
#include <kdebug.h>
#include <tdetempfile.h>
#include <kcolorbutton.h>
#include <kprocess.h>

#include <KoMainWindow.h>
#include <KoPartSelectDia.h>
#include <KoQueryTrader.h>
#include <KoPageLayoutDia.h>
#include <KoRuler.h>
#include <KoTemplateCreateDia.h>
#include <kcoloractions.h>
#include <tkcoloractions.h>
#include <tdeaction.h>
#include <tqspinbox.h>
#include <tqcombobox.h>
#include <KoPartSelectAction.h>
#include <KoTextZoomHandler.h>
#include <Kolinestyleaction.h>
#include <Kolinewidthaction.h>

#include <stdlib.h>
#include <signal.h>

#include <kstandarddirs.h>

#include "KPrViewIface.h"
#include "KPrConfig.h"

#include <KoTextParag.h>
#include <KoTextObject.h>
#include "KPrCommand.h"
#include <KoFontDia.h>
#include <KoCharSelectDia.h>
#include <KoInsertLink.h>
#include <KoAutoFormatDia.h>
#include <KoParagCounter.h>
#include <KoParagDia.h>
#include <KoVariable.h>
#include <KoCustomVariablesDia.h>
#include <KoGuideLineDia.h>
#include <KoCompletionDia.h>

#include <tdespell.h>
#include <kstatusbar.h>
#include "KPrTextDocument.h"

#include <KoChangeCaseDia.h>
#include <tqregexp.h>

#include <KoSearchDia.h>
#include "KPrFindReplace.h"
#include "KPrVariableCollection.h"
#include "KPrCanvas.h"
#include <tqpaintdevicemetrics.h>
#include <KoStyleCollection.h>
#include "KPrStyleManager.h"
#include "KPrPixmapObject.h"
#include <KoCommentDia.h>

#include "KPrDuplicateObjDia.h"
#include <tdestdaccel.h>
#include <KoDocumentInfo.h>
#include <tdeaccelgen.h>
#include "KPrImportStyleDia.h"
#include <kurldrag.h>
#include <config.h>
#include <KoStore.h>
#include <KoStoreDrag.h>

#include <tdespell2/broker.h>
#include <tdespell2/defaultdictionary.h>
#include <tdespell2/dialog.h>
#include "KoSpell.h"
#include "KPrPrinterDlg.h"
using namespace KSpell2;

#define COPYOASISFORMAT 0

static const char * const pageup_xpm[] = {
    "    14    14        2            1",
    ". c #000000",
    "# c none",
    "##############",
    "##############",
    "######..######",
    "#####....#####",
    "####......####",
    "###........###",
    "##############",
    "######..######",
    "#####....#####",
    "####......####",
    "###........###",
    "##############",
    "##############",
    "##############"
};

static const char * const pagedown_xpm[] = {
    "    14    14        2            1",
    ". c #000000",
    "# c none",
    "##############",
    "##############",
    "##############",
    "###........###",
    "####......####",
    "#####....#####",
    "######..######",
    "##############",
    "###........###",
    "####......####",
    "#####....#####",
    "######..######",
    "##############",
    "##############"
};

#ifdef HAVE_DPMS
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xmd.h>
#include <X11/extensions/dpms.h>
#include <fixx11h.h>

extern "C" {
#ifndef HAVE_DPMSCAPABLE_PROTO
Bool DPMSCapable( Display * );
#endif
}

#if defined(XIMStringConversionRetrival) || defined (__sun)
extern "C" {
#endif
    Bool DPMSQueryExtension(Display *, int *, int *);
    Status DPMSEnable(Display *);
    Status DPMSDisable(Display *);
    Bool DPMSGetTimeouts(Display *, CARD16 *, CARD16 *, CARD16 *);
#if defined(XIMStringConversionRetrival) || defined (__sun)
}
#endif
#endif

static void enableDPMS(bool enable)
{
#ifdef HAVE_DPMS
    Display *display = tqt_xdisplay();

    int dummy;
    bool hasDPMS = DPMSQueryExtension(display, &dummy, &dummy);
    if (hasDPMS && DPMSCapable(display)) {
        if (enable) {
            kdDebug() << "Enabling DPMS" << endl;
            DPMSEnable(display);
        } else {
            kdDebug() << "Disabling DPMS" << endl;
            DPMSDisable(display);
        }
    } else
        tqWarning("Server has no DPMS extension");

    XFlush(display);
#else
    Q_UNUSED(enable); /* keep gcc silent */
#endif
}

static bool isDPMSEnabled()
{
    bool result = false;
#ifdef HAVE_DPMS
    int event_base;
    int error_base;
    CARD16 x_standby;
    CARD16 x_suspend;
    CARD16 x_off;
    Display *display = tqt_xdisplay();
    if (DPMSQueryExtension(display, &event_base, &error_base))
        if (DPMSCapable(display))
            result = (DPMSGetTimeouts(display, &x_standby, &x_suspend, &x_off));
#endif
    return result;
}

KPrView::KPrView( KPrDocument* _doc, TQWidget *_parent, const char *_name )
    : KoView( _doc, _parent, _name )
{

    setInstance( KPrFactory::global() );
    if ( !_doc->isReadWrite() )
        setXMLFile( "kpresenter_readonly.rc" );
    else
        setXMLFile( "kpresenter.rc" );

    dcop = 0;
    dcopObject(); // build it

    m_bDisplayFieldCode=false;
    // init
    afChoose = 0;
    m_propertyEditor = 0;
    pgConfDia = 0;
    rotateDia = 0;
    shadowDia = 0;
    imageEffectDia = 0;
    presDurationDia = 0;
    v_ruler = 0;
    h_ruler = 0;
    pen = KoPen( black, 1.0, TQt::SolidLine );
    brush = TQBrush( white, SolidPattern );
    lineBegin = L_NORMAL;
    lineEnd = L_NORMAL;
    gColor1 = red;
    gColor2 = green;
    gType = BCT_GHORZ;
    gUnbalanced = false;
    gXFactor = 100;
    gYFactor = 100;
    fillType = FT_BRUSH;
    pieType = PT_PIE;
    pieLength = 90 * 16;
    pieAngle = 45 * 16;
    setMouseTracking( true );
    m_bShowGUI = true;
    presStarted = false;
    m_autoPresRestart = false;
    m_autoPresStop = false;
    rndX = 0;
    rndY = 0;
    m_editMaster = false;
    allowWebPres = true;
    allowMSPres = true;
    currPg = 0;
    sidebar = 0;
    notebar = 0;
    splitter = 0;
    pageBase = 0;
    m_canvas = 0L;
    m_spell.kospell = 0;
    m_spell.textIterator = 0L;
    m_spell.macroCmdSpellCheck = 0L;
    m_spell.dlg = 0;
    m_broker = Broker::openBroker( TDESharedConfig::openConfig( "kpresenterrc" ) );

    m_autoPresTimerConnected = false;
    m_actionList.setAutoDelete( true );
    checkConcavePolygon = false;
    cornersValue = 3;
    sharpnessValue = 0;
    tbAlign = TQt::AlignLeft;
    tbFont = font();
    tbColor = black;

    mirrorType = PM_NORMAL;
    depth = 0;
    swapRGB = false;
    grayscal = false;
    bright = 0;

    m_currentLineTool = LtLine;
    m_currentShapeTool = StRectangle;
    m_currentClosedLineTool = CltFreehand;

    m_searchEntry = 0L;
    m_replaceEntry = 0L;
    m_findReplace = 0L;
    m_switchPage=-1;

    m_fontDlg=0L;
    m_paragDlg=0L;
    m_pKPresenterDoc = _doc;

    createGUI();

    setKeyCompression( true );
    m_specialCharDlg=0L;

    if ( shell() )
        changeNbOfRecentFiles( m_pKPresenterDoc->maxRecentFiles() );

    connect(this, TQ_SIGNAL(embeddImage(const TQString &)), TQ_SLOT(insertPicture(const TQString &)));
    connect( m_pKPresenterDoc, TQ_SIGNAL( sig_refreshMenuCustomVariable()),
             this, TQ_SLOT( refreshCustomMenu()));

    // Cut and copy are directly connected to the selectionChanged signal
    if ( m_pKPresenterDoc->isReadWrite() )
        connect( m_canvas, TQ_SIGNAL(selectionChanged(bool)),
                 this, TQ_SLOT(slotChangeCutState(bool )/*setEnabled(bool)*/) );
    else
        actionEditCut->setEnabled( false );

    connect( m_canvas, TQ_SIGNAL(selectionChanged(bool)),
             actionEditCopy, TQ_SLOT(setEnabled(bool)) );

    connect (m_canvas, TQ_SIGNAL(selectionChanged(bool)),
             actionChangeCase, TQ_SLOT(setEnabled(bool)));
    //connect (m_canvas, TQ_SIGNAL(selectionChanged(bool)),
    //         actionCreateStyleFromSelection, TQ_SLOT(setEnabled(bool)));

    connect( m_canvas, TQ_SIGNAL( currentObjectEditChanged() ), this,  TQ_SLOT( slotObjectEditChanged()));

    connect( h_ruler, TQ_SIGNAL( tabListChanged( const KoTabulatorList & ) ), this,
             TQ_SLOT( tabListChanged( const KoTabulatorList & ) ) );

    //statusbar stuff
    connect( m_pKPresenterDoc, TQ_SIGNAL( pageNumChanged() ), this, TQ_SLOT( pageNumChanged()) );
    connect( this, TQ_SIGNAL( currentPageChanged(int) ), this, TQ_SLOT( pageNumChanged()) );
    connect( m_canvas, TQ_SIGNAL( objectSizeChanged() ), this, TQ_SLOT( updateObjectStatusBarItem() ));
    connect( m_canvas, TQ_SIGNAL( objectSelectedChanged() ), this, TQ_SLOT( updateObjectStatusBarItem() ));
    connect (m_pKPresenterDoc, TQ_SIGNAL(sig_updateRuler()),this, TQ_SLOT( slotUpdateRuler()));
    connect (m_pKPresenterDoc, TQ_SIGNAL(sig_updateRuler()),this, TQ_SLOT( slotUpdateScrollBarRanges()));
    connect (m_pKPresenterDoc, TQ_SIGNAL(sig_updateMenuBar()),this, TQ_SLOT(updateSideBarMenu()));
    connect (m_pKPresenterDoc, TQ_SIGNAL(unitChanged(KoUnit::Unit)),this, TQ_SLOT(slotUnitChanged(KoUnit::Unit)));
    connect( m_pKPresenterDoc, TQ_SIGNAL( modified( bool ) ), this, TQ_SLOT( documentModified( bool )) );
    connect( m_pKPresenterDoc, TQ_SIGNAL(completed()), this, TQ_SLOT(loadingFinished()));

    KStatusBar * sb = statusBar();
    m_sbPageLabel = 0L;
    m_sbObjectLabel = 0L;
    m_sbModifiedLabel = 0;
    m_sbZoomLabel = 0;
    m_sbUnitLabel = 0;

    if ( sb ) // No statusbar in e.g. konqueror
    {
        m_sbPageLabel = new KStatusBarLabel( TQString(), 0, sb );
        m_sbPageLabel->setAlignment( AlignLeft | AlignVCenter );
        addStatusBarItem( m_sbPageLabel, 0 );

        m_sbModifiedLabel = new KStatusBarLabel( "   ", 0, sb );
        m_sbModifiedLabel->setAlignment( AlignLeft | AlignVCenter );
        addStatusBarItem( m_sbModifiedLabel, 0 );

        m_sbObjectLabel = new KStatusBarLabel( TQString(), 0, sb );
        m_sbObjectLabel->setAlignment( AlignLeft | AlignVCenter );
        addStatusBarItem( m_sbObjectLabel, 1 );

        m_sbZoomLabel = new KStatusBarLabel( ' ' + TQString::number( m_pKPresenterDoc->zoomHandler()->zoom() ) + "% ", 0, sb );
        m_sbZoomLabel->setAlignment( AlignHCenter | AlignVCenter );
        addStatusBarItem( m_sbZoomLabel, 0 );

        m_sbUnitLabel = new KStatusBarLabel( ' ' + KoUnit::unitDescription( m_pKPresenterDoc->unit() ) + ' ', 0, sb );
        m_sbUnitLabel->setAlignment( AlignHCenter | AlignVCenter );
        addStatusBarItem( m_sbUnitLabel, 0 );
    }

    //when kword is embedded into konqueror apply a zoom=100
    //in konqueror we can't change zoom -- ### TODO ?
    if(!m_pKPresenterDoc->isReadWrite())
    {
        setZoom( 100, true );
        slotUpdateRuler();
        initGui();
        m_pKPresenterDoc->updateZoomRuler();
    }


    setAcceptDrops( true );
}

DCOPObject* KPrView::dcopObject()
{
    if ( !dcop )
        dcop = new KPrViewIface( this );

    return dcop;
}

KPrView::~KPrView()
{
    delete m_findReplace;
    m_findReplace = 0L;
    if(sidebar) {
        TDEConfig *config=TDEGlobal::config();
        config->setGroup("Global");
        config->writeEntry("Sidebar", sidebar->isVisible());
    }
    if(notebar) {
        TDEConfig *config=TDEGlobal::config();
        config->setGroup("Global");
        config->writeEntry("Notebar", notebar->isVisible());
    }
    clearSpellChecker();

    delete rb_lbegin;
    delete rb_lend;
    delete dcop;

    delete m_sbPageLabel;
    delete m_sbObjectLabel;
    delete notebar;
    delete m_searchEntry;
    m_searchEntry = 0L;
    delete m_replaceEntry;
    m_replaceEntry = 0L;
    delete m_propertyEditor;
    delete pgConfDia;
    delete rotateDia;
    delete shadowDia;
    delete afChoose;
    delete m_fontDlg;
    delete m_paragDlg;
    delete m_arrangeObjectsPopup;

    if ( m_specialCharDlg )
        m_specialCharDlg->closeDialog(); // will call slotSpecialCharDlgClosed
    delete m_specialCharDlg;
}

void KPrView::setupPrinter( KPrinter &prt )
{
    deSelectAllObjects();
    m_pKPresenterDoc->recalcVariables( VT_TIME );
    m_pKPresenterDoc->recalcVariables( VT_DATE );
    prt.setMinMax( 1, m_pKPresenterDoc->getPageNums() );
    prt.setFromTo( 1, m_pKPresenterDoc->getPageNums() );
    prt.setOption( "kde-range", m_pKPresenterDoc->selectedForPrinting() );
    prt.setPageSelection(KPrinter::ApplicationSide);
    KoPageLayout layout = m_pKPresenterDoc->pageLayout();
    prt.setOption( "kde-margin-top", TQString::number(layout.ptTop) );
    prt.setOption( "kde-margin-bottom", TQString::number(layout.ptBottom) );
    prt.setOption( "kde-margin-left", TQString::number(layout.ptLeft) );
    prt.setOption( "kde-margin-right", TQString::number(layout.ptRight) );
    KoFormat pageFormat = layout.format;
    prt.setPageSize( static_cast<KPrinter::PageSize>( KoPageFormat::printerPageSize( pageFormat ) ) );

    if ( m_pKPresenterDoc->pageLayout().orientation == PG_LANDSCAPE || pageFormat == PG_SCREEN )
        prt.setOrientation( KPrinter::Landscape );
    else
        prt.setOrientation( KPrinter::Portrait );
    if ( !prt.previewOnly() )
      prt.addDialogPage( new KPrPrinterDlg( 0, "KPresenter page" ) );
}

void KPrView::unZoomDocument(int &dpiX,int &dpiY)
{
    // ### HACK: disable zooming-when-printing if embedded parts are used.
    // No koffice app supports zooming in paintContent currently.
    // Disable in ALL cases now
    bool doZoom=false;
    dpiX = doZoom ? 300 : KoGlobal::dpiX();
    dpiY = doZoom ? 300 : KoGlobal::dpiY();
    zoomHandler()->setZoomAndResolution( 100, dpiX, dpiY );
    m_pKPresenterDoc->newZoomAndResolution( false, true /* for printing*/ );
}

void KPrView::zoomDocument(int zoom)
{
    zoomHandler()->setZoomAndResolution( zoom, KoGlobal::dpiX(), KoGlobal::dpiY() );
    m_pKPresenterDoc->newZoomAndResolution( false, false );
    updateRuler();
}

void KPrView::print( KPrinter &prt )
{
    float left_margin = 0.0;
    float top_margin = 0.0;
    int dpiX=0;
    int dpiY=0;
    int oldZoom = zoomHandler()->zoom();
    bool displayFieldCode = m_pKPresenterDoc->getVariableCollection()->variableSetting()->displayFieldCode();
    if ( displayFieldCode )
    {
        m_pKPresenterDoc->getVariableCollection()->variableSetting()->setDisplayFieldCode(false);
        m_pKPresenterDoc->recalcVariables( VT_ALL );
    }

    TQPaintDeviceMetrics metrics( &prt );
    unZoomDocument(dpiX,dpiY);
    if ( m_pKPresenterDoc->pageLayout().format == PG_SCREEN )
    {
        left_margin = 28.5;
        top_margin = 15.0;
    }

    TQPainter painter;
    painter.begin( &prt );
    TQRect rect = m_pKPresenterDoc->pageList().at( 0 )->getZoomPageRect();
    double zoom = TQMIN( double( metrics.width() ) / double( rect.width() ),
                        double( metrics.height() ) / double( rect.height() ) );
    double newZoom = zoom * m_pKPresenterDoc->zoomHandler()->zoom();
    kdDebug(33001) << "KPrView::print newZoom = " << newZoom << endl;
    setZoom( int( newZoom ), false );
    TQRect paintingRect = m_pKPresenterDoc->pageList().at( 0 )->getZoomPageRect();

    m_canvas->print( &painter, &prt, left_margin, top_margin );
    painter.end();

    zoomDocument(oldZoom);
    if ( displayFieldCode )
    {
        m_pKPresenterDoc->getVariableCollection()->variableSetting()->setDisplayFieldCode(true);
        m_pKPresenterDoc->recalcVariables( VT_ALL );
    }

    m_canvas->repaint();
    kdDebug(33001) << "KPrView::print zoom&res reset" << endl;
    m_pKPresenterDoc->getVariableCollection()->variableSetting()->setLastPrintingDate(TQDateTime::currentDateTime());
    m_pKPresenterDoc->recalcVariables( VT_DATE );
}

void KPrView::editCut()
{
    if ( !m_canvas->currentTextObjectView() ) {
        m_canvas->setToolEditMode( TEM_MOUSE );
#if COPYOASISFORMAT
        m_canvas->copyOasisObjs();
#else
        m_canvas->copyObjs();
#endif
        m_canvas->deleteObjs();
    } else {
        if ( !m_canvas->currentTextObjectView()->kpTextObject()->isProtectContent())
            m_canvas->currentTextObjectView()->cut();
    }
}

void KPrView::editCopy()
{
    if ( !m_canvas->currentTextObjectView() ) {
        m_canvas->setToolEditMode( TEM_MOUSE );
#if COPYOASISFORMAT
        m_canvas->copyOasisObjs();
#else
        m_canvas->copyObjs();
#endif
    }
    else
        m_canvas->currentTextObjectView()->copy();
}

void KPrView::editPaste()
{
    if ( !m_canvas->currentTextObjectView() ) {
        m_canvas->setToolEditMode( TEM_MOUSE );
        deSelectAllObjects();
        TQMimeSource *data = TQApplication::clipboard()->data();
        if ( data->provides( "text/uri-list" ) )
        {
            m_pKPresenterDoc->pastePage( data, currPg );
            setRanges();
            skipToPage( currPg );
            updateSideBarMenu();
        }
        else if ( data->provides( KoStoreDrag::mimeType("application/x-kpresenter" ) ))
        {
            // TODO: it would be nice to have no offset when pasting onto a different page...
            m_canvas->activePage()->pasteObjs(
                data->encodedData(KoStoreDrag::mimeType("application/x-kpresenter")),
                1, 0.0, 0.0, 0.0, 20.0, 20.0);

            m_canvas->setMouseSelectedObject(true);
            emit objectSelectedChanged();
        }
#if COPYOASISFORMAT
        else if ( data->provides( KoStoreDrag::mimeType("application/vnd.oasis.opendocument.presentation" ) ))
        {
            //TODO : fix me !!
            // TODO: it would be nice to have no offset when pasting onto a different page...
            //m_canvas->activePage()->pasteObjs(
            //data->encodedData(KoStoreDrag::mimeType("application/vnd.oasis.opendocument.presentation")),
            //1, 0.0, 0.0, 0.0, 20.0, 20.0);

            m_canvas->setMouseSelectedObject(true);
            emit objectSelectedChanged();
        }
#endif
        else if (TQImageDrag::canDecode (data)) {
            m_canvas->dropImage( data );
        }
    } else {
        if ( !m_canvas->currentTextObjectView()->kpTextObject()->isProtectContent())
            m_canvas->currentTextObjectView()->paste();
    }
}

void KPrView::editDelete()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->deleteObjs();
}

void KPrView::editSelectAll()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( !edit ) {
        m_canvas->setToolEditMode( TEM_MOUSE );
        m_canvas->selectAllObj();
    }
    else
        edit->selectAll();
}

void KPrView::editDeSelectAll()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( !edit ) {
        m_canvas->setToolEditMode( TEM_MOUSE );
        deSelectAllObjects();
    }
    else
        edit->selectAll(false);
}

void KPrView::editCopyPage()
{
    m_pKPresenterDoc->copyPageToClipboard( currPg );
}

void KPrView::editDuplicatePage()
{
    m_pKPresenterDoc->copyPage( currPg );
    setRanges();
}

void KPrView::updateSideBarMenu()
{
    actionEditDelPage->setEnabled( m_pKPresenterDoc->getPageNums() > 1 );
}

void KPrView::editDelPage()
{
    if ( KMessageBox::warningContinueCancel( this,
                                     i18n( "Do you want to remove the current slide?"),i18n("Remove Slide"),KGuiItem(i18n("&Delete"),"edit-delete") )
         != KMessageBox::Continue )
        return;
    m_canvas->exitEditMode();
    m_pKPresenterDoc->deletePage( currPg );
    setRanges();
}

void KPrView::insertPage()
{
    InsertPageDia dia( this, 0, true );
#if COPYOASISFORMAT
    TQString templ = locateLocal( "data", "kpresenter/default.oop" );
#else
    TQString templ = locateLocal( "data", "kpresenter/default.kpr" );
#endif
    if ( !TQFile::exists( templ ) ) {
        dia.radioDifferent->setChecked( true );
        dia.radioDefault->setEnabled( false );
    }
    if ( dia.exec() != TQDialog::Accepted )
        return;

    if (dia.radioCurrentDefault->isChecked())
    {
#if COPYOASISFORMAT
        TQString file = locateLocal( "data", "kpresenter/default.oop" );
        m_pKPresenterDoc->savePage( file, currPg, true /*ignore stickies*/ );
#else
        TQString file = locateLocal( "data", "kpresenter/default.kpr" );
        m_pKPresenterDoc->savePage( file, currPg, true /*ignore stickies*/ );
#endif
    }

    InsertPos pos = (InsertPos)dia.locationCombo->currentItem();
    m_pKPresenterDoc->insertNewPage( i18n("Insert new slide"), currPg, pos,
                                     dia.radioDifferent->isChecked(), TQString() );
    setRanges();
}

void KPrView::insertPicture()
{
    m_canvas->setToolEditMode( INS_PICTURE );
    deSelectAllObjects();

    TQStringList mimetypes;
    mimetypes += KImageIO::mimeTypes( KImageIO::Reading );
    mimetypes += KoPictureFilePreview::clipartMimeTypes();

    KFileDialog fd( m_pKPresenterDoc->picturePath(), TQString(), this, 0, true );
    fd.setCaption( i18n( "Insert Picture" ) );
    fd.setMimeFilter( mimetypes );
    fd.setPreviewWidget( new KoPictureFilePreview( &fd ) );

    KURL url;
    if ( fd.exec() == TQDialog::Accepted )
        url = fd.selectedURL();

    if( url.isEmpty() || !url.isValid() )
    {
        m_canvas->setToolEditMode( TEM_MOUSE, false );
        return;
    }
    TQString file;
    if ( !TDEIO::NetAccess::download( url, file, this ) )
    {
        m_canvas->setToolEditMode( TEM_MOUSE, false );
        return;
    }

    if ( !file.isEmpty() )
        m_canvas->activePage()->setInsPictureFile( file );
}

void KPrView::insertPicture(const TQString &file)
{
    m_canvas->setToolEditMode( INS_PICTURE );
    deSelectAllObjects();

    if ( !file.isEmpty() )
        m_canvas->activePage()->setInsPictureFile( file );
}

void KPrView::savePicture()
{
    m_canvas->savePicture();
}

void KPrView::savePicture( const TQString& oldName, KoPicture& picture)
{
    TQString oldFile(oldName);
    KURL url;
    url.setPath( oldFile );
    if (!TQDir(url.directory()).exists())
        oldFile = url.fileName();

    TQString mimetype=picture.getMimeType();
    kdDebug(33001) << "Picture has mime type: " << mimetype << endl;
    TQStringList mimetypes;
    mimetypes << mimetype;

    KFileDialog fd( oldFile, TQString(), this, 0, true );
    fd.setMimeFilter( mimetypes );
    fd.setCaption(i18n("Save Picture"));
    fd.setOperationMode(KFileDialog::Saving);
    if ( fd.exec() == TQDialog::Accepted )
    {
        url = fd.selectedURL();
        if ( url.isValid() )
            {
                if ( url.isLocalFile() )
                {
                    TQFile file( url.path() );
                    if ( file.open( IO_ReadWrite ) )
                    {
                        picture.save( &file );
                        file.close();
                    }
                    else
                    {
                        KMessageBox::error(this,
                                   i18n("Error during saving: could not open '%1' for writing.").arg ( url.path() ),
                                   i18n("Save Picture"));
                    }
                }
                else
                {
                    KTempFile tempFile;
                    tempFile.setAutoDelete( true );
                    if ( tempFile.status() == 0 )
                    {
                        TQFile file( tempFile.name() );
                        if ( file.open( IO_ReadWrite ) )
                        {
                            picture.save( &file );
                            file.close();
                            if ( !TDEIO::NetAccess::upload( tempFile.name(), url, this ) )
                            {
                                KMessageBox::sorry( this, i18n(
                                   "Unable to save the file to '%1'. %2.").arg( url.prettyURL() ).arg( TDEIO::NetAccess::lastErrorString() ),
                                   i18n("Save Failed") );
                            }
                        }
                        else
                            KMessageBox::error(this,
                                   i18n("Error during saving: could not open '%1' temporary file for writing.").arg ( file.name() ),
                                   i18n("Save Picture"));
                    }
                    else
                        KMessageBox::sorry( this, i18n(
                            "Error during saving: could not create temporary file: %1.").arg( strerror( tempFile.status() ) ),
                            i18n("Save Picture") );
                }
            }
    }
}

void KPrView::savePicture( KPrPixmapObject* obj )
{
    TQString oldFile=obj->getFileName();
    KoPicture picture(obj->picture());
    savePicture(oldFile, picture);
}

void KPrView::toolsMouse()
{
    if ( actionToolsMouse->isChecked() )
        m_canvas->setToolEditMode( TEM_MOUSE, false );
    else
        actionToolsMouse->setChecked(true);
    //deSelectAllObjects();
}

void KPrView::toolsRotate()
{
    if ( actionToolsRotate->isChecked() )
        m_canvas->setToolEditMode( TEM_ROTATE, false );
    else
        actionToolsRotate->setChecked(true);
}

void KPrView::toolsZoom()
{
    if ( actionToolsZoom->isChecked() )
        m_canvas->setToolEditMode( TEM_ZOOM, false );
    else
        actionToolsZoom->setChecked(true);
}

void KPrView::toolsLinePopup()
{
    switch (m_currentLineTool)
    {
    case LtLine:
        actionToolsLine->activate();
        break;
    case LtFreehand:
        actionToolsFreehand->activate();
        break;
    case LtPolyline:
        actionToolsPolyline->activate();
        break;
    case LtQuadricBezier:
        actionToolsQuadricBezierCurve->activate();
        break;
    case LtCubicBezier:
        actionToolsCubicBezierCurve->activate();
        break;
    }
}

void KPrView::toolsLine()
{
    if ( actionToolsLine->isChecked() )
    {
        m_canvas->setToolEditMode( INS_LINE, false );
        deSelectAllObjects();
        m_currentLineTool = LtLine;
        actionToolsLinePopup->setIcon("line");
    }
    else
        actionToolsLine->setChecked(true);
}

void KPrView::toolsShapePopup()
{
    switch (m_currentShapeTool)
    {
    case StRectangle:
        actionToolsRectangle->activate();
        break;
    case StCircle:
        actionToolsCircleOrEllipse->activate();
        break;
    case StPie:
        actionToolsPie->activate();
        break;
    case StPolygon:
        actionToolsConvexOrConcavePolygon->activate();
        break;
    }
}

void KPrView::toolsRectangle()
{
    if ( actionToolsRectangle->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( INS_RECT, false );
        m_currentShapeTool = StRectangle;
        actionToolsShapePopup->setIcon("rectangle");
    }
    else
        actionToolsRectangle->setChecked(true);
}

void KPrView::toolsCircleOrEllipse()
{
    if ( actionToolsCircleOrEllipse->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( INS_ELLIPSE, false );
        m_currentShapeTool = StCircle;
        actionToolsShapePopup->setIcon("circle");
    }
    else
        actionToolsCircleOrEllipse->setChecked(true);
}

void KPrView::toolsPie()
{
    if ( actionToolsPie->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( INS_PIE, false );
        m_currentShapeTool = StPie;
        actionToolsShapePopup->setIcon("pie");
    }
    else
        actionToolsPie->setChecked(true);
}

void KPrView::toolsDiagramm()
{
    if ( actionToolsDiagramm->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( INS_DIAGRAMM, false );

        KoDocumentEntry entry = KoDocumentEntry::queryByMimeType( "application/x-kchart" );
        if (entry.isEmpty())
        {
            KMessageBox::sorry( this, i18n( "No chart component registered" ) );
            m_canvas->setToolEditMode( TEM_MOUSE );
        }
        else
            m_canvas->setPartEntry( entry );
    }
    else
        actionToolsDiagramm->setChecked(true);
}

void KPrView::toolsTable()
{
    if ( actionToolsTable->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( INS_TABLE, false );

        KoDocumentEntry entry = KoDocumentEntry::queryByMimeType( "application/x-kspread" );
        if (entry.isEmpty())
        {
            KMessageBox::sorry( this, i18n( "No table component registered" ) );
            m_canvas->setToolEditMode( TEM_MOUSE );
        }
        else
            m_canvas->setPartEntry( entry );
    }
    else
        actionToolsTable->setChecked(true);
}

void KPrView::toolsFormula()
{
    if ( actionToolsFormula->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( INS_FORMULA, false );

        KoDocumentEntry entry = KoDocumentEntry::queryByMimeType( "application/x-kformula" );
        if (entry.isEmpty())
        {
            KMessageBox::sorry( this, i18n( "No formula component registered" ) );
            m_canvas->setToolEditMode( TEM_MOUSE );
        }
        else
            m_canvas->setPartEntry( entry );
    }
    else
        actionToolsFormula->setChecked(true);
}

void KPrView::toolsText()
{
    if ( actionToolsText->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( INS_TEXT, false );
    }
    else
        actionToolsText->setChecked(true);
}

void KPrView::toolsAutoform()
{
    if ( actionToolsAutoform->isChecked() )
    {
        deSelectAllObjects();
        m_canvas->setToolEditMode( TEM_MOUSE, false );
        if ( afChoose ) {
            delete afChoose;
            afChoose = 0;
        }
        afChoose = new AFChoose( this, i18n( "Autoform-Choose" ) );
        afChoose->resize( 400, 300 );
        afChoose->setCaption( i18n( "Insert Autoform" ) );

        TQObject::connect( afChoose, TQ_SIGNAL( formChosen( const TQString & ) ),
                          this, TQ_SLOT( afChooseOk( const TQString & ) ) );
        TQObject::connect( afChoose, TQ_SIGNAL( afchooseCanceled()),
                          this, TQ_SLOT(slotAfchooseCanceled()));
        afChoose->exec();

        TQObject::disconnect( afChoose, TQ_SIGNAL( formChosen( const TQString & ) ),
                             this, TQ_SLOT( afChooseOk( const TQString & ) ) );
        delete afChoose;
        afChoose = 0;
    }
    else
        actionToolsAutoform->setChecked(true);
}

void KPrView::toolsObject()
{
    KoDocumentEntry pe = actionToolsObject->documentEntry();
    if ( pe.isEmpty() ) {
        m_canvas->setToolEditMode( TEM_MOUSE );
        return;
    }

    m_canvas->setToolEditMode( INS_OBJECT );
    m_canvas->setPartEntry( pe );
}

void KPrView::toolsFreehand()
{
    if ( actionToolsFreehand->isChecked() ) {
        m_canvas->setToolEditMode( INS_FREEHAND, false );
        deSelectAllObjects();
        m_currentLineTool = LtFreehand;
        actionToolsLinePopup->setIcon("freehand");
    }
    else
        actionToolsFreehand->setChecked(true);
}

void KPrView::toolsPolyline()
{
    if ( actionToolsPolyline->isChecked() ) {
        m_canvas->setToolEditMode( INS_POLYLINE, false );
        deSelectAllObjects();
        m_currentLineTool = LtPolyline;
        actionToolsLinePopup->setIcon("polyline");
    }
    else
        actionToolsPolyline->setChecked(true);
}

void KPrView::toolsQuadricBezierCurve()
{
    if ( actionToolsQuadricBezierCurve->isChecked() ) {
        m_canvas->setToolEditMode( INS_QUADRICBEZIERCURVE, false );
        deSelectAllObjects();
        m_currentLineTool = LtQuadricBezier;
        actionToolsLinePopup->setIcon("quadricbeziercurve");
    }
    else
        actionToolsQuadricBezierCurve->setChecked(true);
}

void KPrView::toolsCubicBezierCurve()
{
    if ( actionToolsCubicBezierCurve->isChecked() ) {
        m_canvas->setToolEditMode( INS_CUBICBEZIERCURVE, false );
        deSelectAllObjects();
        m_currentLineTool = LtCubicBezier;
        actionToolsLinePopup->setIcon("cubicbeziercurve");
    }
    else
        actionToolsCubicBezierCurve->setChecked(true);
}

void KPrView::toolsConvexOrConcavePolygon()
{
    if ( actionToolsConvexOrConcavePolygon->isChecked() ) {
        m_canvas->setToolEditMode( INS_POLYGON, false );
        deSelectAllObjects();
        m_currentShapeTool = StPolygon;
        actionToolsShapePopup->setIcon("polygon");
    }
    else
        actionToolsConvexOrConcavePolygon->setChecked(true);
}

void KPrView::toolsClosedLinePopup()
{
    switch (m_currentClosedLineTool)
    {
    case CltFreehand:
        actionToolsClosedFreehand->activate();
        break;
    case CltPolyline:
        actionToolsClosedPolyline->activate();
        break;
    case CltQuadricBezier:
        actionToolsClosedQuadricBezierCurve->activate();
        break;
    case CltCubicBezier:
        actionToolsClosedCubicBezierCurve->activate();
        break;
    }
}

void KPrView::toolsClosedFreehand()
{
    if ( actionToolsClosedFreehand->isChecked() ) {
        m_canvas->setToolEditMode( INS_CLOSED_FREEHAND, false );
        deSelectAllObjects();
        m_currentClosedLineTool = CltFreehand;
        actionToolsClosedLinePopup->setIcon("closed_freehand");
    }
    else
        actionToolsClosedFreehand->setChecked( true );
}

void KPrView::toolsClosedPolyline()
{
    if ( actionToolsClosedPolyline->isChecked() ) {
        m_canvas->setToolEditMode( INS_CLOSED_POLYLINE, false );
        deSelectAllObjects();
        m_currentClosedLineTool = CltPolyline;
        actionToolsClosedLinePopup->setIcon("closed_polyline");
    }
    else
        actionToolsClosedPolyline->setChecked( true );
}

void KPrView::toolsClosedQuadricBezierCurve()
{
    if ( actionToolsClosedQuadricBezierCurve->isChecked() ) {
        m_canvas->setToolEditMode( INS_CLOSED_QUADRICBEZIERCURVE, false );
        deSelectAllObjects();
        m_currentClosedLineTool = CltQuadricBezier;
        actionToolsClosedLinePopup->setIcon("closed_quadricbeziercurve");
    }
    else
        actionToolsClosedQuadricBezierCurve->setChecked( true );
}

void KPrView::toolsClosedCubicBezierCurve()
{
    if ( actionToolsClosedCubicBezierCurve->isChecked() ) {
        m_canvas->setToolEditMode( INS_CLOSED_CUBICBEZIERCURVE, false );
        deSelectAllObjects();
        m_currentClosedLineTool = CltCubicBezier;
        actionToolsClosedLinePopup->setIcon("closed_cubicbeziercurve");
    }
    else
        actionToolsClosedCubicBezierCurve->setChecked( true );
}

void KPrView::extraProperties()
{
    m_canvas->setToolEditMode( TEM_MOUSE );

    m_propertyEditor = new KPrPropertyEditor( this, "KPrPropertyEditor", m_canvas->activePage(), m_pKPresenterDoc );
    m_propertyEditor->setCaption( i18n( "Properties" ) );

    connect( m_propertyEditor, TQ_SIGNAL( propertiesOk() ), this, TQ_SLOT( propertiesOk() ) );
    m_propertyEditor->exec();
    disconnect( m_propertyEditor, TQ_SIGNAL( propertiesOk() ), this, TQ_SLOT( propertiesOk() ) );

    delete m_propertyEditor;
    m_propertyEditor = 0;
}

void KPrView::extraRaise()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->raiseObjs( true );
}

void KPrView::extraLower()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->lowerObjs( true );
}

void KPrView::extraRotate()
{
    if ( m_canvas->numberOfObjectSelected() > 0 ) {
        if ( !rotateDia ) {
            rotateDia = new KPrRotationDialogImpl( this );
            connect( rotateDia, TQ_SIGNAL( apply() ), this, TQ_SLOT( rotateOk() ) );
        }
        rotateDia->setAngle( m_canvas->getSelectedObj()->getAngle() );
        m_canvas->setToolEditMode( TEM_MOUSE );
        rotateDia->exec();
    }
}

void KPrView::extraShadow()
{
    if ( m_canvas->numberOfObjectSelected() > 0 ) {

        if ( !shadowDia ) {
            shadowDia = new KPrShadowDialogImpl( this );
            shadowDia->resize( shadowDia->minimumSize() );
            connect( shadowDia, TQ_SIGNAL( apply() ), this, TQ_SLOT( shadowOk() ) );
        }

        KPrObject *object=m_canvas->getSelectedObj();
        shadowDia->setShadowDirection( object->getShadowDirection() );
        if ( object->getShadowDistance() != 0 )
            shadowDia->setShadowDistance( object->getShadowDistance() );
        else
            shadowDia->setShadowDistance( 3 );

        shadowDia->setShadowColor( object->getShadowColor() );
        m_canvas->setToolEditMode( TEM_MOUSE );
        shadowDia->exec();
    }
}

void KPrView::extraBackground()
{
    KPrPage *page=m_canvas->activePage();
    KPrBackDialog* backDia = new KPrBackDialog( this, "InfoDia", page->getBackType(  ),
                                    page->getBackColor1(  ),
                                    page->getBackColor2(  ),
                                    page->getBackColorType(  ),
                                    page->getBackPicture(  ),
                                    page->getBackView(),
                                    page->getBackUnbalanced(),
                                    page->getBackXFactor(),
                                    page->getBackYFactor( ),
                                    page );
    backDia->setCaption( i18n( "Slide Background" ) );
    TQObject::connect( backDia, TQ_SIGNAL( backOk( KPrBackDialog*, bool ) ), this, TQ_SLOT( backOk( KPrBackDialog*, bool ) ) ) ;
    backDia->exec();

    TQObject::disconnect( backDia, TQ_SIGNAL( backOk( KPrBackDialog*, bool ) ), this, TQ_SLOT( backOk( KPrBackDialog*, bool ) ) );
    delete backDia;
}

void KPrView::extraLayout()
{
    KoPageLayout pgLayout = m_pKPresenterDoc->pageLayout();
    KoPageLayout oldLayout = pgLayout;
    KoHeadFoot hf;
    KoUnit::Unit oldUnit = m_pKPresenterDoc->unit();
    KoUnit::Unit unit = oldUnit;

    if ( KoPageLayoutDia::pageLayout( pgLayout, hf, FORMAT_AND_BORDERS, unit, this ) ) {
        KPrPgLayoutCmd *pgLayoutCmd = new KPrPgLayoutCmd( i18n( "Set Page Layout" ),
                                                    pgLayout, oldLayout, oldUnit, unit,kPresenterDoc() );
        pgLayoutCmd->execute();
        kPresenterDoc()->addCommand( pgLayoutCmd );
        updateRuler();
    }
}

void KPrView::extraConfigure()
{
    KPrConfig configDia( this );
    configDia.exec();
}

void KPrView::extraCreateTemplate()
{
    int width = 60;
    int height = 60;
    TQPixmap pix = m_pKPresenterDoc->generatePreview(TQSize(width, height));
#if COPYOASISFORMAT
    KTempFile tempFile( TQString(), ".otp" );
    tempFile.setAutoDelete( true );
    m_pKPresenterDoc->savePage( tempFile.name(), getCurrPgNum() - 1);
#else
    KTempFile tempFile( TQString(), ".kpt" );
    tempFile.setAutoDelete( true );
    m_pKPresenterDoc->savePage( tempFile.name(), getCurrPgNum() - 1);
#endif

    KoTemplateCreateDia::createTemplate( "kpresenter_template", KPrFactory::global(),
                                         tempFile.name(), pix, this);
    KPrFactory::global()->dirs()->addResourceType("kpresenter_template",
                                                         TDEStandardDirs::kde_default( "data" ) +
                                                         "kpresenter/templates/");
}

void KPrView::extraDefaultTemplate()
{
#if COPYOASISFORMAT
    TQString file = locateLocal( "data", "kpresenter/default.oop" );
    m_pKPresenterDoc->savePage( file, currPg );
#else
    TQString file = locateLocal( "data", "kpresenter/default.kpr" );
    m_pKPresenterDoc->savePage( file, currPg );
#endif
}

void KPrView::extraWebPres()
{
    if ( !allowWebPres )
        return;

    KURL url;
    TQString config = TQString();
    int ret =KMessageBox::questionYesNoCancel( this,
                                               i18n( "Do you want to load a previously saved configuration"
                                                     " which will be used for this HTML Presentation?" ),
                                               i18n( "Create HTML Presentation" ) );
    if( ret == KMessageBox::Cancel )
        return;
    else if ( ret == KMessageBox::Yes )
    {
        url = KFileDialog::getOpenURL( TQString(), i18n("*.kpweb|KPresenter HTML Presentation (*.kpweb)"), this );

        if( url.isEmpty() )
            return;

        if( !url.isLocalFile() )
        {
            KMessageBox::sorry( this, i18n( "Only local files are currently supported." ) );
            return;
        }

        config = url.path();
    }

    KPrWebPresentationWizard::createWebPresentation( config, m_pKPresenterDoc, this );
}

void KPrView::extraMSPres()
{
    if ( !allowMSPres )
        return;

    KPrMSPresentationSetup::createMSPresentation( m_pKPresenterDoc, this );
}

void KPrView::extraLineBegin()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    TQPoint pnt( TQCursor::pos() );
    rb_lbegin->popup( pnt );
}

void KPrView::extraLineEnd()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    TQPoint pnt( TQCursor::pos() );
    rb_lend->popup( pnt );
}

void KPrView::extraGroup()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->groupObjects();
    objectSelectedChanged();
}

void KPrView::extraUnGroup()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->ungroupObjects();
    objectSelectedChanged();
}

void KPrView::extraPenStyle( int newStyle )
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    setExtraPenStyle(static_cast<TQt::PenStyle>(newStyle));
}

void KPrView::extraPenWidth( double newWidth )
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    setExtraPenWidth( newWidth );
}

void KPrView::screenConfigPages()
{
    delete pgConfDia;
    pgConfDia = new KPrPgConfDia( this, kPresenterDoc() );
    pgConfDia->setCaption( i18n( "Configure Slide Show" ) );
    TQObject::connect( pgConfDia, TQ_SIGNAL( pgConfDiaOk() ), this, TQ_SLOT( pgConfOk() ) );
    pgConfDia->exec();

    TQObject::disconnect( pgConfDia, TQ_SIGNAL( pgConfDiaOk() ), this, TQ_SLOT( pgConfOk() ) );
    delete pgConfDia;
    pgConfDia = 0;
}


void KPrView::screenTransEffect()
{
    KPrSlideTransitionDia transitionDia( this, "slideTransitionDialog", this );
    transitionDia.exec();
}

void KPrView::screenAssignEffect()
{
    m_canvas->setToolEditMode( TEM_MOUSE );

    TQPtrList<KPrObject> objs;
    if ( m_canvas->canAssignEffect( objs ) ) {
        KPrEffectDia *effectDia = new KPrEffectDia( this, "Object Effect", objs, this );
        effectDia->setCaption( i18n( "Object Effect" ) );
        effectDia->exec(); //the dialog executes the command itself
        delete effectDia;
    }
}

void KPrView::screenStart()
{
    startScreenPres( getCurrPgNum() );
}

void KPrView::screenStartFromFirst()
{
    startScreenPres( 1 );
}

void KPrView::startScreenPres( int pgNum /*1-based*/ )
{
    // no slide is selected ?
    if( !kPresenterDoc()->displaySelectedSlides().count() )
    {
        KMessageBox::sorry( this, i18n("You didn't select any slide." ),
                            i18n("No Slide") );
        return;
    }

    m_canvas->setToolEditMode( TEM_MOUSE );

    if ( m_canvas && !presStarted ) {
        const TQString xdgScreenSaver = TDEStandardDirs::findExe("xdg-screensaver");
        if (!xdgScreenSaver.isEmpty()) {
            TDEProcess proc;
            proc << xdgScreenSaver;
            proc << "suspend";
            proc << TQString::number( topLevelWidget()->winId() );
            kdDebug() << k_funcinfo << proc.args() << endl;
            proc.start( TDEProcess::DontCare );
        } else {
            TQByteArray data;
            TQByteArray replyData;
            TQCString replyType;
            m_screenSaverWasEnabled = false;
            // is screensaver enabled?
            if (kapp->dcopClient()->call("kdesktop", "KScreensaverIface", "isEnabled()", data, replyType, replyData)
                && replyType=="bool")
            {
                TQDataStream replyArg(replyData, IO_ReadOnly);
                replyArg >> m_screenSaverWasEnabled;
                kdDebug() << "Screensaver was enabled:" << m_screenSaverWasEnabled << endl;
                if ( m_screenSaverWasEnabled )
                {
                    // disable screensaver
                    TQDataStream arg(data, IO_WriteOnly);
                    arg << false;
                    if (!kapp->dcopClient()->send("kdesktop", "KScreensaverIface", "enable(bool)", data))
                        kdWarning(33001) << "Couldn't disable screensaver (using dcop to kdesktop)!" << endl;
                    else
                        kdDebug(33001) << "Screensaver successfully disabled" << endl;
                }
            } else {
                kdWarning(33001) << "Couldn't check screensaver (using dcop to kdesktop)!" << endl;
            }
            // is DPMS enabled?
            m_dpmsWasEnabled = isDPMSEnabled();
            kdDebug() << "DPMS was enabled:" << m_dpmsWasEnabled << endl;
            if ( m_dpmsWasEnabled ) {
                enableDPMS( false );
            }
        }

        deSelectAllObjects();
        presStarted = true;
        m_autoPresRestart = false;
        TQRect desk = TDEGlobalSettings::desktopGeometry(this);
        kdDebug(33001) << "KPrView::startScreenPres desk=" << desk << endl;
        TQRect pgRect = kPresenterDoc()->pageList().at(0)->getZoomPageRect();
        kdDebug(33001) << "KPrView::startScreenPres pgRect=" << pgRect << endl;

        double zoomX = static_cast<double>( desk.width() ) / (double)pgRect.width();
        double zoomY = static_cast<double>( desk.height() ) / (double)pgRect.height();
        kdDebug(33001) << "KPrView::startScreenPres zoomX=" << zoomX << " zoomY=" << zoomY << endl;

        xOffsetSaved = canvasXOffset();
        yOffsetSaved = canvasYOffset();
        setCanvasXOffset( 0 );
        setCanvasYOffset( 0 );

        m_bDisplayFieldCode = m_pKPresenterDoc->getVariableCollection()->variableSetting()->displayFieldCode();
        if ( m_bDisplayFieldCode )
        {
            m_pKPresenterDoc->getVariableCollection()->variableSetting()->setDisplayFieldCode(false);
            m_pKPresenterDoc->recalcVariables( VT_ALL );
        }

        vert->setEnabled( false );
        horz->setEnabled( false );
        m_bShowGUI = false;
        m_canvas->reparent( ( TQWidget* )0L, 0, TQPoint( 0, 0 ), false );
        m_canvas->setPaletteBackgroundColor( TQt::white );
        m_canvas->showFullScreen();
        m_canvas->setFocusPolicy( TQWidget::StrongFocus );

        if ( !kPresenterDoc()->spManualSwitch() )
        {
             m_autoPresStop = false;
             m_pKPresenterDoc->repaint( false );

             if ( ! m_autoPresTimerConnected ) {
                 connect( &m_autoPresTimer, TQ_SIGNAL( timeout() ), TQ_SLOT( doAutomaticScreenPres() ) );
                 m_autoPresTimerConnected = true;
             }
        }

        m_canvas->startScreenPresentation( zoomX, zoomY, pgNum );

        actionScreenStart->setEnabled( false );

        if ( kPresenterDoc()->presentationDuration() ) {
            m_duration.start();

            // ### make m_presentationDurationList a TQMemArray or TQValueVector
            for ( unsigned int i = 0; i < kPresenterDoc()->pageList().count(); ++i )
                m_presentationDurationList.append( 0 ); // initialization
        }
    }
}

void KPrView::screenStop()
{
    if ( presStarted ) {
        m_autoPresStop = true;
        m_canvas->stopSound();
        m_canvas->showNormal();
        m_canvas->hide();
        m_canvas->reparent( pageBase, 0, TQPoint( 0, 0 ), true );
        m_canvas->lower();
        setCanvasXOffset( xOffsetSaved );
        setCanvasYOffset( yOffsetSaved );

        if ( m_bDisplayFieldCode )
        {
            m_pKPresenterDoc->getVariableCollection()->variableSetting()->setDisplayFieldCode(true);
            m_pKPresenterDoc->recalcVariables( VT_ALL );
        }

        m_canvas->stopScreenPresentation();
        presStarted = false;
        vert->setEnabled( true );
        horz->setEnabled( true );
        m_bShowGUI = true;
        m_canvas->setMouseTracking( true );
        m_canvas->setBackgroundMode( TQt::NoBackground );

        const TQString xdgScreenSaver = TDEStandardDirs::findExe("xdg-screensaver");
        if (!xdgScreenSaver.isEmpty()) {
            TDEProcess proc;
            proc << xdgScreenSaver;
            proc << "resume";
            proc << TQString::number( topLevelWidget()->winId() );
            kdDebug() << k_funcinfo << proc.args() << endl;
            proc.start( TDEProcess::DontCare );
        } else {
            if ( m_screenSaverWasEnabled )
            {
                // start screensaver again
                TQByteArray data;
                TQDataStream arg(data, IO_WriteOnly);
                arg << true;
                if (!kapp->dcopClient()->send("kdesktop", "KScreensaverIface", "enable(bool)", data))
                    kdWarning(33001) << "Couldn't re-enabled screensaver (using dcop to kdesktop)" << endl;
            }
            if ( m_dpmsWasEnabled )
            {
                // re-enable DPMS
                kdDebug(33001) << "Re-enabling DPMS" << endl;
                enableDPMS( true );
            }
        }

        actionScreenStart->setEnabled( true );
        pageBase->resizeEvent( 0 );

        if ( kPresenterDoc()->presentationDuration() && !m_presentationDurationList.isEmpty() ) {
            openThePresentationDurationDialog();
            m_presentationDurationList.clear();
        }
    }
    emit presentationFinished();
}

void KPrView::screenPause()
{
}

void KPrView::screenFirst()
{
    m_canvas->deSelectAllObj();

    if ( !presStarted )
        skipToPage( 0 );
    else
        gotoPresPage( 1 );
}

void KPrView::screenPrev( bool gotoPreviousPage )
{
    if ( m_canvas->currentTextObjectView() )
        return;

    if ( presStarted ) {
        m_canvas->pPrev( gotoPreviousPage );
        m_canvas->setFocus();
    }
    else
        prevPage();
}

void KPrView::screenNext( bool gotoNextPage )
{
    if ( m_canvas->currentTextObjectView() )
        return;
    if ( presStarted )
    {
        m_canvas->pNext( gotoNextPage );
        m_canvas->setFocus();
    }
    else
    {
        nextPage();
    }
}

void KPrView::screenLast()
{
    m_canvas->deSelectAllObj();

    if ( !presStarted )
        skipToPage( m_pKPresenterDoc->getPageNums() - 1 );
    else
        gotoPresPage( getNumPresPages() );
}

void KPrView::screenSkip()
{
}

void KPrView::sizeSelected( int size )
{
    tbFont.setPointSize( size );
    m_canvas->setTextPointSize( size );
    m_canvas->setFocus();
}

void KPrView::fontSelected( const TQString &fontFamily )
{
    tbFont.setFamily( fontFamily );
    m_canvas->setTextFamily( fontFamily );
    m_canvas->setFocus();
}

void KPrView::textBold()
{
    bool b=actionTextBold->isChecked();
    tbFont.setBold( b );
    m_canvas->setTextBold(b );
}


void KPrView::textStrikeOut()
{
    bool b=actionFormatStrikeOut->isChecked();
    tbFont.setStrikeOut( b );
    m_canvas->setTextStrikeOut( b );
}

void KPrView::textItalic()
{
    bool b=actionTextItalic->isChecked();
    tbFont.setItalic( b );
    m_canvas->setTextItalic( b );
}

void KPrView::textUnderline()
{
    bool b=actionTextUnderline->isChecked();
    tbFont.setUnderline( b );
    m_canvas->setTextUnderline( b );
}

void KPrView::textColor()
{
    tbColor = actionTextColor->color();
    m_canvas->setTextColor( tbColor );
}

void KPrView::textAlignLeft()
{
    if ( actionTextAlignLeft->isChecked() )
    {
        tbAlign = TQt::AlignLeft;
        m_canvas->setTextAlign( tbAlign );
    }
    else
        actionTextAlignLeft->setChecked(true);
}

void KPrView::textAlignCenter()
{
    if ( actionTextAlignCenter->isChecked() )
    {
        tbAlign = TQt::AlignHCenter;
        m_canvas->setTextAlign(tbAlign);
    }
    else
        actionTextAlignCenter->setChecked(true);
}

void KPrView::textAlignRight()
{
    if ( actionTextAlignRight->isChecked() )
    {
        tbAlign = TQt::AlignRight;
        m_canvas->setTextAlign(tbAlign);
    }
    else
        actionTextAlignRight->setChecked(true);

}

void KPrView::textAlignBlock()
{
    if ( actionTextAlignBlock->isChecked() )
    {
        tbAlign = TQt::AlignJustify;
        m_canvas->setTextAlign(tbAlign);
    }
    else
        actionTextAlignBlock->setChecked(true);

}

void KPrView::textInsertPageNum()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->insertVariable( VT_PGNUM, KoPageVariable::VST_PGNUM_CURRENT );
}

void KPrView::mtextFont()
{
    KoTextFormatInterface* textIface = m_canvas->applicableTextInterfaces().first();
    TQColor col;
    if (textIface)
        col = textIface->textBackgroundColor();
    col = col.isValid() ? col : TQApplication::palette().color( TQPalette::Active, TQColorGroup::Base );

    delete m_fontDlg;

    m_fontDlg = new KoFontDia( *textIface->currentFormat()
                               , m_broker
                               , this, 0 );

    connect( m_fontDlg, TQ_SIGNAL( applyFont() ),
             this, TQ_SLOT( slotApplyFont() ) );
    m_fontDlg->exec();

    delete m_fontDlg;
    m_fontDlg=0L;
}

void KPrView::slotApplyFont()
{
    int flags = m_fontDlg->changedFlags();
    if ( flags )
    {
        m_canvas->setTextFormat(m_fontDlg->newFormat(), flags);
    }
}

void KPrView::slotCounterStyleSelected()
{
    TQString actionName = TQString::fromLatin1(sender()->name());
    if ( actionName.startsWith( "counterstyle_" ) )
    {
        TQString styleStr = actionName.mid(13);
        //kdDebug(33001) << "KWView::slotCounterStyleSelected styleStr=" << styleStr << endl;
        KoParagCounter::Style style = (KoParagCounter::Style)(styleStr.toInt());
        KoParagCounter c;
        if ( style == KoParagCounter::STYLE_NONE )
            c.setNumbering( KoParagCounter::NUM_NONE );
        else {
            c.setNumbering( KoParagCounter::NUM_LIST );
            c.setStyle( style );
            if ( c.isBullet() )
                c.setSuffix( TQString() );
            // else the suffix remains the default, '.'
            // TODO save this setting, to use the last one selected in the dialog?
            // (same for custom bullet char etc.)

            // 68927: restart numbering, by default, if last parag wasn't numbered
            // (and if we're not applying this to a selection)
            KPrTextView *edit = m_canvas->currentTextObjectView();
            if ( edit && !edit->textObject()->hasSelection() ) {
                KoTextParag* parag = edit->cursor()->parag();
                if ( parag->prev() && !parag->prev()->counter() )
                    c.setRestartCounter(true);
            }
        }

        TQPtrList<KoTextFormatInterface> lst = m_canvas->applicableTextInterfaces();
        TQPtrListIterator<KoTextFormatInterface> it( lst );
        KMacroCommand* macroCmd = 0L;
        for ( ; it.current() ; ++it )
        {
            KCommand *cmd = it.current()->setCounterCommand( c );
            if ( cmd )
            {
                if ( !macroCmd )
                    macroCmd = new KMacroCommand( i18n("Change List Type") );
                macroCmd->addCommand( cmd );
            }
        }
        if( macroCmd)
            m_pKPresenterDoc->addCommand( macroCmd );
    }

}

void KPrView::textDepthPlus()
{
    m_canvas->setTextDepthPlus();
}

void KPrView::textDepthMinus()
{
    m_canvas->setTextDepthMinus();
}

void KPrView::textContentsToHeight()
{
    m_canvas->textContentsToHeight();
}

void KPrView::textObjectToContents()
{
    m_canvas->textObjectToContents();
}

void KPrView::penChosen()
{
    TQColor c = actionPenColor->color();
    if ( !m_canvas->currentTextObjectView() )
    {
        KCommand * cmd( getPenCmd( i18n( "Change Outline Color" ), KoPen(c),
                                   L_NORMAL, L_NORMAL, KoPenCmd::Color ) );
        if( cmd )
            m_pKPresenterDoc->addCommand( cmd );
        else
            pen.setColor( c );
    }
    else
    {
        tbColor = c;
        m_canvas->setTextColor( tbColor );
    }
}

void KPrView::brushChosen()
{
    TQColor c = actionBrushColor->color();
    KPrTextView *edit = m_canvas->currentTextObjectView();
    if ( !edit )
    {
        //tz TODO remove macro, add parameter for command name in setBrush()
        KMacroCommand * macro= NULL;
        KCommand * cmd = NULL;
        TQBrush newBrush( c );

        cmd = m_canvas->activePage()->setBrush( newBrush, FT_BRUSH, TQColor(), TQColor(), BCT_PLAIN, false,
                                                0, 0, KPrBrushCmd::BrushColor | KPrBrushCmd::BrushStyle |
                                                KPrBrushCmd::BrushGradientSelect );
        if( cmd )
        {
            if ( !macro )
                macro = new KMacroCommand( i18n( "Change Fill Color" ) );
            macro->addCommand( cmd );
        }


        if( macro )
            m_pKPresenterDoc->addCommand( macro );
        else
            brush.setColor( c );
    }
    else
    {
        tbColor = c;
        m_canvas->setTextBackgroundColor( c );
    }
}

void KPrView::extraAlignObjLeft()
{
    m_canvas->alignObjects( AT_LEFT );
}

void KPrView::extraAlignObjCenterH()
{
    m_canvas->alignObjects( AT_HCENTER );
}

void KPrView::extraAlignObjRight()
{
    m_canvas->alignObjects( AT_RIGHT );
}

void KPrView::extraAlignObjTop()
{
    m_canvas->alignObjects( AT_TOP );
}

void KPrView::extraAlignObjCenterV()
{
    m_canvas->alignObjects( AT_VCENTER );
}

void KPrView::extraAlignObjBottom()
{
    m_canvas->alignObjects( AT_BOTTOM );
}

void KPrView::extraLineBeginNormal()
{
    setExtraLineBegin(L_NORMAL);
}

void KPrView::extraLineBeginArrow()
{
    setExtraLineBegin(L_ARROW);
}

void KPrView::extraLineBeginRect()
{
    setExtraLineBegin(L_SQUARE);
}

void KPrView::extraLineBeginCircle()
{
    setExtraLineBegin(L_CIRCLE);
}

void KPrView::extraLineBeginLineArrow()
{
    setExtraLineBegin( L_LINE_ARROW );
}

void KPrView::extraLineBeginDimensionLine()
{
    setExtraLineBegin( L_DIMENSION_LINE );
}

void KPrView::extraLineBeginDoubleArrow()
{
    setExtraLineBegin( L_DOUBLE_ARROW );
}

void KPrView::extraLineBeginDoubleLineArrow()
{
    setExtraLineBegin( L_DOUBLE_LINE_ARROW );
}

void KPrView::setExtraLineBegin(LineEnd lb)
{
    KCommand * cmd( getPenCmd( i18n("Change Line Begin"), KoPen(),
                               lb, L_NORMAL, KoPenCmd::BeginLine ) );
    if( cmd )
        kPresenterDoc()->addCommand( cmd );
    else
        lineBegin = lb;
}

void KPrView::extraLineEndNormal()
{
    setExtraLineEnd(L_NORMAL);
}

void KPrView::extraLineEndArrow()
{
    setExtraLineEnd(L_ARROW);
}

void KPrView::extraLineEndRect()
{
    setExtraLineEnd(L_SQUARE);
}

void KPrView::extraLineEndCircle()
{
    setExtraLineEnd(L_CIRCLE);
}

void KPrView::extraLineEndLineArrow()
{
    setExtraLineEnd( L_LINE_ARROW );
}

void KPrView::extraLineEndDimensionLine()
{
    setExtraLineEnd( L_DIMENSION_LINE );
}

void KPrView::extraLineEndDoubleArrow()
{
    setExtraLineEnd( L_DOUBLE_ARROW );
}

void KPrView::extraLineEndDoubleLineArrow()
{
    setExtraLineEnd( L_DOUBLE_LINE_ARROW );
}

void KPrView::setExtraLineEnd(LineEnd le)
{
    KCommand * cmd( getPenCmd( i18n("Change Line End"), KoPen(),
                               L_NORMAL, le, KoPenCmd::EndLine ) );
    if( cmd )
        kPresenterDoc()->addCommand( cmd );
    else
        lineEnd = le;
}

void KPrView::setExtraPenStyle( TQt::PenStyle style )
{
    KoPen tmpPen;
    tmpPen.setStyle( style );

    KCommand * cmd( getPenCmd( i18n("Change Outline Style"), tmpPen,
                               L_NORMAL, L_NORMAL, KoPenCmd::Style ) );
    if( cmd )
        kPresenterDoc()->addCommand( cmd );
    else
        pen.setStyle( style );
}

void KPrView::setExtraPenWidth( double width )
{
    KoPen tmpPen;
    tmpPen.setPointWidth( width );
    KCommand * cmd( getPenCmd( i18n("Change Outline Width"), tmpPen,
                               L_NORMAL, L_NORMAL, KoPenCmd::Width ) );

    if( cmd )
        kPresenterDoc()->addCommand( cmd );
    else
        pen.setPointWidth( width );
}

void KPrView::newPageLayout( const KoPageLayout &_layout )
{
    KoPageLayout oldLayout = m_pKPresenterDoc->pageLayout();
    KoUnit::Unit unit = m_pKPresenterDoc->unit(); // unchanged

    KPrPgLayoutCmd *pgLayoutCmd = new KPrPgLayoutCmd( i18n( "Set Page Layout" ), _layout, oldLayout, unit, unit,kPresenterDoc() );
    pgLayoutCmd->execute();
    kPresenterDoc()->addCommand( pgLayoutCmd );
    updateRuler();
}

void KPrView::updateRuler()
{
    //update koruler
    TQRect r=m_canvas->activePage()->getZoomPageRect();
    getHRuler()->setFrameStartEnd( r.left(), r.right()/*+m_canvas->diffx()*/ );
    getVRuler()->setFrameStartEnd( r.top(), r.bottom()/*+m_canvas->diffy()*/ );
}

void KPrView::createGUI()
{
    splitter = new TQSplitter( this );

    if ( !m_pKPresenterDoc->isEmbedded()
         && !m_pKPresenterDoc->isSingleViewMode() ) // No sidebar if the document is embedded
    {
        sidebar = new KPrSideBar( splitter, m_pKPresenterDoc, this );
        connect( sidebar, TQ_SIGNAL( movePage( int, int ) ),
                 m_pKPresenterDoc, TQ_SLOT( movePage( int, int ) ) );
        connect( sidebar, TQ_SIGNAL( selectPage( int, bool ) ),
                 m_pKPresenterDoc, TQ_SLOT( selectPage( int, bool ) ) );
        connect( sidebar, TQ_SIGNAL( showPage( int ) ),
                 this, TQ_SLOT( skipToPage( int ) ) );
        // This sucks when resizing the window
        //splitter->setResizeMode( sidebar, TQSplitter::FollowSizeHint );
        //splitter->setResizeMode( pageBase, TQSplitter::Stretch );
        splitter->setResizeMode( sidebar, TQSplitter::KeepSize );
    }

    TQSplitter *splitterVertical = new TQSplitter( TQt::Vertical, splitter );

    // setup page
    pageBase = new PageBase( splitterVertical, this );
    pageBase->setSizePolicy( TQSizePolicy( TQSizePolicy::Expanding, TQSizePolicy::Expanding ) );

    m_canvas=new KPrCanvas( pageBase, "Canvas", this );

    TQObject::connect( m_canvas, TQ_SIGNAL( fontChanged( const TQFont & ) ),
                      this, TQ_SLOT( fontChanged( const TQFont & ) ) );
    TQObject::connect( m_canvas, TQ_SIGNAL( colorChanged( const TQColor & ) ),
                      this, TQ_SLOT( colorChanged( const TQColor & ) ) );
    TQObject::connect( m_canvas, TQ_SIGNAL( alignChanged( int ) ),
                      this, TQ_SLOT( alignChanged( int ) ) );
    TQObject::connect( m_canvas, TQ_SIGNAL( objectSelectedChanged()),
                      this, TQ_SLOT( objectSelectedChanged()));
    TQObject::connect( m_canvas, TQ_SIGNAL( sigMouseWheelEvent( TQWheelEvent* ) ),
                      this, TQ_SLOT( getPageMouseWheelEvent( TQWheelEvent* ) ) );


    // setup notebar.
    if ( !m_pKPresenterDoc->isEmbedded()
         && !m_pKPresenterDoc->isSingleViewMode() ) // No notebar if the document is embedded
    {
        notebar = new KPrNoteBar( splitterVertical, this );
    }

    // setup GUI
    setupActions();
    setupPopupMenus();
    setupScrollbars();
    setRanges();
    setupRulers();

    if ( m_pKPresenterDoc && m_canvas )
    {
        TQObject::connect( m_canvas, TQ_SIGNAL( stopAutomaticPresentation() ), this, TQ_SLOT( stopAutomaticPresentation() ) );
        TQObject::connect( m_canvas, TQ_SIGNAL( restartPresentation() ), this, TQ_SLOT( restartPresentation() ) );

        connect( getVRuler(), TQ_SIGNAL( addGuide( const TQPoint &, bool, int ) ),
                 &( m_canvas->guideLines() ), TQ_SLOT( addGuide( const TQPoint &, bool, int ) ) );
        connect( getVRuler(), TQ_SIGNAL( moveGuide( const TQPoint &, bool, int ) ),
                 &( m_canvas->guideLines() ), TQ_SLOT( moveGuide( const TQPoint &, bool, int ) ) );
        connect( getHRuler(), TQ_SIGNAL( addGuide( const TQPoint &, bool, int ) ),
                 &( m_canvas->guideLines() ), TQ_SLOT( addGuide( const TQPoint &, bool, int ) ) );
        connect( getHRuler(), TQ_SIGNAL( moveGuide( const TQPoint &, bool, int ) ),
                 &( m_canvas->guideLines() ), TQ_SLOT( moveGuide( const TQPoint &, bool, int ) ) );
        connect( &( m_canvas->guideLines() ), TQ_SIGNAL( moveGuides( bool ) ),
                 m_canvas, TQ_SLOT( setPaintGuides( bool ) ) );
        connect( &( m_canvas->guideLines() ), TQ_SIGNAL( paintGuides( bool ) ),
                 m_canvas, TQ_SLOT( setPaintGuides( bool ) ) );
        connect( &( m_canvas->guideLines() ), TQ_SIGNAL( guideLinesChanged( KoView * ) ),
                 m_pKPresenterDoc, TQ_SLOT( slotGuideLinesChanged( KoView * ) ) );
        m_canvas->guideLines().setGuideLines( m_pKPresenterDoc->horizontalGuideLines(), m_pKPresenterDoc->verticalGuideLines() );
    }

    if ( sidebar )
    {
        sidebar->outline()->setCurrentItem( sidebar->outline()->firstChild() );
        sidebar->outline()->setSelected( sidebar->outline()->firstChild(), true );
        TDEConfig *config=TDEGlobal::config();
        config->setGroup("Global");
        if(!config->readBoolEntry("Sidebar", true)) {
            sidebar->hide();
            actionViewShowSideBar->setChecked(false);
        }
        else actionViewShowSideBar->setChecked(true);
    }

    if ( notebar )
    {
        TDEConfig *config=TDEGlobal::config();
        config->setGroup("Global");
        if(!config->readBoolEntry("Notebar", true)) {
            notebar->hide();
            actionViewShowNoteBar->setChecked(false);
        }
        else actionViewShowNoteBar->setChecked(true);
        // HACK This is needed to be able to calculate a good initial size for the notebar
        TQTimer::singleShot( 0, this, TQ_SLOT( initialLayoutOfSplitter() ) );
    }
    KPrPage *initPage=m_pKPresenterDoc->initialActivePage();
    if ( !initPage )
        m_pKPresenterDoc->pageList().at( 0 ); // first page
    skipToPage( m_pKPresenterDoc->pageList().findRef( initPage ) );
}

void KPrView::initGui()
{
    tbColor = TQt::black;
    actionTextColor->setCurrentColor( TQt::black );
    actionBrushColor->setCurrentColor( TQt::white );
    actionPenColor->setCurrentColor( TQt::black );
    updateSideBarMenu();
    objectSelectedChanged();
    refreshPageButton();

    KStatusBar * sb = statusBar();
    if ( sb )
        sb->show();
    showZoom( zoomHandler()->zoom() );
    updateHeaderFooterButton();
    updateDisplayObjectMasterPageButton();
    updateDisplayBackgroundButton();

    actionAllowAutoFormat->setChecked( m_pKPresenterDoc->allowAutoFormat() );
    actionViewFormattingChars->setChecked( m_pKPresenterDoc->viewFormattingChars() );

    updateGuideLineButton();

    updateGridButton();

    m_pKPresenterDoc->updateZoomRuler();
    updatePageInfo();
    actionAllowBgSpellCheck->setChecked( m_pKPresenterDoc->backgroundSpellCheckEnabled());
    updateDirectCursorButton();
    m_pKPresenterDoc->updatePresentationButton();
}

void KPrView::updateDisplayObjectMasterPageButton()
{
    actionDisplayObjectFromMasterPage->setChecked( !m_canvas->activePage()->displayObjectFromMasterPage());
}

void KPrView::updateDisplayBackgroundButton()
{
    actionDisplayBackgroundPage->setChecked( !m_canvas->activePage()->displayBackground() );
}

void KPrView::updateHeaderFooterButton()
{
    m_actionExtraHeader->setChecked( m_canvas->activePage()->hasHeader() );
    m_actionExtraFooter->setChecked( m_canvas->activePage()->hasFooter() );
}

void KPrView::guiActivateEvent( KParts::GUIActivateEvent *ev )
{
    if ( ev->activated() )
        initGui();

    KoView::guiActivateEvent( ev );
}

void KPrView::setupActions()
{
    actionEditCut = KStdAction::cut( this, TQ_SLOT( editCut() ), actionCollection(), "edit_cut" );
    actionEditCopy = KStdAction::copy( this, TQ_SLOT( editCopy() ), actionCollection(), "edit_copy" );
    actionEditPaste = KStdAction::paste( this, TQ_SLOT( editPaste() ), actionCollection(), "edit_paste" );
    connect( m_pKPresenterDoc, TQ_SIGNAL( enablePaste( bool ) ),
             actionEditPaste, TQ_SLOT( setEnabled( bool ) ) );
    m_pKPresenterDoc->clipboardDataChanged(); // set paste's initial state

    actionEditDelete = new TDEAction( i18n( "&Delete" ), "edit-delete", CTRL + TQt::Key_Delete,
                                    this, TQ_SLOT( editDelete() ),
                                    actionCollection(), "edit_delete" );
    actionEditSelectAll = KStdAction::selectAll( this, TQ_SLOT( editSelectAll() ), actionCollection(), "edit_selectall" );
    actionEditDeSelectAll= KStdAction::deselect( this, TQ_SLOT( editDeSelectAll()), actionCollection(), "edit_deselectall");
    actionEditCopyPage = new TDEAction( i18n( "Copy Slide" ), "edit-copy",
                                      0, this, TQ_SLOT( editCopyPage() ),
                                      actionCollection(), "edit_copypage" );
    actionEditDuplicatePage = new TDEAction( i18n( "Duplicate Slide" ), "newslide",
                                      0, this, TQ_SLOT( editDuplicatePage() ),
                                      actionCollection(), "edit_duplicatepage" );
    actionEditDelPage = new TDEAction( i18n( "Delete Slide" ), "delslide", 0,
                                     this, TQ_SLOT( editDelPage() ),
                                     actionCollection(), "edit_delpage" );

    actionEditFind=KStdAction::find( this, TQ_SLOT( editFind() ), actionCollection(), "edit_find" );
    actionEditFindNext = KStdAction::findNext( this, TQ_SLOT( editFindNext() ), actionCollection(), "edit_findnext" );
    actionEditFindPrevious = KStdAction::findPrev( this, TQ_SLOT( editFindPrevious() ), actionCollection(), "edit_findprevious" );
    actionEditReplace=KStdAction::replace( this, TQ_SLOT( editReplace() ), actionCollection(), "edit_replace" );

    // ---------------- View actions

    if ( !m_pKPresenterDoc->isEmbedded() )
    {
        actionViewShowSideBar = new TDEToggleAction( i18n("Show Sidebar"), 0,
                                                   this, TQ_SLOT( viewShowSideBar() ),
                                                   actionCollection(), "view_showsidebar" );
        actionViewShowSideBar->setCheckedState(i18n("Hide Sidebar"));

        actionViewShowNoteBar = new TDEToggleAction( i18n("Show Notebar"), 0,
                                                   this, TQ_SLOT( viewShowNoteBar() ),
                                                   actionCollection(), "view_shownotebar" );
        actionViewShowNoteBar->setCheckedState(i18n("Hide Notebar"));
    }

    actionViewFormattingChars = new TDEToggleAction( i18n( "&Formatting Characters" ), 0,
                                                   this, TQ_SLOT( slotViewFormattingChars() ),
                                                   actionCollection(), "view_formattingchars" );
    actionViewFormattingChars->setToolTip( i18n( "Toggle the display of non-printing characters." ) );
    actionViewFormattingChars->setWhatsThis( i18n( "Toggle the display of non-printing characters.<br><br>When this is enabled, KPresenter shows you tabs, spaces, carriage returns and other non-printing characters." ) );

    actionViewSlideMaster = new TDEToggleAction( i18n( "Slide &Master" ), 0,
                                               this, TQ_SLOT( viewSlideMaster() ),
                                               actionCollection(), "view_master" );

    actionViewShowGuideLine= new TDEToggleAction( i18n( "Guide Lines" ), 0,
                                               this, TQ_SLOT( viewGuideLines() ),
                                               actionCollection(), "view_guidelines" );

    actionViewShowGrid = new TDEToggleAction( i18n( "Show &Grid" ), 0,
                                            this, TQ_SLOT( viewGrid() ),
                                            actionCollection(), "view_grid" );
    actionViewShowGrid->setCheckedState(i18n("Hide &Grid"));

    actionViewSnapToGrid= new TDEToggleAction( i18n( "Snap to Grid" ), 0,
                                             this, TQ_SLOT(viewSnapToGrid() ),
                                             actionCollection(), "view_snaptogrid" );

    // ---------------- insert actions

    actionInsertPage = new TDEAction( i18n( "&Slide..." ), "slide", TQt::Key_F2,
                                    this, TQ_SLOT( insertPage() ),
                                    actionCollection(), "insert_page" );

    new TDEAction( i18n( "Insert &Slide..." ), "newslide", 0,
                                    this, TQ_SLOT( insertPage() ),
                                    actionCollection(), "insert_page_popup" );

    actionInsertPicture = new TDEAction( i18n( "P&icture..." ), "frame_image", TQt::SHIFT+TQt::Key_F5,
                                       this, TQ_SLOT( insertPicture() ),
                                       actionCollection(), "insert_picture" );

    // ----------------- tools actions

    actionToolsMouse = new TDEToggleAction( i18n( "Select" ), "select", 0,
                                          this, TQ_SLOT( toolsMouse() ),
                                          actionCollection(), "tools_mouse" );
    actionToolsMouse->setExclusiveGroup( "tools" );
    actionToolsMouse->setChecked( true );

    actionToolsRotate = new TDEToggleAction( i18n( "&Rotate" ), "rotate", 0,
                                           this, TQ_SLOT( toolsRotate() ),
                                           actionCollection(), "tools_rotate" );
    actionToolsRotate->setExclusiveGroup( "tools" );

    actionToolsZoom = new TDEToggleAction( i18n( "&Zoom" ), "viewmag", 0,
                                         this, TQ_SLOT( toolsZoom() ),
                                         actionCollection(), "tools_zoom" );
    actionToolsZoom->setExclusiveGroup( "tools" );

    actionToolsShapePopup = new TDEActionMenu( i18n( "&Shape" ), "rectangle",
                                             actionCollection(), "tools_shapepopup" );
    actionToolsShapePopup->setDelayed(true);
    connect(actionToolsShapePopup, TQ_SIGNAL(activated()), this, TQ_SLOT(toolsShapePopup()));

    actionToolsRectangle = new TDEToggleAction( i18n( "&Rectangle" ), "rectangle",
                                              0, this, TQ_SLOT( toolsRectangle() ),
                                              actionCollection(), "tools_rectangle" );
    actionToolsRectangle->setExclusiveGroup( "tools" );

    actionToolsCircleOrEllipse = new TDEToggleAction( i18n( "&Circle/Ellipse" ), "circle",
                                                    0, this, TQ_SLOT( toolsCircleOrEllipse() ),
                                                    actionCollection(), "tools_circle" );
    actionToolsCircleOrEllipse->setExclusiveGroup( "tools" );

    actionToolsPie = new TDEToggleAction( i18n( "&Pie/Arc/Chord" ), "pie", 0,
                                        this, TQ_SLOT( toolsPie() ),
                                        actionCollection(), "tools_pie" );
    actionToolsPie->setExclusiveGroup( "tools" );

    actionToolsText = new TDEToggleAction( i18n( "&Text" ), "frame_text", TQt::Key_F10, // same shortcut as KWord
                                         this, TQ_SLOT( toolsText() ),
                                         actionCollection(), "tools_text" );
    actionToolsText->setExclusiveGroup( "tools" );

    actionToolsAutoform = new TDEToggleAction( i18n( "&Arrows && Connections" ), "autoform",
                                             0, this, TQ_SLOT( toolsAutoform() ),
                                             actionCollection(), "tools_autoform" );
    actionToolsAutoform->setExclusiveGroup( "tools" );

    actionToolsDiagramm = new TDEToggleAction( i18n( "&Chart" ), "frame_chart", 0,
                                             this, TQ_SLOT( toolsDiagramm() ),
                                             actionCollection(), "tools_diagramm" );
    actionToolsDiagramm->setExclusiveGroup( "tools" );

    actionToolsTable = new TDEToggleAction( i18n( "Ta&ble"), "frame_spreadsheet", TQt::Key_F5 /*same as kword*/,
                                          this, TQ_SLOT( toolsTable() ),
                                          actionCollection(), "tools_table" );
    actionToolsTable->setExclusiveGroup( "tools" );

    actionToolsObject = new KoPartSelectAction( i18n( "&Object" ), "frame_query",
                                                this, TQ_SLOT( toolsObject() ),
                                                actionCollection(), "tools_object" );

    actionToolsLinePopup = new TDEActionMenu( i18n( "&Line" ), "line",
                                            actionCollection(), "tools_linepopup" );
    actionToolsLinePopup->setDelayed(true);
    connect(actionToolsLinePopup, TQ_SIGNAL(activated()), this, TQ_SLOT(toolsLinePopup()));

    actionToolsLine = new TDEToggleAction( i18n( "&Line" ), "line", 0,
                                         this, TQ_SLOT( toolsLine() ),
                                         actionCollection(), "tools_line" );
    actionToolsLine->setExclusiveGroup( "tools" );

    actionToolsFreehand = new TDEToggleAction( i18n( "&Freehand" ), "freehand", 0,
                                             this, TQ_SLOT( toolsFreehand() ),
                                             actionCollection(), "tools_freehand" );
    actionToolsFreehand->setExclusiveGroup( "tools" );

    actionToolsPolyline = new TDEToggleAction( i18n( "Po&lyline" ), "polyline", 0,
                                             this, TQ_SLOT( toolsPolyline() ),
                                             actionCollection(), "tools_polyline" );
    actionToolsPolyline->setExclusiveGroup( "tools" );

    actionToolsQuadricBezierCurve = new TDEToggleAction( i18n( "&Quadric Bezier Curve" ), "quadricbeziercurve", 0,
                                                       this, TQ_SLOT( toolsQuadricBezierCurve() ),
                                                       actionCollection(), "tools_quadricbeziercurve" );
    actionToolsQuadricBezierCurve->setExclusiveGroup( "tools" );

    actionToolsCubicBezierCurve = new TDEToggleAction( i18n( "C&ubic Bezier Curve" ), "cubicbeziercurve", 0,
                                                     this, TQ_SLOT( toolsCubicBezierCurve() ),
                                                     actionCollection(), "tools_cubicbeziercurve" );
    actionToolsCubicBezierCurve->setExclusiveGroup( "tools" );

    actionToolsConvexOrConcavePolygon = new TDEToggleAction( i18n( "Co&nvex/Concave Polygon" ), "polygon", 0,
                                                           this, TQ_SLOT( toolsConvexOrConcavePolygon() ),
                                                           actionCollection(), "tools_polygon" );
    actionToolsConvexOrConcavePolygon->setExclusiveGroup( "tools" );


    actionToolsClosedLinePopup = new TDEActionMenu( i18n( "&Closed Line" ), "closed_freehand",
                                                  actionCollection(), "tools_closed_linepopup" );
    actionToolsClosedLinePopup->setDelayed(true);
    connect(actionToolsClosedLinePopup, TQ_SIGNAL(activated()), this, TQ_SLOT(toolsClosedLinePopup()));

    actionToolsClosedFreehand = new TDEToggleAction( i18n( "Closed &Freehand" ), "closed_freehand", 0,
                                                   this, TQ_SLOT( toolsClosedFreehand() ),
                                                   actionCollection(), "tools_closed_freehand" );
    actionToolsClosedFreehand->setExclusiveGroup( "tools" );


    actionToolsClosedPolyline = new TDEToggleAction( i18n( "Closed Po&lyline" ), "closed_polyline", 0,
                                                   this, TQ_SLOT( toolsClosedPolyline() ),
                                                   actionCollection(), "tools_closed_polyline" );
    actionToolsClosedPolyline->setExclusiveGroup( "tools" );


    actionToolsClosedQuadricBezierCurve = new TDEToggleAction( i18n( "Closed &Quadric Bezier Curve" ), "closed_quadricbeziercurve", 0,
                                                             this, TQ_SLOT( toolsClosedQuadricBezierCurve() ),
                                                             actionCollection(), "tools_closed_quadricbeziercurve" );
    actionToolsClosedQuadricBezierCurve->setExclusiveGroup( "tools" );


    actionToolsClosedCubicBezierCurve = new TDEToggleAction( i18n( "Closed C&ubic Bezier Curve" ), "closed_cubicbeziercurve", 0,
                                                           this, TQ_SLOT( toolsClosedCubicBezierCurve() ),
                                                           actionCollection(), "tools_closed_cubicbeziercurve" );
    actionToolsClosedCubicBezierCurve->setExclusiveGroup( "tools" );

    // ----------------- text actions

    actionTextFont = new TDEAction( i18n( "&Font..." ), 0, this, TQ_SLOT( mtextFont() ),
                                  actionCollection(), "text_font" );

    actionTextFontSize = new TDEFontSizeAction( i18n( "Font Size" ), 0, actionCollection(), "text_fontsize" );
    connect( actionTextFontSize, TQ_SIGNAL( fontSizeChanged( int ) ),
             this, TQ_SLOT( sizeSelected( int ) ) );

#ifdef KFONTACTION_HAS_CRITERIA_ARG
    actionTextFontFamily = new TDEFontAction( TDEFontChooser::SmoothScalableFonts,
                                            i18n( "Font Family" ), 0,
                                            actionCollection(), "text_fontfamily" );
#else
    actionTextFontFamily = new TDEFontAction( i18n( "Font Family" ), 0,
                                            actionCollection(), "text_fontfamily" );
#endif
    connect( actionTextFontFamily , TQ_SIGNAL( activated( const TQString & ) ),
             this, TQ_SLOT( fontSelected( const TQString & ) ) );

    actionTextBold = new TDEToggleAction( i18n( "&Bold" ), "format-text-bold", CTRL + TQt::Key_B,
                                        this, TQ_SLOT( textBold() ),
                                        actionCollection(), "format-text-bold" );

    actionTextItalic = new TDEToggleAction( i18n( "&Italic" ), "format-text-italic", CTRL + TQt::Key_I,
                                          this, TQ_SLOT( textItalic() ),
                                          actionCollection(), "format-text-italic" );

    actionTextUnderline = new TDEToggleAction( i18n( "&Underline" ), "format-text-underline", CTRL + TQt::Key_U,
                                             this, TQ_SLOT( textUnderline() ),
                                             actionCollection(), "text_underline" );

    actionFormatStrikeOut = new TDEToggleAction( i18n( "&Strike Out" ), "format-text-strikethrough", 0 ,
                                               this, TQ_SLOT( textStrikeOut() ),
                                               actionCollection(), "format_strike" );

    actionTextColor = new TTDESelectColorAction( i18n( "&Color..." ), TTDESelectColorAction::TextColor,
                                               actionCollection(), "text_color" ,true);
    connect( actionTextColor, TQ_SIGNAL( activated() ), TQ_SLOT( textColor() ) );
    actionTextColor->setDefaultColor(TQColor());


    actionTextAlignLeft = new TDEToggleAction( i18n( "Align &Left" ), "format-text-direction-ltr", ALT + TQt::Key_L,
                                             this, TQ_SLOT( textAlignLeft() ),
                                             actionCollection(), "text_alignleft" );
    actionTextAlignLeft->setExclusiveGroup( "align" );
    actionTextAlignLeft->setChecked( true );

    actionTextAlignCenter = new TDEToggleAction( i18n( "Align &Center" ), "text_center", ALT + TQt::Key_C,
                                               this, TQ_SLOT( textAlignCenter() ),
                                               actionCollection(), "text_aligncenter" );
    actionTextAlignCenter->setExclusiveGroup( "align" );

    actionTextAlignRight = new TDEToggleAction( i18n( "Align &Right" ), "format-text-direction-rtl", ALT + TQt::Key_R,
                                              this, TQ_SLOT( textAlignRight() ),
                                              actionCollection(), "text_alignright" );
    actionTextAlignRight->setExclusiveGroup( "align" );

    actionTextAlignBlock = new TDEToggleAction( i18n( "Align &Block" ), "text_block", CTRL + TQt::Key_J,
                                              this, TQ_SLOT( textAlignBlock() ),
                                              actionCollection(), "text_alignblock" );
    actionTextAlignBlock->setExclusiveGroup( "align" );


    actionFormatNumber = new TDEActionMenu( i18n( "Number" ), "enumList", actionCollection(), "format_number" );
    actionFormatNumber->setDelayed( false );
    actionFormatBullet = new TDEActionMenu( i18n( "Bullet" ), "unsortedList", actionCollection(), "format_bullet" );
    actionFormatBullet->setDelayed( false );
    TQPtrList<KoCounterStyleWidget::StyleRepresenter> stylesList;
    KoCounterStyleWidget::makeCounterRepresenterList( stylesList );
    TQPtrListIterator<KoCounterStyleWidget::StyleRepresenter> styleIt( stylesList );
    for ( ; styleIt.current() ; ++styleIt ) {
        // Dynamically create toggle-actions for each list style.
        // This approach allows to edit toolbars and extract separate actions from this menu
        TDEToggleAction* act = new TDEToggleAction( styleIt.current()->name(), /*TODO icon,*/
                                                0, this, TQ_SLOT( slotCounterStyleSelected() ),
                                                actionCollection(), TQString("counterstyle_%1").arg( styleIt.current()->style() ).latin1() );
        act->setExclusiveGroup( "counterstyle" );
        // Add to the right menu: both for "none", bullet for bullets, numbers otherwise
        if ( styleIt.current()->style() == KoParagCounter::STYLE_NONE ) {
            actionFormatBullet->insert( act );
            actionFormatNumber->insert( act );
        } else if ( styleIt.current()->isBullet() )
            actionFormatBullet->insert( act );
        else
            actionFormatNumber->insert( act );
    }
    actionTextDepthPlus = new TDEAction( i18n( "&Increase Depth" ),  TQApplication::reverseLayout() ?"format_decreaseindent" : "format_increaseindent",
                                       CTRL + TQt::Key_Plus, this, TQ_SLOT( textDepthPlus() ),
                                       actionCollection(), "text_depthPlus" );

    actionTextDepthMinus = new TDEAction( i18n( "&Decrease Depth" ), TQApplication::reverseLayout() ?"format_increaseindent" : "format_decreaseindent",
                                        CTRL + TQt::Key_Minus, this, TQ_SLOT( textDepthMinus() ),
                                        actionCollection(), "text_depthMinus" );

    actionTextExtentCont2Height = new TDEAction( i18n( "Extend Contents to Object &Height" ), 0,
                                               this, TQ_SLOT( textContentsToHeight() ),
                                               actionCollection(), "text_con2hei" );

    actionTextExtendObj2Cont = new TDEAction( i18n( "&Extend Object to Fit Contents" ), 0,
                                            this, TQ_SLOT( textObjectToContents() ),
                                            actionCollection(), "text_obj2cont" );

    actionTextInsertPageNum = new TDEAction( i18n( "&Insert Slide Number" ), "pgnum", 0,
                                           this, TQ_SLOT( textInsertPageNum() ),
                                           actionCollection(), "text_inspgnum" );

    // ----------------- format actions

    actionExtraProperties = new TDEAction( i18n( "&Properties" ), "penbrush", 0,
                                         this, TQ_SLOT( extraProperties() ),
                                         actionCollection(), "extra_properties" );

    actionExtraArrangePopup = new TDEActionMenu( i18n( "Arra&nge Objects" ), "arrange",
                                               actionCollection(), "extra_arrangepopup" );
    actionExtraArrangePopup->setDelayed( false );

    actionExtraRaise = new TDEAction( i18n( "Ra&ise Objects" ), "raise",
                                    CTRL+TQt::SHIFT+TQt::Key_R, this, TQ_SLOT( extraRaise() ),
                                    actionCollection(), "extra_raise" );

    actionExtraLower = new TDEAction( i18n( "&Lower Objects" ), "lower", CTRL +TQt::SHIFT+ TQt::Key_L,
                                    this, TQ_SLOT( extraLower() ),
                                    actionCollection(), "extra_lower" );

    actionExtraBringForward= new TDEAction( i18n( "Bring to Front" ), "bring_forward",
                                          0, this, TQ_SLOT( extraBringForward() ),
                                          actionCollection(), "extra_bring_forward" );

    actionExtraSendBackward= new TDEAction( i18n( "Send to Back" ), "send_backward",
                                          0, this, TQ_SLOT( extraSendBackward() ),
                                          actionCollection(), "extra_send_backward" );



    actionExtraRotate = new TDEAction( i18n( "R&otate Objects..." ), "object-rotate-right", 0,
                                     this, TQ_SLOT( extraRotate() ),
                                     actionCollection(), "extra_rotate" );

    actionExtraShadow = new TDEAction( i18n( "&Shadow Objects..." ), "shadow", 0,
                                     this, TQ_SLOT( extraShadow() ),
                                     actionCollection(), "extra_shadow" );

    actionExtraAlignObjLeft = new TDEAction( i18n( "Align &Left" ), "aoleft", 0,
                                           this, TQ_SLOT( extraAlignObjLeft() ),
                                           actionCollection(), "extra_alignleft" );

    actionExtraAlignObjCenterH = new TDEAction( i18n( "Align Center (&horizontally)" ),
                                              "aocenterh", 0,
                                              this, TQ_SLOT( extraAlignObjCenterH() ),
                                              actionCollection(), "extra_aligncenterh" );

    actionExtraAlignObjRight = new TDEAction( i18n( "Align &Right" ), "aoright", 0,
                                            this, TQ_SLOT( extraAlignObjRight() ),
                                            actionCollection(), "extra_alignright" );

    actionExtraAlignObjTop = new TDEAction( i18n( "Align &Top" ), "aotop", 0,
                                          this, TQ_SLOT( extraAlignObjTop() ),
                                          actionCollection(), "extra_aligntop" );

    actionExtraAlignObjCenterV = new TDEAction( i18n( "Align Center (&vertically)" ),
                                              "aocenterv", 0,
                                              this, TQ_SLOT( extraAlignObjCenterV() ),
                                              actionCollection(), "extra_aligncenterv" );

    actionExtraAlignObjBottom = new TDEAction( i18n( "Align &Bottom" ), "aobottom", 0,
                                             this, TQ_SLOT( extraAlignObjBottom() ),
                                             actionCollection(), "extra_alignbottom" );


    actionExtraBackground = new TDEAction( i18n( "Slide Bac&kground..." ), "background", 0,
                                         this, TQ_SLOT( extraBackground() ),
                                         actionCollection(), "extra_background" );

    actionExtraLayout = new TDEAction( i18n( "Page &Layout..." ), 0,
                                     this, TQ_SLOT( extraLayout() ),
                                     actionCollection(), "extra_layout" );

    m_actionExtraHeader = new TDEToggleAction( i18n( "Enable Document &Header" ), 0,
                                             this, TQ_SLOT( viewHeader() ),
                                             actionCollection(), "extra_header" );
    m_actionExtraHeader->setCheckedState( i18n( "Disable Document &Header" ) );
    m_actionExtraHeader->setToolTip( i18n( "Shows and hides header display for the current slide." ) );

    m_actionExtraFooter = new TDEToggleAction( i18n( "Enable Document Foo&ter" ), 0,
                                             this, TQ_SLOT( viewFooter() ),
                                             actionCollection(), "extra_footer" );
    m_actionExtraFooter->setCheckedState( i18n( "Disable Document Foo&ter" ) );
    m_actionExtraFooter->setToolTip( i18n( "Shows and hides footer display for the current slide." ) );

    actionExtraConfigure = new TDEAction( i18n( "Configure KPresenter..." ),
                                        "configure", 0,
                                        this, TQ_SLOT( extraConfigure() ),
                                        actionCollection(), "extra_configure" );

    actionExtraWebPres = new TDEAction( i18n( "Create &HTML Slideshow..." ),
                                      "webpres", 0,
                                      this, TQ_SLOT( extraWebPres() ),
                                      actionCollection(), "extra_webpres" );

    actionExtraMSPres = new TDEAction( i18n( "Create Memor&y Stick Slideshow..." ),
				     0, this, TQ_SLOT( extraMSPres() ),
				     actionCollection(), "extra_mspres" );

    actionExtraCreateTemplate = new TDEAction( i18n( "Template Manager" ), 0,
                                             this, TQ_SLOT( extraCreateTemplate() ),
                                             actionCollection(), "extra_template" );

    actionExtraDefaultTemplate = new TDEAction( i18n( "Use Current Slide as Default Template" ), 0,
                                              this, TQ_SLOT( extraDefaultTemplate() ),
                                              actionCollection(), "extra_defaulttemplate" );

    actionExtraAlignObjsPopup = new TDEActionMenu( i18n("Align O&bjects"), "alignobjs",
                                            actionCollection(), "extra_alignobjs" );
    actionExtraAlignObjsPopup->setDelayed( false );

    actionExtraLineBegin = new TDEAction( i18n("Line Begin"), "line_begin", 0,
                                        this, TQ_SLOT( extraLineBegin() ),
                                        actionCollection(), "extra_linebegin" );

    actionExtraLineEnd = new TDEAction( i18n("Line End"), "line_end", 0,
                                      this, TQ_SLOT( extraLineEnd() ),
                                      actionCollection(), "extra_lineend" );

    actionExtraPenStyle = new KoLineStyleAction( i18n("Outline Style"), "pen_style",
                                       this, TQ_SLOT( extraPenStyle(int) ),
                                       actionCollection(), "extra_penstyle" );
    actionExtraPenStyle->setShowCurrentSelection(false);

    actionExtraPenWidth = new KoLineWidthAction( i18n("Outline Width"), "pen_width",
                                       this, TQ_SLOT( extraPenWidth(double) ),
                                       actionCollection(), "extra_penwidth" );
    actionExtraPenWidth->setUnit( kPresenterDoc()->unit() );
    actionExtraPenWidth->setShowCurrentSelection(false);
    connect( kPresenterDoc(), TQ_SIGNAL( unitChanged( KoUnit::Unit ) ),
             actionExtraPenWidth, TQ_SLOT( setUnit( KoUnit::Unit ) ) );

    actionExtraGroup = new TDEAction( i18n( "&Group Objects" ), "group",
                                    TQKeySequence( "Ctrl+G" ),
                                    this, TQ_SLOT( extraGroup() ),
                                    actionCollection(), "extra_group" );

    actionExtraUnGroup = new TDEAction( i18n( "&Ungroup Objects" ), "ungroup",
                                      TQKeySequence( "Ctrl+Shift+G" ),
                                      this, TQ_SLOT( extraUnGroup() ),
                                      actionCollection(), "extra_ungroup" );

    // ----------------- slideshow actions

    actionScreenConfigPages = new TDEAction( i18n( "&Configure Slide Show..." ),
                                           "configure", 0,
                                           this, TQ_SLOT( screenConfigPages() ),
                                           actionCollection(), "screen_configpages" );

    actionScreenAssignEffect = new TDEAction( i18n( "Edit &Object Effect..." ),
                                            "effect", 0,
                                            this, TQ_SLOT( screenAssignEffect() ),
                                            actionCollection(), "screen_assigneffect");

    actionScreenTransEffect = new TDEAction( i18n( "Edit Slide &Transition..." ),
                                           "slide_effect", 0,
                                           this, TQ_SLOT( screenTransEffect() ),
                                           actionCollection(), "screen_transeffect");


    actionScreenStart = new TDEAction( i18n( "&Start" ),
                                     "2rightarrow", TQt::Key_F12,
                                     this, TQ_SLOT( screenStart() ),
                                     actionCollection(), "screen_start" );

    actionScreenStartFromFirst = new TDEAction( i18n( "Start From &First Slide" ),
                                              "1rightarrow", 0,
                                              this, TQ_SLOT( screenStartFromFirst() ),
                                              actionCollection(), "screen_startfromfirst" );

    actionScreenFirst = new TDEAction( i18n( "&Go to Start" ),
                                     "go-first", 0,
                                     this, TQ_SLOT( screenFirst() ),
                                     actionCollection(), "screen_first" );

    actionScreenPrev = new TDEAction( i18n( "&Previous Slide" ),
                                    "back", TQt::Key_Prior,
                                    this, TQ_SLOT( screenPrev() ),
                                    actionCollection(), "screen_prev" );

    actionScreenNext = new TDEAction( i18n( "&Next Slide" ),
                                    "forward", TQt::Key_Next,
                                    this, TQ_SLOT( screenNext() ),
                                    actionCollection(), "screen_next" );

    actionScreenLast = new TDEAction( i18n( "Go to &End" ),
                                    "go-last", 0,
                                    this, TQ_SLOT( screenLast() ),
                                    actionCollection(), "screen_last" );

    actionScreenSkip = new TDEAction( i18n( "Goto &Slide..." ),
                                    "goto", 0,
                                    this, TQ_SLOT( screenSkip() ),
                                    actionCollection(), "screen_skip" );

    // ----------------- colorbar(Brush and Pen) action

    actionBrushColor = new TTDESelectColorAction( i18n( "Fill Color..." ), TTDESelectColorAction::FillColor,
                                                actionCollection(), "brush_color" ,true);
    connect( actionBrushColor, TQ_SIGNAL( activated() ), TQ_SLOT( brushChosen() ) );
    actionBrushColor->setDefaultColor(TQColor());

    actionPenColor = new TTDESelectColorAction( i18n( "Outline Color..." ), TTDESelectColorAction::LineColor,
                                              actionCollection(), "pen_color" );
    connect( actionPenColor, TQ_SIGNAL( activated() ), TQ_SLOT( penChosen() ) );
    actionPenColor->setDefaultColor(TQColor());
    actionExtendObjectHeight = new TDEAction( i18n( "&Extend Contents to Object Height" ),0, this, TQ_SLOT( textContentsToHeight() ),
                                            actionCollection(), "extendobjectheight" );

    actionResizeTextObject = new TDEAction( i18n( "&Resize Object to Fit Contents" ),0, this, TQ_SLOT( textObjectToContents() ),
                                          actionCollection(), "resizetextobject" );

    actionRenamePage=new TDEAction(i18n( "&Rename Slide..." ),0,this,
                                 TQ_SLOT( renamePageTitle() ),
                                 actionCollection(), "rename_page" );

    actionPicOriginalSize = new TDEAction( i18n( "Sca&le to Original Size" ), 0, this,
                                         TQ_SLOT( picViewOriginalSize() ),
                                         actionCollection(), "pic_original_size" );

    actionPic640x480=new TDEAction(i18n( "640x480" ),0,this,
                                 TQ_SLOT( picViewOrig640x480() ),
                                 actionCollection(), "pic_640_480" );

    actionPic800x600=new TDEAction(i18n( "800x600" ),0,this,
                                 TQ_SLOT( picViewOrig800x600() ),
                                 actionCollection(), "pic_800_600" );

    actionPic1024x768=new TDEAction(i18n( "1024x768" ),0,this,
                                  TQ_SLOT( picViewOrig1024x768() ),
                                  actionCollection(), "pic_1024_768" );

    actionPic1280x1024=new TDEAction(i18n( "1280x1024" ),0,this,
                                   TQ_SLOT( picViewOrig1280x1024() ),
                                   actionCollection(), "pic_1280_1024" );

    actionPic1600x1200=new TDEAction(i18n( "1600x1200" ),0,this,
                                   TQ_SLOT( picViewOrig1600x1200() ),
                                   actionCollection(), "pic_1600_1200" );

    actionChangePic=new TDEAction( i18n( "&Change Picture..." ),"frame_image",0,this,
                                 TQ_SLOT( chPic() ), actionCollection(), "change_picture" );


    actionImageEffect = new TDEAction( i18n("Image &Effect..."), 0, this,
                                     TQ_SLOT(imageEffect()), actionCollection(), "image_effect");


    actionFormatSuper = new TDEToggleAction( i18n( "Superscript" ), "super", 0,
                                           this, TQ_SLOT( textSuperScript() ),
                                           actionCollection(), "format_super" );
    actionFormatSuper->setExclusiveGroup( "valign" );
    actionFormatSub = new TDEToggleAction( i18n( "Subscript" ), "sub", 0,
                                         this, TQ_SLOT( textSubScript() ),
                                         actionCollection(), "format_sub" );
    actionFormatSub->setExclusiveGroup( "valign" );


    actionInsertSpecialChar = new TDEAction( i18n( "Sp&ecial Character..." ), "char",
                                           ALT + TQt::SHIFT + TQt::Key_C,
                                           this, TQ_SLOT( insertSpecialChar() ),
                                           actionCollection(), "insert_specialchar" );

    actionInsertLink = new TDEAction( i18n( "Link..." ), 0,
                                    this, TQ_SLOT( insertLink() ),
                                    actionCollection(), "insert_link" );

#if 0
    //code from page.cpp
    //not implemented
    picResizeMenu->insertSeparator();
    picResizeMenu->insertItem( i18n( "Enter Custom Factor..." ), this, TQ_SLOT( picViewOrigFactor() ) );
#endif
    (void) new TDEAction( i18n( "Configure &Autocorrection..." ), 0,
                        this, TQ_SLOT( extraAutoFormat() ),
                        actionCollection(), "extra_autocorrection" );
    actionExtraSpellCheck = KStdAction::spelling( this, TQ_SLOT( slotSpellCheck() ), actionCollection(), "extra_spellcheck" );

    actionFormatParag = new TDEAction( i18n( "&Paragraph..." ), ALT + CTRL + TQt::Key_P,
                                     this, TQ_SLOT( formatParagraph() ),
                                     actionCollection(), "format_paragraph" );

    actionFormatDefault=new TDEAction( i18n( "Default Format" ), 0,
                                     this, TQ_SLOT( textDefaultFormat() ),
                                     actionCollection(), "text_default" );

    actionOpenLink = new TDEAction( i18n( "Open Link" ), 0,
                                  this, TQ_SLOT( openLink() ),
                                  actionCollection(), "open_link" );

    actionChangeLink=new TDEAction( i18n("Change Link..."), 0,
                                  this, TQ_SLOT(changeLink()),
                                  actionCollection(), "change_link");

    actionCopyLink = new TDEAction( i18n( "Copy Link" ), 0,
                                  this, TQ_SLOT( copyLink() ),
                                  actionCollection(), "copy_link" );

    actionRemoveLink = new TDEAction( i18n( "Remove Link" ), 0,
                                    this, TQ_SLOT( removeLink() ),
                                    actionCollection(), "remove_link" );


    actionAddLinkToBookmak = new TDEAction( i18n( "Add to Bookmark" ), 0,
                                          this, TQ_SLOT( addToBookmark() ),
                                          actionCollection(), "add_to_bookmark" );

    actionEditCustomVarsEdit = new TDEAction( i18n( "&Custom Variables..." ), 0,
                                            this, TQ_SLOT( editCustomVars() ),
                                            actionCollection(), "edit_vars" );

    actionEditCustomVars = new TDEAction( i18n( "Edit Variable..." ), 0,
                                        this, TQ_SLOT( editCustomVariable() ),
                                        actionCollection(), "edit_customvars" );


    m_variableDefMap.clear();
    actionInsertVariable = new TDEActionMenu( i18n( "&Variable" ),
                                            actionCollection(), "insert_variable" );
    // The last argument is only needed if a submenu is to be created
    addVariableActions( VT_FIELD, KoFieldVariable::actionTexts(), actionInsertVariable, i18n("&Property") );
    addVariableActions( VT_DATE, KoDateVariable::actionTexts(), actionInsertVariable, i18n("&Date") );
    addVariableActions( VT_TIME, KoTimeVariable::actionTexts(), actionInsertVariable, i18n("&Time") );

    actionInsertCustom = new TDEActionMenu( i18n( "&Custom" ),
                                          actionCollection(), "insert_custom" );
    actionInsertVariable->insert(actionInsertCustom);
    refreshCustomMenu();

    addVariableActions( VT_PGNUM, KoPageVariable::actionTexts(), actionInsertVariable, i18n("&Page") );
    addVariableActions( VT_STATISTIC, KPrStatisticVariable::actionTexts(), actionInsertVariable, i18n("&Statistic") );

    actionInsertVariable->popupMenu()->insertSeparator();
    actionRefreshAllVariable = new TDEAction( i18n( "&Refresh All Variables" ), 0,
                                            this, TQ_SLOT( refreshAllVariable() ),
                                            actionCollection(), "refresh_all_variable" );
    actionInsertVariable->insert(actionRefreshAllVariable);

    actionIncreaseFontSize = new TDEAction( i18n("Increase Font Size"),"fontsizeup", 0,
                                          this, TQ_SLOT( increaseFontSize() ),
                                          actionCollection(), "increaseFontSize" );

    actionDecreaseFontSize = new TDEAction( i18n("Decrease Font Size"),"fontsizedown", 0,
                                          this, TQ_SLOT( decreaseFontSize() ),
                                          actionCollection(), "decreaseFontSize" );

    actionChangeCase=new TDEAction( i18n( "Change Case..." ), 0,
                                  this, TQ_SLOT( changeCaseOfText() ),
                                  actionCollection(), "change_case" );

    actionViewZoom = new TDESelectAction( i18n( "Zoom" ), "viewmag", 0,
                                        actionCollection(), "view_zoom" );
    connect( actionViewZoom, TQ_SIGNAL( activated( const TQString & ) ),
             this, TQ_SLOT( viewZoom( const TQString & ) ) );
    actionViewZoom->setEditable(true);
    changeZoomMenu( );

    actionFormatStylist = new TDEAction( i18n( "&Style Manager" ), ALT + CTRL + TQt::Key_S,
                                       this, TQ_SLOT( extraStylist() ),
                                       actionCollection(), "format_stylist" );

    actionFormatStyleMenu = new TDEActionMenu( i18n( "St&yle" ), 0,
                                             actionCollection(), "format_stylemenu" );


    actionFormatStyle = new TDESelectAction( i18n( "St&yle" ), 0,
                                           actionCollection(), "format_style" );
    connect( actionFormatStyle, TQ_SIGNAL( activated( int ) ),
             this, TQ_SLOT( textStyleSelected( int ) ) );
    actionFormatStyle->setMenuAccelsEnabled( true );
    updateStyleList();

    actionAllowAutoFormat = new TDEToggleAction( i18n( "Enable Autocorrection" ), 0,
                                               this, TQ_SLOT( slotAllowAutoFormat() ),
                                               actionCollection(), "enable_autocorrection" );
    actionAllowAutoFormat->setCheckedState(i18n("Disable Autocorrection"));

    // ------------------- Actions with a key binding and no GUI item
    new TDEAction( i18n( "Insert Non-Breaking Space" ), CTRL+TQt::Key_Space,
                 this, TQ_SLOT( slotNonbreakingSpace() ), actionCollection(), "nonbreaking_space" );
    new TDEAction( i18n( "Insert Non-Breaking Hyphen" ), CTRL+TQt::SHIFT+TQt::Key_Minus,
                 this, TQ_SLOT( slotNonbreakingHyphen() ), actionCollection(), "nonbreaking_hyphen" );
    new TDEAction( i18n( "Insert Soft Hyphen" ), CTRL+TQt::Key_Minus,
                 this, TQ_SLOT( slotSoftHyphen() ), actionCollection(), "soft_hyphen" );
    new TDEAction( i18n( "Line Break" ), TQt::SHIFT+TQt::Key_Return,
                 this, TQ_SLOT( slotLineBreak() ), actionCollection(), "line_break" );
    new TDEAction( i18n( "Completion" ), TDEStdAccel::shortcut(TDEStdAccel::TextCompletion),
                 this, TQ_SLOT( slotCompletion() ), actionCollection(), "completion" );

    new TDEAction( i18n( "Increase Numbering Level" ), ALT+TQt::Key_Right,
                 this, TQ_SLOT( slotIncreaseNumberingLevel() ), actionCollection(), "increase_numbering_level" );
    new TDEAction( i18n( "Decrease Numbering Level" ), ALT+TQt::Key_Left,
                 this, TQ_SLOT( slotDecreaseNumberingLevel() ), actionCollection(), "decrease_numbering_level" );


    actionInsertComment = new TDEAction( i18n( "Comment..." ), 0,
                                       this, TQ_SLOT( insertComment() ),
                                       actionCollection(), "insert_comment" );
    actionEditComment = new TDEAction( i18n("Edit Comment..."), 0,
                                     this, TQ_SLOT(editComment()),
                                     actionCollection(), "edit_comment");

    actionAddGuideLine = new TDEAction( i18n( "Add Guide Line..."), 0,
                                      this, TQ_SLOT( addGuideLine()),
                                      actionCollection(), "add_guideline");

    actionRemoveComment = new TDEAction( i18n("Remove Comment"), 0,
                                       this, TQ_SLOT(removeComment()),
                                       actionCollection(), "remove_comment");

    actionCopyTextOfComment = new TDEAction( i18n("Copy Text of Comment..."), 0,
                                           this, TQ_SLOT(copyTextOfComment()),
                                           actionCollection(), "copy_text_comment");

    actionConfigureCompletion = new TDEAction( i18n( "&Configure Completion..." ), 0,
                                             this, TQ_SLOT( configureCompletion() ),
                                             actionCollection(), "configure_completion" );

    actionZoomMinus = new TDEAction( i18n( "Zoom Out" ), "zoom-out",0,
                                   this, TQ_SLOT( zoomMinus() ),
                                   actionCollection(), "zoom_minus" );
    actionZoomPlus = new TDEAction( i18n( "Zoom In" ), "zoom-in",0,
                                  this, TQ_SLOT( zoomPlus() ),
                                  actionCollection(), "zoom_plus" );
    actionZoomEntirePage = new TDEAction( i18n( "Zoom Entire Slide" ), 0,
                                        this, TQ_SLOT( zoomEntirePage() ),
                                        actionCollection(), "zoom_entire_page" );

    actionZoomMinus = new TDEAction( i18n( "Zoom Slide Width" ), 0,
                                   this, TQ_SLOT( zoomPageWidth() ),
                                   actionCollection(), "zoom_page_width" );
    actionZoomSelectedObject= new TDEAction( i18n( "Zoom Selected Objects" ), "zoom-fit-best",0,
                                           this, TQ_SLOT( zoomSelectedObject() ),
                                           actionCollection(), "zoom_selected_object" );
    actionZoomPageHeight= new TDEAction( i18n( "Zoom Slide Height" ), 0,
                                       this, TQ_SLOT( zoomPageHeight() ),
                                       actionCollection(), "zoom_page_height" );

    actionZoomAllObject= new TDEAction( i18n( "Zoom All Objects" ), 0,
                                      this, TQ_SLOT( zoomAllObject() ),
                                      actionCollection(), "zoom_all_object" );

    actionFlipHorizontal= new TDEAction( i18n( "Horizontal Flip" ), 0,
                                       this, TQ_SLOT( flipHorizontal() ),
                                       actionCollection(), "horizontal_flip" );

    actionFlipVertical= new TDEAction( i18n( "Vertical Flip" ), 0,
                                     this, TQ_SLOT( flipVertical() ),
                                     actionCollection(), "vertical_flip" );

    actionDuplicateObj = new TDEAction( i18n( "Duplicate Object..." ), 0,
                                      this, TQ_SLOT( duplicateObj() ),
                                      actionCollection(), "duplicate_obj" );

    actionApplyAutoFormat= new TDEAction( i18n( "Apply Autocorrection" ), 0,
                                        this, TQ_SLOT( applyAutoFormat() ),
                                        actionCollection(), "apply_autoformat" );

    actionCreateStyleFromSelection = new TDEAction( i18n( "Create Style From Selection..." ), 0,
                                                  this, TQ_SLOT( createStyleFromSelection()),
                                                  actionCollection(), "create_style" );

    actionCloseObject = new TDEAction( i18n( "Close Object" ), 0,
                                     this, TQ_SLOT( closeObject()),
                                     actionCollection(), "close_object" );


    actionAlignVerticalTop = new TDEToggleAction( i18n( "Align Top" ), 0,
                                                this, TQ_SLOT( alignVerticalTop() ),
                                                actionCollection(), "align_top" );
    actionAlignVerticalTop->setExclusiveGroup( "vertical_alignment" );
    actionAlignVerticalTop->setChecked( true );


    actionAlignVerticalBottom = new TDEToggleAction( i18n( "Align Bottom" ), 0,
                                                   this, TQ_SLOT( alignVerticalBottom() ),
                                                   actionCollection(), "align_bottom" );
    actionAlignVerticalBottom->setExclusiveGroup( "vertical_alignment" );

    actionAlignVerticalCenter = new TDEToggleAction( i18n( "Align Middle" ), 0,
                                                   this, TQ_SLOT( alignVerticalCenter() ),
                                                   actionCollection(), "align_center" );
    actionAlignVerticalCenter->setExclusiveGroup( "vertical_alignment" );


    actionSavePicture= new TDEAction( i18n("Save Picture..."), 0,
                                    this, TQ_SLOT( savePicture() ),
                                    actionCollection(), "save_picture");

    actionAllowBgSpellCheck = new TDEToggleAction( i18n( "Autospellcheck" ), 0,
                                                 this, TQ_SLOT( autoSpellCheck() ),
                                                 actionCollection(), "tool_auto_spellcheck" );

    actionInsertFile= new TDEAction( i18n( "File..." ), 0,
                                   this, TQ_SLOT( insertFile() ),
                                   actionCollection(), "insert_file" );
    actionImportStyle= new TDEAction( i18n( "Import Styles..." ), 0,
                                    this, TQ_SLOT( importStyle() ),
                                    actionCollection(), "import_style" );

    actionSaveBackgroundPicture= new TDEAction( i18n( "Save Background Picture..." ), 0,
                                              this, TQ_SLOT(backgroundPicture() ),
                                              actionCollection(), "save_bgpicture" );
#if 0
    actionInsertDirectCursor = new TDEToggleAction( i18n( "Type Anywhere Cursor" ), 0,
                                                  this, TQ_SLOT( insertDirectCursor() ),
                                                  actionCollection(), "direct_cursor" );
#endif

    actionSpellIgnoreAll = new TDEAction( i18n( "Ignore All" ), 0,
                                        this, TQ_SLOT( slotAddIgnoreAllWord() ),
                                        actionCollection(), "ignore_all" );

    actionAddWordToPersonalDictionary=new TDEAction( i18n( "Add Word to Dictionary" ),0,
                                                   this, TQ_SLOT( addWordToDictionary() ),
                                                   actionCollection(), "add_word_to_dictionary" );
    actionCustomSlideShow = new TDEAction( i18n( "Custom Slide Show..." ), 0,
                                         this, TQ_SLOT( customSlideShow() ),
                                         actionCollection(), "custom_slide_show" );

    actionDisplayObjectFromMasterPage = new TDEToggleAction( i18n( "Hide Object From Slide Master" ), 0,
                                         this, TQ_SLOT( displayObjectFromMasterPage() ),
                                         actionCollection(), "display_object_from_master_page" );
    actionDisplayObjectFromMasterPage->setCheckedState(i18n("Display Object From Slide Master"));


    actionDisplayBackgroundPage = new TDEToggleAction( i18n( "Hide Background" ), 0,
                                         this, TQ_SLOT( displayBackground() ),
                                         actionCollection(), "display_background" );
    actionDisplayBackgroundPage->setCheckedState(i18n("Display Background"));
}

void KPrView::displayObjectFromMasterPage()
{
    bool state=actionDisplayObjectFromMasterPage->isChecked();
    m_canvas->activePage()->setDisplayObjectFromMasterPage( !state );
    KPrDisplayObjectFromMasterPage * cmd =new KPrDisplayObjectFromMasterPage( state ? i18n("Hide Object From Slide Master") : i18n("Display Object From Slide Master"), m_pKPresenterDoc, m_canvas->activePage(), !state);
    m_pKPresenterDoc->addCommand(cmd);
    m_pKPresenterDoc->updateSideBarItem( m_pKPresenterDoc->masterPage() );
}

void KPrView::displayBackground()
{
    bool state=actionDisplayBackgroundPage->isChecked();
    m_canvas->activePage()->setDisplayBackground( !state );
    KPrDisplayBackgroundPage * cmd =new KPrDisplayBackgroundPage( state ? i18n("Hide Background") : i18n("Display Background"), m_pKPresenterDoc, m_canvas->activePage(), !state);
    m_pKPresenterDoc->addCommand(cmd);
    m_pKPresenterDoc->updateSideBarItem( m_pKPresenterDoc->masterPage() );
}

void KPrView::customSlideShow()
{
    KPrCustomSlideShowDia *dlg = new KPrCustomSlideShowDia( this, m_pKPresenterDoc, "custom slide show" );
    dlg->exec();
    delete dlg;
    //clear element
    m_pKPresenterDoc->clearTestCustomSlideShow();
}

void KPrView::textSubScript()
{
    m_canvas->setTextSubScript(actionFormatSub->isChecked());
}

void KPrView::textSuperScript()
{
    m_canvas->setTextSuperScript(actionFormatSuper->isChecked());
}

void KPrView::decreaseFontSize()
{
    m_canvas->setDecreaseFontSize();
}

void KPrView::increaseFontSize()
{
    m_canvas->setIncreaseFontSize();
}

void KPrView::objectSelectedChanged()
{

    bool state=m_canvas->isOneObjectSelected();
    bool headerfooterselected=false;

    if (m_canvas->numberOfObjectSelected()==1)
    {
        KPrObject *obj=m_canvas->getSelectedObj();
        //disable this action when we select a header/footer
        if (obj==m_pKPresenterDoc->header() || obj==m_pKPresenterDoc->footer())
            headerfooterselected=true;
        else
            headerfooterselected=false;
    }
    actionScreenAssignEffect->setEnabled(state && !headerfooterselected);
    actionEditDelete->setEnabled(state && !headerfooterselected);
    actionEditCut->setEnabled(state && !headerfooterselected);

    KPrObjectProperties objectProperties( m_canvas->activePage()->getSelectedObjects() );
    int flags = objectProperties.getPropertyFlags();
    // only button when object support them or none object is selected
    actionBrushColor->setEnabled( !state || ( flags & KPrObjectProperties::PtBrush ) );
    actionExtraLineBegin->setEnabled( !state || ( flags & KPrObjectProperties::PtLineEnds ) );
    actionExtraLineEnd->setEnabled( !state || ( flags & KPrObjectProperties::PtLineEnds ) );
    actionExtraPenWidth->setEnabled( !state || ( flags & KPrObjectProperties::PtPenWidth ) );

    actionExtraProperties->setEnabled(state && !headerfooterselected);
    actionExtraRotate->setEnabled(state && !headerfooterselected);
    actionExtraShadow->setEnabled(state && !m_canvas->haveASelectedPartObj() && !headerfooterselected);

    actionExtraAlignObjsPopup->setEnabled(state && !headerfooterselected);
    actionExtraGroup->setEnabled(state && m_canvas->numberOfObjectSelected()>1);
    actionExtraUnGroup->setEnabled(state && m_canvas->haveASelectedGroupObj());

    bool enableAlign = m_canvas->canMoveOneObject() && state && !headerfooterselected;

    actionExtraAlignObjLeft->setEnabled(enableAlign);
    actionExtraAlignObjCenterH->setEnabled(enableAlign);
    actionExtraAlignObjRight->setEnabled(enableAlign);
    actionExtraAlignObjTop->setEnabled(enableAlign);
    actionExtraAlignObjCenterV->setEnabled(enableAlign);
    actionExtraAlignObjBottom->setEnabled(enableAlign);
    //actionEditDelete->setEnabled(state);
    int nbobj=m_canvas->numberOfObjectSelected();
    actionDuplicateObj->setEnabled(state && (nbobj>=1));

    state = state && (nbobj==1);
    actionExtraArrangePopup->setEnabled(state);

    actionExtraRaise->setEnabled( nbobj > 0 );
    actionExtraLower->setEnabled( nbobj > 0 );
    actionExtraBringForward->setEnabled( nbobj > 0 );
    actionExtraSendBackward->setEnabled( nbobj > 0 );

    //actionExtraConfigPicture->setEnabled( state && m_canvas->haveASelectedPixmapObj() );
    //actionPenColor->setEnabled(state);
    //actionExtraPenStyle->setEnabled(state);
    //actionExtraPenWidth->setEnabled(state);

    actionApplyAutoFormat->setEnabled(m_canvas->oneObjectTextExist());
    slotObjectEditChanged();
}

void KPrView::backOk( KPrBackDialog* backDia, bool takeGlobal )
{
    KPrPage *page=m_canvas->activePage();
    KPrSetBackCmd *setBackCmd = new KPrSetBackCmd( i18n( "Set Background" ),
                                             backDia->getBackGround(),
                                             page->background()->getBackGround(),
                                             backDia->useMasterBackground(),
                                             takeGlobal, m_pKPresenterDoc,page);
    setBackCmd->execute();
    m_pKPresenterDoc->addCommand( setBackCmd );
}

void KPrView::afChooseOk( const TQString & c )
{
    TQFileInfo fileInfo( c );
    TQString fileName = locate( "autoforms",
                               fileInfo.dirPath( false ) + "/" + fileInfo.baseName() + ".atf",
                               KPrFactory::global() );

    deSelectAllObjects();
    m_canvas->setToolEditMode( INS_AUTOFORM );
    m_canvas->setAutoForm( fileName );
}

void KPrView::slotAfchooseCanceled()
{
    setTool( TEM_MOUSE );
}

void KPrView::propertiesOk()
{
    KCommand *cmd = m_propertyEditor->getCommand();

    if ( cmd )
    {
        cmd->execute();
        kPresenterDoc()->addCommand( cmd );
    }
}

void KPrView::pgConfOk()
{
    TQValueList<bool> selectedSlides;
    for( unsigned i = 0; i < kPresenterDoc()->pageList().count(); i++ ) {
        selectedSlides.append( kPresenterDoc()->pageList().at( i )->isSlideSelected() );
    }

    KPrPgConfCmd *pgConfCmd = new KPrPgConfCmd( i18n( "Configure Slide Show" ),
                                          pgConfDia->getManualSwitch(), pgConfDia->getInfiniteLoop(),
                                          pgConfDia->getShowEndOfPresentationSlide(), pgConfDia->getPresentationDuration(), pgConfDia->getPen(),
                                          pgConfDia->getSelectedSlides(),
                                          pgConfDia->presentationName(),
                                          kPresenterDoc()->spManualSwitch(),
                                          kPresenterDoc()->spInfiniteLoop(),
                                          kPresenterDoc()->spShowEndOfPresentationSlide(),
                                          kPresenterDoc()->presentationDuration(),
                                          kPresenterDoc()->presPen(),
                                          selectedSlides,
                                          kPresenterDoc()->presentationName(),
                                          kPresenterDoc() );
    pgConfCmd->execute();
    kPresenterDoc()->addCommand( pgConfCmd );

    TQPtrListIterator<KPrPage> it( kPresenterDoc()->pageList() );
    for ( ; it.current(); ++it )
        updateSideBarItem( it.current() );
}


void KPrView::rotateOk()
{
    float _newAngle=rotateDia->angle();

    KCommand *cmd=m_canvas->activePage()->rotateSelectedObjects(_newAngle);
    if( cmd )
        kPresenterDoc()->addCommand( cmd );
}

void KPrView::shadowOk()
{
    KCommand *cmd=m_canvas->activePage()->shadowObj(shadowDia->shadowDirection(),
                                                    shadowDia->shadowDistance(),
                                                    shadowDia->shadowColor());
    if( cmd)
        kPresenterDoc()->addCommand( cmd );
}

unsigned int KPrView::getCurrPgNum() const
{
    return currPg + 1;
}


void KPrView::recalcCurrentPageNum()
{
    KPrPage *activePage = m_canvas->activePage();

    TQPtrList<KPrPage> pageList( m_pKPresenterDoc->pageList() );

    int pos = pageList.findRef( activePage );

    if ( pos != -1 )
    {
        currPg = pos;
    }
    else
    {
        kdDebug(33001) << "KPrView::recalcCurrentPageNum: activePage not found" << endl;
        currPg = 0;
    }

    if( sidebar )
        sidebar->setCurrentPage( currPg );
}


void KPrView::scrollH( int value )
{
    if ( !presStarted ) {
        m_canvas->scrollX( value );
        if ( h_ruler )
            h_ruler->setOffset( value, 0 );
    }
}

void KPrView::scrollV( int value )
{
    if ( !presStarted ) {
        m_canvas->scrollY( value );
        if ( v_ruler )
            v_ruler->setOffset( 0, value );
    }
}

void KPrView::fontChanged( const TQFont &font )
{
    tbFont.setFamily( font.family() );
    tbFont.setBold( font.bold() );
    tbFont.setItalic( font.italic() );
    tbFont.setUnderline( font.underline() );
    tbFont.setPointSize( font.pointSize() );

    actionTextFontFamily->setFont( tbFont.family() );
    actionTextFontSize->setFontSize( tbFont.pointSize() );
    actionTextBold->setChecked( tbFont.bold() );
    actionTextItalic->setChecked( tbFont.italic() );
    actionTextUnderline->setChecked( tbFont.underline() );
}

void KPrView::colorChanged( const TQColor &color )
{
    tbColor = color;
    actionTextColor->setEnabled( true );
    actionTextColor->setCurrentColor( tbColor );
}

void KPrView::alignChanged( int align )
{
    if ( align != tbAlign ) {
        tbAlign = align;
        if ( ( align & AlignLeft ) == AlignLeft ) {
            actionTextAlignLeft->blockSignals( true );
            actionTextAlignLeft->setChecked( true );
            actionTextAlignLeft->blockSignals( false );
        } else if ( ( align & AlignHCenter ) == AlignHCenter ||
                    ( align & AlignCenter ) == AlignCenter ) {
            actionTextAlignCenter->blockSignals( true );
            actionTextAlignCenter->setChecked( true );
            actionTextAlignCenter->blockSignals( false );
        } else if ( ( align & AlignRight ) == AlignRight ) {
            actionTextAlignRight->blockSignals( true );
            actionTextAlignRight->setChecked( true );
            actionTextAlignRight->blockSignals( false );
        } else if ( (align & AlignJustify ) == AlignJustify ) {
            actionTextAlignBlock->blockSignals( true );
            actionTextAlignBlock->setChecked( true );
            actionTextAlignBlock->blockSignals( false );
        }
    }
}

void KPrView::changePicture( const TQString & filename )
{
    TQStringList mimetypes;
    mimetypes += KImageIO::mimeTypes( KImageIO::Reading );
    mimetypes += KoPictureFilePreview::clipartMimeTypes();

    KFileDialog fd( filename, TQString(), this, 0, true );
    fd.setCaption(i18n("Select New Picture"));
    fd.setMimeFilter( mimetypes );
    fd.setPreviewWidget( new KoPictureFilePreview( &fd ) );

    KURL url;
    if ( fd.exec() == TQDialog::Accepted )
        url = fd.selectedURL();

    if( url.isEmpty() || !url.isValid())
        return;

    m_canvas->changePicture( url, m_canvas );
}

void KPrView::resizeEvent( TQResizeEvent *e )
{
    if ( !presStarted )
        TQWidget::resizeEvent( e );

    TQSize s = e ? e->size() : size();
    splitter->setGeometry( 0, 0, s.width(), s.height() );
}

void KPrView::reorganize()
{
    if (m_bShowGUI ) {

        horz->show();
        vert->show();
        pgNext->show();
        pgPrev->show();

        if(kPresenterDoc()->showRuler())
        {
            int hSpace = v_ruler->minimumSizeHint().width();
            int vSpace = h_ruler->minimumSizeHint().height();

            m_canvas->move( hSpace, vSpace );
            if ( h_ruler )
            {
                h_ruler->show();
                h_ruler->setGeometry( hSpace, 0, m_canvas->width(), vSpace );
            }
            if (v_ruler )
            {
                v_ruler->show();
                v_ruler->setGeometry( 0, vSpace, hSpace, m_canvas->height() );
            }
            if(getTabChooser())
            {
                getTabChooser()->setGeometry(0, 0, hSpace, vSpace);
                getTabChooser()->show();
            }
        }
        else
        {
            m_canvas->move( 0, 0 );
            if ( h_ruler )
                h_ruler->hide();
            if ( v_ruler )
                v_ruler->hide();
            getTabChooser()->hide();
        }

        if( statusBar())
        {
            if(m_pKPresenterDoc->showStatusBar())
                statusBar()->show();
            else
                statusBar()->hide();
        }

        setRanges();
    }
    else
    {
        horz->hide();
        vert->hide();
        pgNext->hide();
        pgPrev->hide();
        h_ruler->hide();
        v_ruler->hide();
        getTabChooser()->hide();
        m_canvas->move( 0, 0 );
    }
}

void PageBase::resizeEvent( TQResizeEvent *e )
{
    if ( !view->presStarted )
        TQWidget::resizeEvent( e );

    TQSize s = e ? e->size() : size();

    if ( view->m_bShowGUI ) {
        int hSpace = view->getVRuler()->minimumSizeHint().width();
        int vSpace = view->getVRuler()->minimumSizeHint().height();

        view->m_canvas->resize( s.width() - ( hSpace + 16 ), s.height() - ( vSpace + 16 ) );
        view->vert->setGeometry( s.width() - 16, 0, 16, s.height() - 32 );
        view->pgPrev->setGeometry( s.width() - 15, s.height() - 32, 15, 16 );
        view->pgNext->setGeometry( s.width() - 15, s.height() - 16, 15, 16 );
        view->horz->setGeometry( 0, s.height() - 16, s.width() - 16, 16 );
    }
    else
    {
        view->m_canvas->move( 0, 0 );
        view->m_canvas->resize( s.width(), s.height() );
    }
    view->reorganize();
}

void KPrView::dragEnterEvent( TQDragEnterEvent *e )
{
    TQApplication::sendEvent( m_canvas, e );
}

void KPrView::dragMoveEvent( TQDragMoveEvent *e )
{
    TQApplication::sendEvent( m_canvas, e );
}

void KPrView::dragLeaveEvent( TQDragLeaveEvent *e )
{
    TQApplication::sendEvent( m_canvas, e );
}

void KPrView::dropEvent( TQDropEvent *e )
{
    TQApplication::sendEvent( m_canvas, e );
}

void KPrView::getPageMouseWheelEvent( TQWheelEvent *e )
{
    TQApplication::sendEvent( vert, e );
}

void KPrView::keyPressEvent( TQKeyEvent *e )
{
    if ( e->key() == TQt::Key_Delete && !m_canvas->currentTextObjectView() )
        editDelete();
    else
        TQApplication::sendEvent( m_canvas, e );
}

void KPrView::doAutomaticScreenPres()
{
    if ( m_autoPresStop ) // A user pushed Escape key or clicked "Exit presentation" menu.
        return;
    else if ( m_autoPresRestart && kPresenterDoc()->spInfiniteLoop() ) {
        m_autoPresRestart = false;
        m_canvas->presGotoFirstPage(); // return to first page.
    }
    else
        screenNext();
}

void KPrView::updateReadWrite( bool readwrite )
{
    // First disable or enable everything
    TQValueList<TDEAction*> actions = actionCollection()->actions();
    // Also grab actions from the document
    actions += m_pKPresenterDoc->actionCollection()->actions();
    TQValueList<TDEAction*>::ConstIterator aIt = actions.begin();
    TQValueList<TDEAction*>::ConstIterator aEnd = actions.end();
    for (; aIt != aEnd; ++aIt )
        (*aIt)->setEnabled( readwrite );


    if ( !readwrite )
    {
        // Readonly -> re-enable a few harmless actions
        refreshPageButton();
        actionViewFormattingChars->setEnabled( true );
        actionViewZoom->setEnabled( true );
        actionEditFind->setEnabled( true );
        actionEditFindNext->setEnabled( true );
        actionEditFindPrevious->setEnabled( true );
        actionEditReplace->setEnabled( true );
        actionEditSelectAll->setEnabled( true );
        actionEditDeSelectAll->setEnabled( true );
    }
    else
    {
        refreshPageButton();
        objectSelectedChanged();

        refreshCustomMenu();

        // Correctly enable or disable undo/redo actions again
        m_pKPresenterDoc->commandHistory()->updateActions();
        updateSideBarMenu();
    }

}

void KPrView::setupPopupMenus()
{
    // create right button line begin
    rb_lbegin = new TQPopupMenu();
    TQ_CHECK_PTR( rb_lbegin );
    rb_lbegin->insertItem( KPBarIcon("line_normal_begin" ), this, TQ_SLOT( extraLineBeginNormal() ) );
    rb_lbegin->insertSeparator();
    rb_lbegin->insertItem( KPBarIcon("line_arrow_begin" ), this, TQ_SLOT( extraLineBeginArrow() ) );
    rb_lbegin->insertSeparator();
    rb_lbegin->insertItem( KPBarIcon("line_rect_begin" ), this, TQ_SLOT( extraLineBeginRect() ) );
    rb_lbegin->insertSeparator();
    rb_lbegin->insertItem( KPBarIcon("line_circle_begin" ), this, TQ_SLOT( extraLineBeginCircle() ) );
    rb_lbegin->insertSeparator();
    rb_lbegin->insertItem( KPBarIcon("line_line_arrow_begin" ), this, TQ_SLOT( extraLineBeginLineArrow() ) );
    rb_lbegin->insertSeparator();
    rb_lbegin->insertItem( KPBarIcon("line_dimension_line_begin" ), this, TQ_SLOT( extraLineBeginDimensionLine() ) );
    rb_lbegin->insertSeparator();
    rb_lbegin->insertItem( KPBarIcon("line_double_arrow_begin" ), this, TQ_SLOT( extraLineBeginDoubleArrow() ) );
    rb_lbegin->insertSeparator();
    rb_lbegin->insertItem( KPBarIcon("line_double_line_arrow_begin" ), this, TQ_SLOT( extraLineBeginDoubleLineArrow() ) );
    rb_lbegin->setMouseTracking( true );
    rb_lbegin->setCheckable( false );

    // create right button line end
    rb_lend = new TQPopupMenu();
    TQ_CHECK_PTR( rb_lend );
    rb_lend->insertItem( KPBarIcon("line_normal_end" ), this, TQ_SLOT( extraLineEndNormal() ) );
    rb_lend->insertSeparator();
    rb_lend->insertItem( KPBarIcon("line_arrow_end" ), this, TQ_SLOT( extraLineEndArrow() ) );
    rb_lend->insertSeparator();
    rb_lend->insertItem( KPBarIcon("line_rect_end" ), this, TQ_SLOT( extraLineEndRect() ) );
    rb_lend->insertSeparator();
    rb_lend->insertItem( KPBarIcon("line_circle_end" ), this, TQ_SLOT( extraLineEndCircle() ) );
    rb_lend->insertSeparator();
    rb_lend->insertItem( KPBarIcon("line_line_arrow_end" ), this, TQ_SLOT( extraLineEndLineArrow() ) );
    rb_lend->insertSeparator();
    rb_lend->insertItem( KPBarIcon("line_dimension_line_end" ), this, TQ_SLOT( extraLineEndDimensionLine() ) );
    rb_lend->insertSeparator();
    rb_lend->insertItem( KPBarIcon("line_double_arrow_end" ), this, TQ_SLOT( extraLineEndDoubleArrow() ) );
    rb_lend->insertSeparator();
    rb_lend->insertItem( KPBarIcon("line_double_line_arrow_end" ), this, TQ_SLOT( extraLineEndDoubleLineArrow() ) );
    rb_lend->setMouseTracking( true );
    rb_lend->setCheckable( false );

    // create arrange-objects popup
    m_arrangeObjectsPopup = new TQPopupMenu();
    TQ_CHECK_PTR(m_arrangeObjectsPopup);
    m_arrangeObjectsPopup->insertItem(KPBarIcon("lower"), this, TQ_SLOT(extraLower()));
    m_arrangeObjectsPopup->insertSeparator();
    m_arrangeObjectsPopup->insertItem(KPBarIcon("send_backward"), this, TQ_SLOT(extraSendBackward()));
    m_arrangeObjectsPopup->insertSeparator();
    m_arrangeObjectsPopup->insertItem(KPBarIcon("bring_forward"), this, TQ_SLOT(extraBringForward()));
    m_arrangeObjectsPopup->insertSeparator();
    m_arrangeObjectsPopup->insertItem(KPBarIcon("raise"), this, TQ_SLOT(extraRaise()));
    m_arrangeObjectsPopup->setMouseTracking(true);
    m_arrangeObjectsPopup->setCheckable(false);

    // create insert-line popup
    actionToolsLinePopup->insert(actionToolsLine);
    actionToolsLinePopup->insert(actionToolsFreehand);
    actionToolsLinePopup->insert(actionToolsPolyline);
    actionToolsLinePopup->insert(actionToolsCubicBezierCurve);
    actionToolsLinePopup->insert(actionToolsQuadricBezierCurve);

    // create insert-shape popup
    actionToolsShapePopup->insert(actionToolsRectangle);
    actionToolsShapePopup->insert(actionToolsCircleOrEllipse);
    actionToolsShapePopup->insert(actionToolsPie);
    actionToolsShapePopup->insert(actionToolsConvexOrConcavePolygon);

    // create insert-closed-line popup
    actionToolsClosedLinePopup->insert(actionToolsClosedFreehand);
    actionToolsClosedLinePopup->insert(actionToolsClosedPolyline);
    actionToolsClosedLinePopup->insert(actionToolsClosedQuadricBezierCurve);
    actionToolsClosedLinePopup->insert(actionToolsClosedCubicBezierCurve);

    actionExtraAlignObjsPopup->insert( actionExtraAlignObjLeft );
    actionExtraAlignObjsPopup->insert( actionExtraAlignObjCenterV );
    actionExtraAlignObjsPopup->insert( actionExtraAlignObjRight );
    actionExtraAlignObjsPopup->insert( actionExtraAlignObjTop );
    actionExtraAlignObjsPopup->insert( actionExtraAlignObjCenterH );
    actionExtraAlignObjsPopup->insert( actionExtraAlignObjBottom );

    actionExtraArrangePopup->insert( actionExtraLower );
    actionExtraArrangePopup->insert( actionExtraSendBackward );
    actionExtraArrangePopup->insert( actionExtraBringForward );
    actionExtraArrangePopup->insert( actionExtraRaise );
}

void KPrView::setupScrollbars()
{
    vert = new TQScrollBar( TQt::Vertical, pageBase );
    horz = new TQScrollBar( TQt::Horizontal, pageBase );
    vert->show();
    horz->show();
    TQObject::connect( vert, TQ_SIGNAL( valueChanged( int ) ), this, TQ_SLOT( scrollV( int ) ) );
    TQObject::connect( horz, TQ_SIGNAL( valueChanged( int ) ), this, TQ_SLOT( scrollH( int ) ) );
    vert->setValue(vert->maxValue());
    horz->setValue(horz->maxValue());
    vert->setValue(vert->minValue());
    horz->setValue(horz->minValue());
    pgNext = new TQToolButton( pageBase );
    pgNext->setPixmap( TQPixmap( pagedown_xpm ) );
    pgNext->setAutoRepeat( true );
    TQToolTip::add( pgNext, i18n( "Next slide" ) );
    connect( pgNext, TQ_SIGNAL( clicked() ), this, TQ_SLOT( nextPage() ) );
    pgPrev = new TQToolButton( pageBase );
    pgPrev->setPixmap( TQPixmap( pageup_xpm ) );
    pgPrev->setAutoRepeat( true );
    TQToolTip::add( pgPrev, i18n( "Previous slide" ) );
    connect( pgPrev, TQ_SIGNAL( clicked() ), this, TQ_SLOT( prevPage() ) );
}

void KPrView::setupRulers()
{
    tabChooser = new KoTabChooser( pageBase, KoTabChooser::TAB_ALL );
    tabChooser->setReadWrite(kPresenterDoc()->isReadWrite());
    h_ruler = new KoRuler( pageBase, m_canvas, TQt::Horizontal, kPresenterDoc()->pageLayout(),
                           KoRuler::F_INDENTS | KoRuler::F_TABS, kPresenterDoc()->unit(), tabChooser );
    h_ruler->changeFlags(0);

    h_ruler->setReadWrite(kPresenterDoc()->isReadWrite());
    v_ruler = new KoRuler( pageBase, m_canvas, TQt::Vertical, kPresenterDoc()->pageLayout(), 0, kPresenterDoc()->unit() );
    v_ruler->setReadWrite(kPresenterDoc()->isReadWrite());

    int hSpace = v_ruler->minimumSizeHint().width();
    int vSpace = h_ruler->minimumSizeHint().height();

    m_canvas->resize( m_canvas->width() - hSpace, m_canvas->height() - vSpace );
    m_canvas->move( hSpace, vSpace );
    h_ruler->setGeometry( hSpace, 0, m_canvas->width(), vSpace );
    v_ruler->setGeometry( 0, vSpace, hSpace, m_canvas->height() );

    TQObject::connect( h_ruler, TQ_SIGNAL( unitChanged( KoUnit::Unit ) ),
                      this, TQ_SLOT( unitChanged( KoUnit::Unit ) ) );
    TQObject::connect( h_ruler, TQ_SIGNAL( newPageLayout( const KoPageLayout & ) ),
                      this, TQ_SLOT( newPageLayout( const KoPageLayout & ) ) );

    connect( h_ruler, TQ_SIGNAL( doubleClicked() ), this,
             TQ_SLOT( slotHRulerDoubleClicked() ) );
    connect( h_ruler, TQ_SIGNAL( doubleClicked(double) ), this,
             TQ_SLOT( slotHRulerDoubleClicked(double) ) );

    TQObject::connect( v_ruler, TQ_SIGNAL( unitChanged( KoUnit::Unit ) ),
                      this, TQ_SLOT( unitChanged( KoUnit::Unit ) ) );
    TQObject::connect( v_ruler, TQ_SIGNAL( newPageLayout( const KoPageLayout & ) ),
                      this, TQ_SLOT( newPageLayout( const KoPageLayout & ) ) );
    TQObject::connect( v_ruler, TQ_SIGNAL( doubleClicked() ),
                      this, TQ_SLOT( openPageLayoutDia() ) );

    connect( h_ruler, TQ_SIGNAL( newLeftIndent( double ) ), this, TQ_SLOT( newLeftIndent( double ) ) );
    connect( h_ruler, TQ_SIGNAL( newFirstIndent( double ) ), this, TQ_SLOT( newFirstIndent( double ) ) );
    connect( h_ruler, TQ_SIGNAL( newRightIndent( double ) ), this, TQ_SLOT( newRightIndent( double ) ) );
}

void KPrView::unitChanged( KoUnit::Unit u )
{
    m_pKPresenterDoc->setUnit( u );
}

void KPrView::setRanges()
{
    if ( vert && horz && m_canvas && m_pKPresenterDoc ) {
        vert->setSteps( 10, m_canvas->height() );
        vert->setRange( 0, TQMAX( 0, m_canvas->activePage()->getZoomPageRect().height()  - m_canvas->height() ) );
        horz->setSteps( 10, m_canvas->width() );
        horz->setRange( 0, TQMAX( 0, m_canvas->activePage()->getZoomPageRect().width() + 16 - m_canvas->width() ) );
    }
}

void KPrView::skipToPage( int num )
{
    setEditMaster( false );
    if ( num < 0 || num > static_cast<int>( m_pKPresenterDoc->getPageNums() ) - 1 || !m_canvas )
        return;
    m_canvas->exitEditMode();
    currPg = num;
    emit currentPageChanged( currPg );
    if( sidebar )
        sidebar->setCurrentPage( currPg );
    KPrPage* page = m_pKPresenterDoc->pageList().at( currPg );
    m_canvas->setActivePage( page );
    // don't scroll before new active page is set,
    // the page active until then might have been deleted
    vert->setValue( 0 );
    horz->setValue( 0 );
    if ( notebar ) {
        TQString text = page->noteText( );
        notebar->setCurrentNoteText( text );
    }
    refreshPageButton();
    //(Laurent) deselect object when we change page.
    //otherwise you can change object properties on other page
    deSelectAllObjects();
    m_pKPresenterDoc->repaint( false );

    m_pKPresenterDoc->displayActivePage( page );
}

void KPrView::refreshPageButton()
{
    bool state = (currPg > 0);
    pgPrev->setEnabled( state );
    actionScreenFirst->setEnabled(state);
    actionScreenPrev->setEnabled(state);
    state=(currPg < (int)m_pKPresenterDoc->getPageNums() - 1);
    pgNext->setEnabled( state );
    actionScreenLast->setEnabled(state);
    actionScreenNext->setEnabled(state);
    updateHeaderFooterButton();
    updateDisplayObjectMasterPageButton();
    updateDisplayBackgroundButton();
}

void KPrView::showObjectRect( const KPrObject * object )
{
    TQRect r = zoomHandler()->zoomRect( object->getRealRect() );
    if ( ! TQRect( getCanvas()->diffx(),
                  getCanvas()->diffy(),
                  getCanvas()->width(),
                  getCanvas()->height() ).contains( r ) )
    {
        int x = r.left() - 20;
        int y = r.top() - 20;
        if ( x < 0 )
        {
            x = 0;
        }
        if ( y < 0 )
        {
            y = 0;
        }
        horz->setValue( x );
        vert->setValue( y );
    }
}

void KPrView::setTool( ToolEditMode toolEditMode )
{
    switch ( toolEditMode ) {
    case TEM_MOUSE:
        actionToolsMouse->setChecked( true );
        break;
    case TEM_ROTATE:
        actionToolsRotate->setChecked( true );
        break;
    case TEM_ZOOM:
        actionToolsZoom->setChecked( true );
        break;
    case INS_LINE:
        actionToolsLine->setChecked( true );
        break;
    case INS_RECT:
        actionToolsRectangle->setChecked( true );
        break;
    case INS_ELLIPSE:
        actionToolsCircleOrEllipse->setChecked( true );
        break;
    case INS_PIE:
        actionToolsPie->setChecked( true );
        break;
    case INS_DIAGRAMM:
        actionToolsDiagramm->setChecked( true );
        break;
    case INS_TABLE:
        actionToolsTable->setChecked( true );
        break;
    case INS_FORMULA:
        actionToolsFormula->setChecked( true );
        break;
    case INS_TEXT:
        actionToolsText->setChecked( true );
        break;
    case INS_AUTOFORM:
        actionToolsAutoform->setChecked( true );
        break;
    default: // Shut up gcc -Wall
        break; // Shut up gcc 3.x
    }
}

void KPrView::setRulerMouseShow( bool _show )
{
    v_ruler->showMousePos( _show );
    h_ruler->showMousePos( _show );
}

void KPrView::setRulerMousePos( int mx, int my )
{
    v_ruler->setMousePos( mx, my );
    h_ruler->setMousePos( mx, my );
}

void KPrView::enableWebPres()
{
}

void KPrView::enableMSPres()
{
}

bool KPrView::doubleClickActivation() const
{
    return true;
}

TQWidget* KPrView::canvas() const
{
    return m_canvas;
}

int KPrView::canvasXOffset() const
{
    return m_canvas->diffx();
}

int KPrView::canvasYOffset() const
{
    return m_canvas->diffy();
}

void KPrView::setCanvasXOffset( int _x )
{
    m_canvas->setDiffX( _x );
}

void KPrView::setCanvasYOffset( int _y )
{
    m_canvas->setDiffY( _y );
}

int KPrView::getCurrentPresPage() const
{
    if ( !presStarted )
        return -1;

    return m_canvas->presPage();
}

int KPrView::getCurrentPresStep() const
{
    if ( !presStarted )
        return -1;

    return m_canvas->presStep();
}

int KPrView::getPresStepsOfPage() const
{
    if ( !presStarted )
        return -1;

    return m_canvas->numPresSteps();
}

int KPrView::getNumPresPages() const
{
    if ( !presStarted )
        return -1;

    return m_canvas->numPresPages();
}

bool KPrView::gotoPresPage( int pg )
{
    if ( !presStarted )
        return false;

    m_canvas->gotoPage( pg );
    return true;
}

void KPrView::nextPage()
{
    // don't move when on master
    if ( m_editMaster )
        return;

    if ( currPg >= (int)m_pKPresenterDoc->getPageNums() - 1 )
        return;

    //kdDebug(33001)<<"currPg :"<<currPg<<"m_pKPresenterDoc->getPageNums() :"<<m_pKPresenterDoc->getPageNums()<<endl;
    skipToPage( currPg+1 );
}

void KPrView::prevPage()
{
    // don't move when on master
    if ( m_editMaster )
        return;

    if ( currPg == 0 )
        return;
    skipToPage( currPg-1 );
}

void KPrView::updateSideBar()
{
    if ( sidebar )
    {
        sidebar->blockSignals( true );
        sidebar->thumbBar()->uptodate = false;
        sidebar->outline()->rebuildItems();
        sidebar->thumbBar()->rebuildItems();
        sidebar->blockSignals( false );
    }
}

void KPrView::updateSideBarItem( KPrPage * page )
{
    if ( sidebar )
        sidebar->updateItem( page );
    m_pKPresenterDoc->recalcVariables( VT_STATISTIC );
}

void KPrView::addSideBarItem( int pos )
{
    if ( sidebar )
    {
        sidebar->blockSignals( true );
        sidebar->addItem( pos );
        sidebar->blockSignals( false );
    }
}

void KPrView::moveSideBarItem( int oldPos, int newPos )
{
    if ( sidebar )
    {
        sidebar->blockSignals( true );
        sidebar->moveItem( oldPos, newPos );
        sidebar->blockSignals( false );
    }
}

void KPrView::removeSideBarItem( int pos )
{
    if ( sidebar )
    {
        sidebar->blockSignals( true );
        sidebar->removeItem( pos );
        sidebar->blockSignals( false );
    }
}

void KPrView::updatePageInfo()
{
    if (m_sbPageLabel)
        m_sbPageLabel->setText( TQString(" ") +
                                i18n("Slide %1/%2").arg(getCurrPgNum()).arg(m_pKPresenterDoc->getPageNums())+
                                TQString(" ") );
}

void KPrView::updateObjectStatusBarItem()
{
    KStatusBar * sb = statusBar();
    int nbObjects = m_canvas->objNums();

    if ( m_pKPresenterDoc->showStatusBar() && sb && nbObjects > 0 ) {
        int nbSelected = m_canvas->numberOfObjectSelected();

        if (nbSelected == 1) {
            KoUnit::Unit unit = m_pKPresenterDoc->unit();
            //TQString unitName = m_pKPresenterDoc->unitName();
            KPrObject * obj = m_canvas->getSelectedObj();
            KoSize size = obj->getSize();
            m_sbObjectLabel->setText( ' ' + i18n( "Statusbar info", "%1: %2, %3 - %4, %5 (width: %6, height: %7)" )
                    .arg( /*frame->frameSet()->name()*/obj->getObjectName() )
                    .arg( KoUnit::toUserStringValue( obj->getOrig().x(), unit ) )
                    .arg( KoUnit::toUserStringValue( obj->getOrig().y() , unit) )
                    .arg( KoUnit::toUserStringValue( obj->getOrig().x() + size.width(), unit ) )
                    .arg( KoUnit::toUserStringValue( obj->getOrig().y() + size.height(), unit ) )
                    .arg( KoUnit::toUserStringValue( size.width(), unit ) )
                    .arg( KoUnit::toUserStringValue( size.height(), unit ) ) );
        }
        else
            m_sbObjectLabel->setText( i18n("1 object selected", "%n objects selected", nbSelected) );
    }
    else if ( sb && m_sbObjectLabel )
        m_sbObjectLabel->setText( TQString() );
}

void KPrView::pageNumChanged()
{
    updatePageInfo();
}

void KPrView::viewShowSideBar()
{
    if ( !sidebar )
        return;
    if ( sidebar->isVisible() )
        sidebar->hide();
    else
        sidebar->show();
}

void KPrView::viewShowNoteBar()
{
    if ( !notebar )
        return;
    if ( notebar->isVisible() )
        notebar->hide();
    else
        notebar->show();
}

void KPrView::viewSlideMaster()
{
    bool state =  actionViewSlideMaster->isChecked();
    setEditMaster( state );
    //not activate action when slide master is displaying
    state = !state;
    m_actionExtraHeader->setEnabled( state );
    m_actionExtraFooter->setEnabled( state );
    actionDisplayBackgroundPage->setEnabled( state );
    actionDisplayObjectFromMasterPage->setEnabled( state );

}

void KPrView::setEditMaster( bool editMaster )
{
    if ( m_editMaster != editMaster )
    {
        m_canvas->exitEditMode();
        m_canvas->deSelectAllObj();
        m_editMaster = editMaster;
        if ( m_editMaster )
        {
            m_canvas->setActivePage( m_pKPresenterDoc->masterPage() );
            pgPrev->setEnabled( false );
            actionScreenFirst->setEnabled( false );
            actionScreenPrev->setEnabled( false );
            pgNext->setEnabled( false  );
            actionScreenLast->setEnabled( false );
            actionScreenNext->setEnabled( false );
            actionEditCopyPage->setEnabled( false );
            actionEditDuplicatePage->setEnabled( false );
            actionEditDelPage->setEnabled( false );
            actionViewSlideMaster->setChecked( true );
        }
        else
        {
            m_canvas->setActivePage( m_pKPresenterDoc->pageList().at( currPg ) );
            actionEditCopyPage->setEnabled( true );
            actionEditDuplicatePage->setEnabled( true );
            actionEditDelPage->setEnabled( m_pKPresenterDoc->getPageNums() > 1 );
            actionViewSlideMaster->setChecked( false );
            refreshPageButton();
        }
        if ( sidebar )
            sidebar->setViewMasterPage( m_editMaster );
        updateNoteBarText();
        m_canvas->repaint( false );
    }
}

void KPrView::updateNoteBarText()
{
    if ( notebar ) {
        if ( m_editMaster )
        {
            TQString text = m_pKPresenterDoc->masterPage()->noteText();
            notebar->setCurrentNoteText( text );
        }
        else
        {
            TQString text = m_pKPresenterDoc->pageList().at( currPg )->noteText( );
            notebar->setCurrentNoteText( text );
        }
    }
}

void KPrView::openPopupMenuMenuPage( const TQPoint & _point )
{
    if(!koDocument()->isReadWrite() || !factory())
        return;
    TQPtrList<TDEAction> actionList= TQPtrList<TDEAction>();
    TDEActionSeparator *separator=new TDEActionSeparator();
    switch( m_canvas->activePage()->getBackType())
    {
    case BT_COLOR:
    case BT_BRUSH:
        break;
    case BT_PICTURE:
    case BT_CLIPART:
        actionList.append(separator);
        actionList.append(actionSaveBackgroundPicture);
        break;
    }
    if ( actionList.count()>0)
        plugActionList( "picture_action", actionList );
    m_mousePos = m_canvas->mapFromGlobal( _point );
    TQPopupMenu* menu = dynamic_cast<TQPopupMenu*>(factory()->container("menupage_popup",this));
    if ( menu )
        menu->exec(_point);
    m_mousePos = TQPoint( 0, 0 );
    unplugActionList( "picture_action" );
    delete separator;
}

void KPrView::openPopupMenuObject( const TQString & name, const TQPoint & _point )
{
    if(!koDocument()->isReadWrite() || !factory())
        return;
    dynamic_cast<TQPopupMenu*>(factory()->container(name, this))->popup(_point);
}

void KPrView::openPopupMenuSideBar(const TQPoint & _point)
{
    if(!koDocument()->isReadWrite() || !factory())
        return;
    dynamic_cast<TQPopupMenu*>(factory()->container("sidebarmenu_popup", this))->popup(_point);
}

void KPrView::renamePageTitle()
{
    if(sidebar)
        sidebar->renamePageTitle();
}

void KPrView::picViewOriginalSize()
{
    m_canvas->picViewOriginalSize();
}

void KPrView::picViewOrig640x480()
{
    m_canvas->picViewOrig640x480();
}

void KPrView::picViewOrig800x600()
{
    m_canvas->picViewOrig800x600();
}

void KPrView::picViewOrig1024x768()
{
    m_canvas->picViewOrig1024x768();
}

void KPrView::picViewOrig1280x1024()
{
    m_canvas->picViewOrig1280x1024();
}

void KPrView::picViewOrig1600x1200()
{
    m_canvas->picViewOrig1600x1200();
}

void KPrView::chPic()
{
    m_canvas->chPic();
}

void KPrView::penColorChanged( const KoPen & _pen )
{
    //actionPenColor->setEnabled( true );
    actionPenColor->setCurrentColor( _pen.color() );
}

void KPrView::brushColorChanged( const TQBrush & _brush )
{
    //actionBrushColor->setEnabled( true );
    actionBrushColor->setCurrentColor(_brush.style ()==TQt::NoBrush ? TQt::white : _brush.color() );
}

void KPrView::restartAutoPresTimer()
{
    m_autoPresTime.start();
    m_autoPresElapsedTime = 0;
    m_autoPresTimer.start( m_autoPresTimerValue, true );
}

void KPrView::continueAutoPresTimer()
{
    m_autoPresTime.restart();
    //m_autoPresTimer.changeInterval( m_autoPresTimerValue - m_autoPresElapsedTime );
    m_autoPresTimer.start( m_autoPresTimerValue - m_autoPresElapsedTime, true );
}

void KPrView::stopAutoPresTimer()
{
    m_autoPresTimer.stop();
    m_autoPresElapsedTime += m_autoPresTime.elapsed();
}

void KPrView::setAutoPresTimer( int sec )
{
    m_autoPresTimerValue = sec * 1000;
    restartAutoPresTimer();
}

void KPrView::insertSpecialChar()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( !edit )
        return;
    TQString f = edit->textFontFamily();
    TQChar c=' ';
    if (m_specialCharDlg==0)
    {
        m_specialCharDlg = new KoCharSelectDia( this, "insert special char", f, c, false );
        connect( m_specialCharDlg, TQ_SIGNAL(insertChar(TQChar,const TQString &)),
                 this, TQ_SLOT(slotSpecialChar(TQChar,const TQString &)));
        connect( m_specialCharDlg, TQ_SIGNAL( finished() ),
                 this, TQ_SLOT( slotSpecialCharDlgClosed() ) );
    }
    m_specialCharDlg->show();
}

void KPrView::slotSpecialCharDlgClosed()
{
    if ( m_specialCharDlg)
    {
        disconnect( m_specialCharDlg, TQ_SIGNAL(insertChar(TQChar,const TQString &)),
                    this, TQ_SLOT(slotSpecialChar(TQChar,const TQString &)));
        disconnect( m_specialCharDlg, TQ_SIGNAL( finished() ),
                    this, TQ_SLOT( slotSpecialCharDlgClosed() ) );
        m_specialCharDlg->deleteLater();
        m_specialCharDlg = 0L;
    }
}

void KPrView::slotSpecialChar(TQChar c, const TQString &_font)
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( !edit )
        return;
    edit->insertSpecialChar(c, _font);
}

void KPrView::insertLink()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( !edit )
        return;
    TQString link;
    TQString ref;
    if ( edit->textObject()->hasSelection() )
    {
        TQString selectedText = edit->textObject()->selectedText();
        if ( edit->textObject()->selectionHasCustomItems() || selectedText.contains('\n') )
            return;
        if ( selectedText.startsWith( "mailto:/" ) ||
             selectedText.startsWith( "ftp:/" ) ||
             selectedText.startsWith( "http:/" ) )
        {
            link=selectedText;
            ref = selectedText;
        }
        else
        {
            //Just add text as link name and not url
            link = selectedText;
        }
    }

    if(KoInsertLinkDia::createLinkDia(link, ref, TQStringList(), false, this))
    {
        if(!link.isEmpty() && !ref.isEmpty())
            edit->insertLink(link, ref);
    }
}

void KPrView::changeLink()
{
    KPrTextView * edit = m_canvas->currentTextObjectView();
    if ( edit )
    {
        KoLinkVariable * var=edit->linkVariable();
        if(var)
        {
            TQString oldhref= var->url();
            TQString oldLinkName=var->value();
            TQString link=oldLinkName;
            TQString ref=oldhref;
            if(KoInsertLinkDia::createLinkDia(link, ref, TQStringList(), false, this))
            {
                if(!link.isEmpty() && !ref.isEmpty())
                {
                    if( ref != oldhref || link!=oldLinkName)
                    {
                        KPrChangeLinkVariable*cmd=new KPrChangeLinkVariable( i18n("Change Link"),
                                                                             m_pKPresenterDoc, oldhref,
                                                                             ref, oldLinkName,link, var);
                        cmd->execute();
                        m_pKPresenterDoc->addCommand(cmd);
                    }
                }
            }
        }
    }
}

void KPrView::showFormat( const KoTextFormat &currentFormat )
{
    actionTextFontFamily->setFont( currentFormat.font().family() );
    actionTextFontSize->setFontSize( currentFormat.pointSize() );
    actionTextBold->setChecked( currentFormat.font().bold());
    actionTextItalic->setChecked( currentFormat.font().italic() );
    actionTextUnderline->setChecked( currentFormat.underline());
    actionFormatStrikeOut->setChecked( currentFormat.strikeOut());

    actionBrushColor->setEnabled(true);
    //actionBrushColor->setText(i18n("Text Background Color..."));

    switch(currentFormat.vAlign())
    {
    case KoTextFormat::AlignSuperScript:
    {
        actionFormatSub->setChecked( false );
        actionFormatSuper->setChecked( true );
        break;
    }
    case KoTextFormat::AlignSubScript:
    {
        actionFormatSub->setChecked( true );
        actionFormatSuper->setChecked( false );
        break;
    }
    case KoTextFormat::AlignNormal:
    default:
    {
        actionFormatSub->setChecked( false );
        actionFormatSuper->setChecked( false );
        break;
    }
    }
}

void KPrView::slotSoftHyphen()
{
    KPrTextView *edit = m_canvas->currentTextObjectView();
    if ( edit )
        edit->insertSoftHyphen();
}

void KPrView::slotNonbreakingSpace()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->insertNonbreakingSpace();
}

void KPrView::slotNonbreakingHyphen()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->insertNonbreakingHyphen();
}

void KPrView::slotLineBreak()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->insertLineBreak();
}

void KPrView::slotIncreaseNumberingLevel()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
    {
        edit->increaseNumberingLevel( m_pKPresenterDoc->styleCollection() );
    }
}

void KPrView::slotDecreaseNumberingLevel()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->decreaseNumberingLevel( m_pKPresenterDoc->styleCollection() );
}

void KPrView::extraAutoFormat()
{
    m_pKPresenterDoc->getAutoFormat()->readConfig();
    KoAutoFormatDia dia( this, 0, m_pKPresenterDoc->getAutoFormat() );
    dia.exec();
    m_pKPresenterDoc->startBackgroundSpellCheck(); // will do so if enabled
}

void KPrView::slotSpellCheck()
{
    if (m_spell.kospell) return; // Already in progress
    //m_doc->setReadWrite(false); // prevent editing text - not anymore
    m_spell.macroCmdSpellCheck = 0L;
    m_spell.replaceAll.clear();
    TQValueList<KoTextObject *> objects;
    KPrTextView *edit=m_canvas->currentTextObjectView();
    int options = 0;
    if ( edit && edit->textObject()->hasSelection() )
    {
        objects.append(edit->kpTextObject()->textObject());
        options = KFindDialog::SelectedText;
    }
    else
    {
        objects = spellAddTextObject();
        TQPtrList<KPrObject> lstObj;
    }
    if ( ! objects.empty() )
    {
        m_spell.textIterator = new KoTextIterator( objects, edit, options );
        startKSpell();
    }
}

TQValueList<KoTextObject *> KPrView::spellAddTextObject() const
{
    TQValueList<KoTextObject *> lst;
    TQPtrList<KPrObject> lstObj;
    m_canvas->activePage()->getAllObjectSelectedList(lstObj, true);
    TQPtrListIterator<KPrObject> it( lstObj );
    for ( ; it.current() ; ++it )
    {
        if(it.current()->getType()==OT_TEXT)
        {
            KPrTextObject* tmp = dynamic_cast<KPrTextObject*>(it.current() );
            if ( tmp && !tmp->isProtectContent())
                lst.append( tmp->textObject() );
        }
    }
    return lst;
}



void KPrView::startKSpell()
{
    // m_spellCurrFrameSetNum is supposed to be set by the caller of this method
    if ( !m_spell.kospell )
        m_spell.kospell = new KoSpell( m_broker, this  );
    m_spell.kospell->check( m_spell.textIterator, true );

    delete m_spell.dlg;
    m_spell.dlg = new KSpell2::Dialog( m_spell.kospell, this );
    m_spell.dlg->activeAutoCorrect( true );
    TQObject::connect( m_spell.dlg, TQ_SIGNAL(misspelling(const TQString&, int)),
                      this, TQ_SLOT(spellCheckerMisspelling(const TQString&, int)) );
    TQObject::connect( m_spell.dlg, TQ_SIGNAL(replace(const TQString&, int, const TQString&)),
                      this, TQ_SLOT(spellCheckerCorrected(const TQString&, int, const TQString&)) );
    TQObject::connect( m_spell.dlg, TQ_SIGNAL(done(const TQString&) ),
                      this, TQ_SLOT(spellCheckerDone(const TQString&)) );
    TQObject::connect( m_spell.dlg, TQ_SIGNAL(cancel() ),
                      this, TQ_SLOT( spellCheckerCancel() ) );
    TQObject::connect( m_spell.dlg, TQ_SIGNAL(autoCorrect(const TQString &, const TQString & ) ),
                      this, TQ_SLOT( spellAddAutoCorrect (const TQString &, const TQString &) ) );

    m_spell.dlg->show();
}



void KPrView::spellCheckerCancel()
{
    kdDebug()<<"void KPrView::spellCheckerCancel() \n";
    spellCheckerRemoveHighlight();
    clearSpellChecker(true);
}


void KPrView::spellCheckerRemoveHighlight()
{
    KoTextObject* textobj = m_spell.kospell->currentTextObject();
    if ( textobj ) {
        KPrTextDocument *textdoc=static_cast<KPrTextDocument *>( textobj->textDocument() );
        if ( textdoc )
            textdoc->textObject()->removeHighlight();
    }
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if (edit)
        edit->drawCursor( true );
}


void KPrView::clearSpellChecker(bool cancelSpellCheck)
{
    kdDebug() << "KPrView::clearSpellChecker()" << endl;
    delete m_spell.textIterator;
    m_spell.textIterator = 0L;


    delete m_spell.kospell;
    m_spell.kospell = 0;
    m_initSwitchPage = -1;
    m_switchPage = -1;

    if ( m_spell.macroCmdSpellCheck )
    {
        if ( !cancelSpellCheck )
            m_pKPresenterDoc->addCommand(m_spell.macroCmdSpellCheck);
        else
        {
            //reverte all changes
            m_spell.macroCmdSpellCheck->unexecute();
            delete m_spell.macroCmdSpellCheck;
        }
    }
    m_spell.macroCmdSpellCheck=0L;

    m_spell.replaceAll.clear();

}

void KPrView::spellCheckerMisspelling( const TQString &old, int pos )
{
    //kdDebug(32001) << "KWView::spellCheckerMisspelling old=" << old << " pos=" << pos << endl;
    KoTextObject* textobj = m_spell.kospell->currentTextObject();
    KoTextParag* parag = m_spell.kospell->currentParag();
    Q_ASSERT( textobj );
    Q_ASSERT( parag );
    if ( !textobj || !parag ) return;
    KPrTextDocument *textdoc=static_cast<KPrTextDocument *>( textobj->textDocument() );
    Q_ASSERT( textdoc );
    if ( !textdoc ) return;
    pos += m_spell.kospell->currentStartIndex();

    kdDebug() << "KWView::spellCheckerMisspelling parag=" << parag->paragId() << " pos=" << pos << " length=" << old.length() << endl;

    textdoc->textObject()->highlightPortion( parag, pos, old.length(), m_canvas,true/*repaint*/ );
}

void KPrView::spellCheckerCorrected( const TQString &old, int pos, const TQString &corr )
{
    //kdDebug(33001) << "KWView::spellCheckerCorrected old=" << old << " corr=" << corr << " pos=" << pos << endl;

    //kdDebug(32001) << "KWView::spellCheckerCorrected old=" << old << " corr=" << corr << " pos=" << pos << endl;
    KoTextObject* textobj = m_spell.kospell->currentTextObject();
    KoTextParag* parag = m_spell.kospell->currentParag();
    Q_ASSERT( textobj );
    Q_ASSERT( parag );
    if ( !textobj || !parag ) return;

    KPrTextDocument *textdoc=static_cast<KPrTextDocument *>( textobj->textDocument() );
    Q_ASSERT( textdoc );
    if ( !textdoc )
        return;
    pos += m_spell.kospell->currentStartIndex();
    textdoc->textObject()->highlightPortion( parag, pos, old.length(), m_canvas,true/*repaint*/ );

    KoTextCursor cursor( textdoc );
    cursor.setParag( parag );
    cursor.setIndex( pos );
    if(!m_spell.macroCmdSpellCheck)
        m_spell.macroCmdSpellCheck=new KMacroCommand(i18n("Correct Misspelled Word"));
    m_spell.macroCmdSpellCheck->addCommand(textobj->replaceSelectionCommand(&cursor, corr, TQString(), KoTextDocument::HighlightSelection));
}

void KPrView::spellCheckerDone( const TQString & )
{
    /* See also KWView::spellCheckerDone from KWord */
    kdDebug() << "KPrView::spellCheckerDone" << endl;
    KPrTextDocument *textdoc=static_cast<KPrTextDocument *>( m_spell.kospell->textDocument() );
    Q_ASSERT( textdoc );
    if ( textdoc )
        textdoc->textObject()->removeHighlight();

    clearSpellChecker();
}

void KPrView::showCounter( KoParagCounter &c )
{
    TQString styleStr("counterstyle_");
    styleStr += TQString::number( c.style() );
    //kdDebug(33001) << "KWView::showCounter styleStr=" << styleStr << endl;
    TDEToggleAction* act = static_cast<TDEToggleAction *>( actionCollection()->action( styleStr.latin1() ) );
    Q_ASSERT( act );
    if ( act )
        act->setChecked( true );
}

void KPrView::formatParagraph()
{
    showParagraphDialog();
}

void KPrView::showParagraphDialog(int initialPage, double initialTabPos)
{
    TQPtrList<KoTextFormatInterface> lst = m_canvas->applicableTextInterfaces();
    if ( lst.isEmpty() )
        return;
    TQPtrList<KPrTextObject> lstObjects = m_canvas->applicableTextObjects();
    if ( lstObjects.isEmpty() )
        return;

    delete m_paragDlg;
    m_paragDlg = new KoParagDia( this, "",
                                 KoParagDia::PD_SPACING | KoParagDia::PD_ALIGN |
                                 KoParagDia::PD_DECORATION | KoParagDia::PD_NUMBERING |
                                 KoParagDia::PD_TABS,
                                 m_pKPresenterDoc->unit(),
                                 lstObjects.first()->getSize().width(),false );
    m_paragDlg->setCaption( i18n( "Paragraph Settings" ) );

    // Initialize the dialog from the current paragraph's settings
    m_paragDlg->setParagLayout( * lst.first()->currentParagLayoutFormat() );
    // Set initial page and initial tabpos if necessary
    if ( initialPage != -1 )
    {
        m_paragDlg->setCurrentPage( initialPage );
        if ( initialPage == KoParagDia::PD_TABS )
            m_paragDlg->tabulatorsWidget()->setCurrentTab( initialTabPos );
    }
    connect( m_paragDlg, TQ_SIGNAL( applyParagStyle() ), this, TQ_SLOT( slotApplyParag() ) );

    m_paragDlg->exec();
    delete m_paragDlg;
    m_paragDlg = 0;

}

void KPrView::slotApplyParag()
{
    TQPtrList<KoTextFormatInterface> lst = m_canvas->applicableTextInterfaces();
    Q_ASSERT( !lst.isEmpty() );
    if ( lst.isEmpty() ) return;
    TQPtrListIterator<KoTextFormatInterface> it( lst );
    KMacroCommand * macroCommand = new KMacroCommand( i18n( "Paragraph Settings" ) );
    KoParagLayout newLayout = m_paragDlg->paragLayout();
    int flags = m_paragDlg->changedFlags();
    kdDebug() << k_funcinfo << "flags=" << flags << endl;
    if ( !flags )
        return;
    for ( ; it.current() ; ++it )
    {
        KCommand* cmd = it.current()->setParagLayoutFormatCommand( &newLayout, flags );
        Q_ASSERT( cmd );
        if(cmd)
        {
            macroCommand->addCommand(cmd);
        }
    }

    if(flags)
        m_pKPresenterDoc->addCommand(macroCommand);
    else
        delete macroCommand;

    // Set "oldLayout" in KoParagDia from the current paragraph's settings
    // Otherwise "isBlahChanged" will return wrong things when doing A -> B -> A
    m_paragDlg->setParagLayout( *lst.first()->currentParagLayoutFormat() );
}

void KPrView::textDefaultFormat()
{
    m_canvas->setTextDefaultFormat( );
}

void KPrView::changeNbOfRecentFiles(int _nb)
{
    if ( shell() ) // 0 when embedded into konq !
        shell()->setMaxRecentItems( _nb );
}

TQPopupMenu * KPrView::popupMenu( const TQString& name )
{
    Q_ASSERT(factory());
    if ( factory() )
        return ((TQPopupMenu*)factory()->container( name, this ));
    return 0L;
}

void KPrView::addVariableActions( int type, const TQStringList & texts,
                                         TDEActionMenu * parentMenu, const TQString & menuText )
{
    // Single items go directly into parentMenu.
    // For multiple items we create a submenu.
    if ( texts.count() > 1 && !menuText.isEmpty() )
    {
        TDEActionMenu * subMenu = new TDEActionMenu( menuText, actionCollection() );
        parentMenu->insert( subMenu );
        parentMenu = subMenu;
    }
    TQStringList::ConstIterator it = texts.begin();
    for ( int i = 0; it != texts.end() ; ++it, ++i )
    {
        if ( !(*it).isEmpty() ) // in case of removed subtypes or placeholders
        {
            VariableDef v;
            v.type = type;
            v.subtype = i;
            TDEAction * act = new TDEAction( (*it), 0, this, TQ_SLOT( insertVariable() ),
                                         actionCollection(), "var-action" );
            m_variableDefMap.insert( act, v );
            parentMenu->insert( act );
        }
    }
}

void KPrView::refreshCustomMenu()
{
    TDEActionPtrList lst2 = actionCollection()->actions("custom-variable-action");
    TQValueList<TDEAction *> actions = lst2;
    TQValueList<TDEAction *>::ConstIterator it2 = lst2.begin();
    TQValueList<TDEAction *>::ConstIterator end = lst2.end();
    TQMap<TQString, TDEShortcut> shortCuts;

    for (; it2 != end; ++it2 )
    {
        shortCuts.insert((*it2)->text(), (*it2)->shortcut());
        delete *it2;
    }

    delete actionInsertCustom;
    actionInsertCustom = new TDEActionMenu( i18n( "&Custom" ),
                                          actionCollection(), "insert_custom" );

    actionInsertVariable->insert(actionInsertCustom, 0);


    actionInsertCustom->popupMenu()->clear();
    TQPtrListIterator<KoVariable> it( m_pKPresenterDoc->getVariableCollection()->getVariables() );
    TDEAction * act=0;
    TQStringList lst;
    TQString varName;
    int i = 0;
    for ( ; it.current() ; ++it )
    {
        KoVariable *var = it.current();
        if ( var->type() == VT_CUSTOM )
        {
            varName=( (KoCustomVariable*) var )->name();
            if ( !lst.contains( varName) )
            {
                lst.append( varName );
                TQCString name = TQString("custom-action_%1").arg(i).latin1();
                act = new TDEAction( varName, shortCuts[varName], this,
                                   TQ_SLOT( insertCustomVariable() ), actionCollection(), name );

                act->setGroup( "custom-variable-action" );
                actionInsertCustom->insert( act );
                i++;
            }
        }
    }
    bool state=!lst.isEmpty();
    if(state)
        actionInsertCustom->popupMenu()->insertSeparator();

    act = new TDEAction( i18n("New..."), 0, this, TQ_SLOT( insertNewCustomVariable() ), actionCollection(),
                       TQString("custom-action_%1").arg(i).latin1() );
    act->setGroup( "custom-variable-action" );
    actionInsertCustom->insert( act );

    actionInsertCustom->popupMenu()->insertSeparator();

    actionEditCustomVars->setEnabled(state);
    actionEditCustomVarsEdit->setEnabled( state );
    actionInsertCustom->insert( actionEditCustomVarsEdit );
}

void KPrView::insertCustomVariable()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
    {
        TDEAction * act = (TDEAction *)(sender());
        edit->insertCustomVariable(act->text());
    }
}

void KPrView::insertNewCustomVariable()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->insertVariable( VT_CUSTOM, 0 );
}

void KPrView::editCustomVariable()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
    {
        KoCustomVariable *var = static_cast<KoCustomVariable *>(edit->variable());
        if (var)
        {
            TQString oldvalue = var->value();
            KoCustomVarDialog dia( this, var );
            if ( dia.exec() )
            {
                if( var->value() != oldvalue )
                {
                    KPrChangeCustomVariableValue *cmd=new KPrChangeCustomVariableValue(i18n( "Change Custom Variable" ),
                                                                                       m_pKPresenterDoc, oldvalue, var->value(), var );
                    m_pKPresenterDoc->addCommand(cmd);
                }
                m_pKPresenterDoc->recalcVariables( VT_CUSTOM );
            }
        }
    }
}

void KPrView::editCustomVars()
{
    KoCustomVariablesDia dia( this, m_pKPresenterDoc->getVariableCollection()->getVariables() );
    TQStringList listOldCustomValue;
    TQPtrListIterator<KoVariable> oldIt( m_pKPresenterDoc->getVariableCollection()->getVariables() );
    for ( ; oldIt.current() ; ++oldIt )
    {
        if(oldIt.current()->type()==VT_CUSTOM)
            listOldCustomValue.append(((KoCustomVariable*)oldIt.current())->value());
    }
    if(dia.exec())
    {
        m_pKPresenterDoc->recalcVariables( VT_CUSTOM );
        TQPtrListIterator<KoVariable> it( m_pKPresenterDoc->getVariableCollection()->getVariables() );
        KMacroCommand * macroCommand = 0L;
        int i=0;
        for ( ; it.current() ; ++it )
        {
            if(it.current()->type() == VT_CUSTOM )
            {
                if(((KoCustomVariable*)it.current())->value()!=*(listOldCustomValue.at(i)))
                {
                    if(!macroCommand)
                        macroCommand = new KMacroCommand( i18n( "Change Custom Variable" ) );
                    KPrChangeCustomVariableValue *cmd=
                        new KPrChangeCustomVariableValue(i18n( "Change Custom Variable" ), m_pKPresenterDoc,
                                                         *(listOldCustomValue.at(i)),
                                                         ((KoCustomVariable*)it.current())->value(),
                                                         ((KoCustomVariable*)it.current()));
                    macroCommand->addCommand(cmd);
                }
                i++;
            }
        }
        if(macroCommand)
            m_pKPresenterDoc->addCommand(macroCommand);
    }
}

void KPrView::insertVariable()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
    {
        TDEAction * act = (TDEAction *)(sender());
        VariableDefMap::ConstIterator it = m_variableDefMap.find( act );
        if ( it == m_variableDefMap.end() )
            kdWarning(33001) << "Action not found in m_variableDefMap." << endl;
        else
        {
            if ( (*it).type == VT_FIELD )
                edit->insertVariable( (*it).type, KoFieldVariable::fieldSubType( (*it).subtype ) );
            else
                edit->insertVariable( (*it).type, (*it).subtype );
        }
    }
}

void KPrView::openLink()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->openLink();
}

void KPrView::showRulerIndent( double _leftMargin, double _firstLine, double _rightMargin, bool rtl )
{
    KoRuler * hRuler = getHRuler();
    if ( hRuler )
    {
        hRuler->setFirstIndent( KoUnit::toUserValue( _firstLine, m_pKPresenterDoc->unit() ) );
        hRuler->setLeftIndent( KoUnit::toUserValue( _leftMargin, m_pKPresenterDoc->unit() ) );
        hRuler->setRightIndent( KoUnit::toUserValue( _rightMargin, m_pKPresenterDoc->unit() ) );
        hRuler->setDirection( rtl );
        actionTextDepthMinus->setEnabled( _leftMargin>0);
    }
}

void KPrView::tabListChanged( const KoTabulatorList & tabList )
{
    if(!m_pKPresenterDoc->isReadWrite())
        return;

    m_canvas->setTabList( tabList );
}

void KPrView::newFirstIndent( double _firstIndent )
{
    m_canvas->setNewFirstIndent(_firstIndent);
}

void KPrView::newLeftIndent( double _leftIndent)
{
    m_canvas->setNewLeftIndent(_leftIndent);
}

void KPrView::newRightIndent( double _rightIndent)
{
    m_canvas->setNewRightIndent(_rightIndent);
}

void KPrView::slotUpdateRuler()
{
    // Set the "frame start" in the ruler (tabs are relative to that position)
    bool isText=!m_canvas->applicableTextObjects().isEmpty();
    if ( isText )
    {
        KPrTextObject *txtobj= m_canvas->applicableTextObjects().first();
        if ( txtobj )
        {
            TQRect r= zoomHandler()->zoomRect(txtobj->getRealRect());
            getHRuler()->setFrameStartEnd( r.left() /*+ m_canvas->diffx()*//*- pc.x()*/, r.right()/*+m_canvas->diffx()*/ /*- pc.x()*/ );
            getVRuler()->setFrameStartEnd( r.top()/*+ m_canvas->diffy()*//*- pc.y()*/, r.bottom()/*+m_canvas->diffy()*//*- pc.y()*/ );
            if( getHRuler())
            {
                int flags = txtobj->isProtectContent() ? 0 : (KoRuler::F_INDENTS | KoRuler::F_TABS);
                if( getHRuler()->flags()!= flags )
                {
                    getHRuler()->changeFlags(flags);
                    getHRuler()->repaint();
                }
            }
            if( getVRuler())
            {
                if( getVRuler()->flags() != 0 )
                {
                    getVRuler()->changeFlags(0);
                    getVRuler()->repaint();
                }
            }
        }
    }
    else
    {
        refreshRuler( kPresenterDoc()->showGuideLines() );
        updateRuler();
    }
}

// This handles Tabulators _only_
void KPrView::slotHRulerDoubleClicked( double ptpos )
{
    showParagraphDialog( KoParagDia::PD_TABS, ptpos );
}

// This handles either:
// - Indents
// - Page Layout
//
// This does _not_ handle Tabulators!
void KPrView::slotHRulerDoubleClicked()
{
    KoRuler *ruler = getHRuler();

    if ( m_canvas && m_canvas->currentTextObjectView() && (ruler->flags() & KoRuler::F_INDENTS) && ruler->doubleClickedIndent() )
        formatParagraph();
    else
        openPageLayoutDia();
}

void KPrView::changeCaseOfText()
{
    TQPtrList<KoTextFormatInterface> lst = m_canvas->applicableTextInterfaces();
    if ( lst.isEmpty() ) return;
    TQPtrListIterator<KoTextFormatInterface> it( lst );
    KoChangeCaseDia *caseDia=new KoChangeCaseDia( this,"change case" );
    if(caseDia->exec())
    {
        KMacroCommand* macroCmd = 0L;
        for ( ; it.current() ; ++it )
        {
            KCommand *cmd = it.current()->setChangeCaseOfTextCommand(caseDia->getTypeOfCase());
            if (cmd)
            {
                if ( !macroCmd )
                    macroCmd = new KMacroCommand( i18n("Change Case of Text") );
                macroCmd->addCommand(cmd);
            }
        }
        if( macroCmd )
            m_pKPresenterDoc->addCommand(macroCmd);
    }
    delete caseDia;
}

void KPrView::editFind()
{
    if (!m_searchEntry)
        m_searchEntry = new KoSearchContext();
    KPrTextView * edit = m_canvas->currentTextObjectView();
    bool hasSelection = edit && (edit->kpTextObject())->textObject()->hasSelection();
    bool hasCursor = edit != 0;
    KoSearchDia dialog( m_canvas, "find", m_searchEntry, hasSelection, hasCursor );

    /// KoFindReplace needs a TQValueList<KoTextObject *>...
    TQValueList<KoTextObject *> list;
    TQPtrList<KoTextObject> list2 = m_pKPresenterDoc->allTextObjects();
    TQPtrListIterator<KoTextObject> it( list2 );
    for ( ; it.current() ; ++it )
        list.append(it.current());

    if( list.isEmpty() )
        return;

    if ( dialog.exec() == TQDialog::Accepted )
    {
        delete m_findReplace;
        m_findReplace = new KPrFindReplace( this, m_canvas, &dialog, list, edit );
        editFindNext();
    }
}

void KPrView::editReplace()
{
    if (!m_searchEntry)
        m_searchEntry = new KoSearchContext();
    if (!m_replaceEntry)
        m_replaceEntry = new KoSearchContext();

    KPrTextView * edit = m_canvas->currentTextObjectView();
    bool hasSelection = edit && (edit->kpTextObject())->textObject()->hasSelection();
    bool hasCursor = edit != 0;
    KoReplaceDia dialog( m_canvas, "replace", m_searchEntry, m_replaceEntry, hasSelection, hasCursor );

    /// KoFindReplace needs a TQValueList<KoTextObject *>...
    TQValueList<KoTextObject *> list;
    TQPtrList<KoTextObject> list2 = m_pKPresenterDoc->allTextObjects();
    TQPtrListIterator<KoTextObject> it( list2 );
    for ( ; it.current() ; ++it )
        list.append(it.current());

    if( list.isEmpty() )
        return;

    if ( dialog.exec() == TQDialog::Accepted )
    {
        delete m_findReplace;
        m_findReplace = new KPrFindReplace( this, m_canvas, &dialog, list, edit );
        editFindNext();
    }
}

void KPrView::editFindPrevious()
{
    if ( !m_findReplace ) // shouldn't be called before find or replace is activated
    {
        editFind();
        return;
    }
    (void) m_findReplace->findPrevious();
}

void KPrView::editFindNext()
{
    if ( !m_findReplace ) // shouldn't be called before find or replace is activated
    {
        editFind();
        return;
    }
    (void) m_findReplace->findNext();
}

void KPrView::refreshAllVariable()
{
    m_pKPresenterDoc->recalcVariables( VT_ALL );
}

void KPrView::changeZoomMenu( int zoom )
{
    TQStringList lst;
    lst << i18n( "Width" );
    lst << i18n( "Whole Slide" );

    if(zoom>0)
    {
        TQValueList<int> list;
        bool ok;
        const TQStringList itemsList ( actionViewZoom->items() );
        TQRegExp regexp("(\\d+)"); // "Captured" non-empty sequence of digits

        for (TQStringList::ConstIterator it = itemsList.begin() ; it != itemsList.end() ; ++it)
        {
            regexp.search(*it);
            const int val=regexp.cap(1).toInt(&ok);
            //zoom : limit inferior=10
            if(ok && val>9 && list.contains(val)==0)
                list.append( val );
        }
        //necessary at the beginning when we read config
        //this value is not in combo list
        if(list.contains(zoom)==0)
            list.append( zoom );

        qHeapSort( list );

        for (TQValueList<int>::Iterator it = list.begin() ; it != list.end() ; ++it)
            lst.append( i18n("%1%").arg(*it) );
    }
    else
    {
        lst << i18n("%1%").arg("33");
        lst << i18n("%1%").arg("50");
        lst << i18n("%1%").arg("75");
        lst << i18n("%1%").arg("100");
        lst << i18n("%1%").arg("125");
        lst << i18n("%1%").arg("150");
        lst << i18n("%1%").arg("200");
        lst << i18n("%1%").arg("250");
        lst << i18n("%1%").arg("350");
        lst << i18n("%1%").arg("400");
        lst << i18n("%1%").arg("450");
        lst << i18n("%1%").arg("500");
    }
    actionViewZoom->setItems( lst );
}

void KPrView::showZoom( int zoom )
{
    TQStringList list = actionViewZoom->items();
    TQString zoomStr( i18n("%1%").arg( zoom ) );
    int pos = list.findIndex(zoomStr);
    if( pos == -1)
    {
        changeZoomMenu( zoom );
        list = actionViewZoom->items();
    }
    actionViewZoom->setCurrentItem( list.findIndex(zoomStr)  );
}

void KPrView::viewZoom( const TQString &s )
{
    bool ok=false;
    int zoom = 0;
    if ( s == i18n("Width") )
    {
        zoom = tqRound( static_cast<double>(m_canvas->visibleRect().width() * 100 ) /
                       (zoomHandler()->resolutionX() * m_pKPresenterDoc->pageLayout().ptWidth ) );
        ok = true;
    }
    else if ( s == i18n("Whole Slide") )
    {
        zoom = getZoomEntirePage();
        ok = true;
    }
    else
    {
        TQRegExp regexp("(\\d+)"); // "Captured" non-empty sequence of digits
        regexp.search(s);
        zoom=regexp.cap(1).toInt(&ok);
    }
    if( !ok || zoom<10 ) //zoom should be valid and >10
        zoom = zoomHandler()->zoom();
    zoom = TQMIN( zoom, 4000);
    //refresh menu
    changeZoomMenu( zoom );
    //refresh menu item
    showZoom(zoom);
    //apply zoom if zoom!=m_doc->zoom()
    if( zoom != zoomHandler()->zoom() )
    {
        setZoom( zoom, true );
        KPrTextView *edit=m_canvas->currentTextObjectView();
        if ( edit )
            edit->ensureCursorVisible();
    }

    m_canvas->setFocus();
    m_canvas->repaint();
}

void KPrView::setZoomRect( const KoRect & rect )
{
    double height = zoomHandler()->resolutionY() * rect.height();
    double width = zoomHandler()->resolutionX() * rect.width();
    int zoom = TQMIN( tqRound( static_cast<double>( m_canvas->visibleRect().height() * 100 ) / height ),
            tqRound( static_cast<double>( m_canvas->visibleRect().width() * 100 ) / width ) );

    m_canvas->setUpdatesEnabled( false );
    viewZoom( TQString::number( zoom ) );

    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->scrollTopLeftPoint( zoomHandler()->zoomPoint( rect.topLeft() ) );
    m_canvas->setUpdatesEnabled( true );
    m_canvas->repaint();
}

void KPrView::setZoom( int zoom, bool updateViews )
{
    zoomHandler()->setZoomAndResolution( zoom, KoGlobal::dpiX(),
                                         KoGlobal::dpiY());
    m_pKPresenterDoc->newZoomAndResolution(updateViews,false);
    m_pKPresenterDoc->updateZoomRuler();

    setRanges();

    if ( statusBar() )
      m_sbZoomLabel->setText( ' ' + TQString::number( zoom ) + "% " );
}

void KPrView::slotUpdateScrollBarRanges()
{
    setRanges();
}

KoTextZoomHandler *KPrView::zoomHandler() const
{
    return m_pKPresenterDoc->zoomHandler();
}

void KPrView::slotViewFormattingChars()
{
    m_pKPresenterDoc->setViewFormattingChars(actionViewFormattingChars->isChecked());
    m_pKPresenterDoc->layout(); // Due to the different formatting when this option is activated
    m_pKPresenterDoc->repaint(false);
}

void KPrView::setPageDuration( int _pgNum )
{
    if ( kPresenterDoc()->presentationDuration() )
    {
        // kdDebug(33001) << "KPrView::setPageDuration( " << _pgNum << " )" << endl;
        *m_presentationDurationList.at( _pgNum ) += m_duration.elapsed();
        m_duration.restart();
    }
}

void KPrView::openThePresentationDurationDialog()
{
    int totalTime = 0;
    TQStringList presentationDurationStringList;
    for ( TQValueList<int>::Iterator it = m_presentationDurationList.begin();
          it != m_presentationDurationList.end(); ++it ) {
        int _time = *it;
        TQString presentationDurationString = presentationDurationDataFormatChange( _time );
        presentationDurationStringList.append( presentationDurationString );
        totalTime += _time;
    }

    TQString presentationTotalDurationString = presentationDurationDataFormatChange( totalTime );

    delete presDurationDia;
    presDurationDia = 0;

    presDurationDia = new KPrPresDurationDia( this, "presDurationDia", kPresenterDoc(),
                                             presentationDurationStringList, presentationTotalDurationString );
    presDurationDia->setCaption( i18n( "Presentation Duration" ) );
    TQObject::connect( presDurationDia, TQ_SIGNAL( presDurationDiaClosed() ), this, TQ_SLOT( pddClosed() ) );
    presDurationDia->exec();

    delete presDurationDia;
    presDurationDia = 0;
}

void KPrView::pddClosed()
{
    presDurationDia = 0;
}

// change from milliseconds to hh:mm:ss
// in tdelibs 3.2.90, an additional method was added that basically
// allows formatting of time as a duration. Before this, you can get
// weirdness (such as a duration of 12:00:03 am) depending on how the
// time format is set.
TQString KPrView::presentationDurationDataFormatChange( int _time )
{
    TQTime time( 0, 0, 0 );
    return TDEGlobal::locale()->formatTime( time.addMSecs( _time ), true, true );
}


void KPrView::viewFooter()
{
    bool state = m_actionExtraFooter->isChecked();
    m_canvas->activePage()->setFooter( state );
    KPrHideShowHeaderFooter * cmd =new KPrHideShowHeaderFooter( state ? i18n("Enable Document Header") : i18n("Disable Document Header"),
                                                                m_pKPresenterDoc, m_canvas->activePage(), state, m_pKPresenterDoc->footer());
    m_pKPresenterDoc->addCommand(cmd);

    m_pKPresenterDoc->updateSideBarItem( m_pKPresenterDoc->masterPage() );
}

void KPrView::viewHeader()
{
    bool state = m_actionExtraHeader->isChecked();
    m_canvas->activePage()->setHeader( state );
    KPrHideShowHeaderFooter * cmd =new KPrHideShowHeaderFooter( state ? i18n("Enable Document Footer") : i18n("Disable Document Footer"),
                                                                m_pKPresenterDoc, m_canvas->activePage(), state, m_pKPresenterDoc->header());
    m_pKPresenterDoc->addCommand(cmd);

    m_pKPresenterDoc->updateSideBarItem( m_pKPresenterDoc->masterPage() );
}

void KPrView::showStyle( const TQString & styleName )
{
    KoParagStyle* style = m_pKPresenterDoc->styleCollection()->findStyle( styleName );
    if ( style ) {
        int pos = m_pKPresenterDoc->styleCollection()->indexOf( style );
        // Select style in combo
        actionFormatStyle->setCurrentItem( pos );
    }
}

void KPrView::updateStyleList()
{
    TQString currentStyle = actionFormatStyle->currentText();
    // Generate list of styles
    const TQStringList lst = m_pKPresenterDoc->styleCollection()->displayNameList();
    const int pos = lst.findIndex( currentStyle );
    // Fill the combo - using a TDESelectAction
    actionFormatStyle->setItems( lst );
    if ( pos > -1 )
        actionFormatStyle->setCurrentItem( pos );

    // Fill the menu - using a TDEActionMenu, so that it's possible to bind keys
    // to individual actions
    TQStringList lstWithAccels;
    // Generate unique accelerators for the menu items
    TDEAccelGen::generate( lst, lstWithAccels );
    TQMap<TQString, TDEShortcut> shortCuts;

    TDEActionPtrList lst2 = actionCollection()->actions("styleList");
    TQValueList<TDEAction *> actions = lst2;
    TQValueList<TDEAction *>::ConstIterator it = lst2.begin();
    TQValueList<TDEAction *>::ConstIterator end = lst2.end();
    for (; it != end; ++it )
    {
        shortCuts.insert( TQString::fromUtf8( (*it)->name() ), (*it)->shortcut() );
        actionFormatStyleMenu->remove( *it );
        delete *it;
    }


    uint i = 0;
    for ( TQStringList::Iterator it = lstWithAccels.begin(); it != lstWithAccels.end(); ++it, ++i )
    {
        KoParagStyle *style = m_pKPresenterDoc->styleCollection()->styleAt( i );
        if ( style )
        {
            TQString name = "paragstyle_" + style->name();
            TDEToggleAction* act = new TDEToggleAction( (*it),
                                     shortCuts[name], this, TQ_SLOT( slotStyleSelected() ),
                                     actionCollection(), name.utf8() );
            act->setGroup( "styleList" );
            act->setExclusiveGroup( "styleListAction" );
            act->setToolTip( i18n( "Apply a paragraph style" ) );
            actionFormatStyleMenu->insert( act );
        }
    }
    bool isText=!m_canvas->applicableTextInterfaces().isEmpty();
    actionFormatStyleMenu->setEnabled( isText );
    actionFormatStyle->setEnabled(isText);
}

void KPrView::extraStylist()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    TQString activeStyleName  = TQString();
    if ( edit )
    {
        edit->hideCursor();
        if (edit->cursor() && edit->cursor()->parag() && edit->cursor()->parag()->style())
            activeStyleName = edit->cursor()->parag()->style()->displayName();
    }
    KPrStyleManager * styleManager = new KPrStyleManager( this, m_pKPresenterDoc->unit(), m_pKPresenterDoc,
                                                          *m_pKPresenterDoc->styleCollection(), activeStyleName);
    styleManager->exec();
    delete styleManager;
    if ( edit )
        edit->showCursor();
}

// Called when selecting a style in the Format / Style menu
void KPrView::slotStyleSelected()
{
    TQString actionName = TQString::fromUtf8(sender()->name());
    kdDebug(33001) << "KPrView::slotStyleSelected " << actionName << endl;
    textStyleSelected( m_pKPresenterDoc->styleCollection()->findStyle( actionName ) );
}

void KPrView::textStyleSelected( int index )
{
    textStyleSelected( m_pKPresenterDoc->styleCollection()->styleAt( index ) );
}

void KPrView::textStyleSelected( KoParagStyle *_sty )
{
    if ( !_sty )
        return;

    KPrTextView *edit=m_canvas->currentTextObjectView();
    if(edit)
    {
        edit->applyStyle( _sty );
        m_canvas->setFocus();
    }
    else
    {
        TQPtrList<KPrTextObject> selectedFrames = m_canvas->selectedTextObjs();

        if (selectedFrames.count() <= 0)
            return; // nope, no frames are selected.
        // yes, indeed frames are selected.
        TQPtrListIterator<KPrTextObject> it( selectedFrames );
        KMacroCommand *globalCmd = 0L;
        for ( ; it.current() ; ++it )
        {
            KoTextObject *textObject = it.current()->textObject();
            textObject->textDocument()->selectAll( KoTextDocument::Temp );
            KCommand *cmd = textObject->applyStyleCommand( 0L, _sty,
                                                           KoTextDocument::Temp, KoParagLayout::All, KoTextFormat::Format,
                                                           true, true );
            textObject->textDocument()->removeSelection( KoTextDocument::Temp );
            if (cmd)
            {
                if ( !globalCmd)
                    globalCmd = new KMacroCommand( selectedFrames.count() == 1 ? i18n("Apply Style to Frame") :
                                                   i18n("Apply Style to Frames"));
                globalCmd->addCommand( cmd );
            }
        }
        if ( globalCmd )
            m_pKPresenterDoc->addCommand( globalCmd );
    }

}

void KPrView::slotAllowAutoFormat()
{
    bool state = actionAllowAutoFormat->isChecked();
    m_pKPresenterDoc->setAllowAutoFormat( state );
}

void KPrView::slotCompletion()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if(edit)
        edit->completion();
}

void KPrView::insertComment()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( !edit )
        return;
    TQString authorName;
    KoDocumentInfo * info = m_pKPresenterDoc->documentInfo();
    KoDocumentInfoAuthor * authorPage = static_cast<KoDocumentInfoAuthor *>(info->page( "author" ));
    if ( !authorPage )
        kdWarning() << "Author information not found in documentInfo !" << endl;
    else
        authorName = authorPage->fullName();

    KoCommentDia *commentDia = new KoCommentDia( this, TQString(),authorName );
    if( commentDia->exec() )
        edit->insertComment(commentDia->commentText());
    delete commentDia;
}

void KPrView::editComment()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
    {
        KoVariable * tmpVar=edit->variable();
        KoNoteVariable * var = dynamic_cast<KoNoteVariable *>(tmpVar);
        if(var)
        {
            TQString authorName;
            KoDocumentInfo * info = m_pKPresenterDoc->documentInfo();
            KoDocumentInfoAuthor * authorPage = static_cast<KoDocumentInfoAuthor *>(info->page( "author" ));
            if ( !authorPage )
                kdWarning() << "Author information not found in documentInfo !" << endl;
            else
                authorName = authorPage->fullName();
            TQString oldValue = var->note();
            TQString createDate = var->createdNote();
            KoCommentDia *commentDia = new KoCommentDia( this, oldValue, authorName, createDate);

            if( commentDia->exec() )
            {
                if ( oldValue != commentDia->commentText())
                {
                    KPrChangeVariableNoteText *cmd = new KPrChangeVariableNoteText( i18n("Change Note Text"), m_pKPresenterDoc, oldValue,commentDia->commentText(), var);
                    m_pKPresenterDoc->addCommand( cmd );
                    cmd->execute();
                }
            }
            delete commentDia;
        }
    }
}

void KPrView::viewGuideLines()
{
    bool state=actionViewShowGuideLine->isChecked();
    m_pKPresenterDoc->setShowGuideLines( state );
    m_pKPresenterDoc->updateGuideLineButton();
    deSelectAllObjects();
    refreshRuler( state );
    m_pKPresenterDoc->repaint(false);
}


void KPrView::viewGrid()
{
    m_pKPresenterDoc->setShowGrid( actionViewShowGrid->isChecked() );
    m_pKPresenterDoc->setModified( true );
    m_pKPresenterDoc->updateGridButton();
    m_pKPresenterDoc->repaint(false);
}


void KPrView::updateGuideLineButton()
{
    bool state = m_pKPresenterDoc->showGuideLines();
    actionViewShowGuideLine->setChecked( state );
    refreshRuler( state );
}

void KPrView::updateGridButton()
{
    actionViewShowGrid->setChecked( m_pKPresenterDoc->showGrid() );
    actionViewSnapToGrid->setChecked ( m_pKPresenterDoc->snapToGrid() );
}

void KPrView::refreshRuler( bool state )
{
    if( getHRuler() )
    {

        if ( !m_pKPresenterDoc->isReadWrite())
        {
            getHRuler()->changeFlags(KoRuler::F_NORESIZE);
            getHRuler()->repaint();
        }
        else
        {
            if( state )
            {
                if( getHRuler()->flags() != KoRuler::F_HELPLINES )
                {
                    getHRuler()->changeFlags(KoRuler::F_HELPLINES);
                    getHRuler()->repaint();
                }
            }
            else
            {
                if( getHRuler()->flags() != 0 )
                {
                    getHRuler()->changeFlags( 0 );
                    getHRuler()->repaint();
                }
            }
        }
    }

    if( getVRuler())
    {
        if ( !m_pKPresenterDoc->isReadWrite())
        {
            getVRuler()->changeFlags(KoRuler::F_NORESIZE);
            getVRuler()->repaint();
        }
        else
        {
            if( state )
            {
                if( getVRuler()->flags() != KoRuler::F_HELPLINES )
                {
                    getVRuler()->changeFlags(KoRuler::F_HELPLINES);
                    getVRuler()->repaint();
                }
            }
            else
            {
                if( getVRuler()->flags()!= 0)
                {
                    getVRuler()->changeFlags(0);
                    getVRuler()->repaint();
                }
            }
        }
    }

}

void KPrView::addGuideLine()
{
    KoRect rect( m_canvas->activePage()->getPageRect() );

    KoPoint pos( zoomHandler()->unzoomPoint( m_mousePos + TQPoint( m_canvas->diffx(), m_canvas->diffy() ) ) );
    KoGuideLineDia dia( 0, pos, rect, m_pKPresenterDoc->unit() );
    if ( dia.exec() == TQDialog::Accepted )
    {
        m_pKPresenterDoc->addGuideLine( dia.orientation(), dia.pos() );
    }
}

void KPrView::removeComment()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->removeComment();
}

void KPrView::configureCompletion()
{
    m_pKPresenterDoc->getAutoFormat()->readConfig();
    KoCompletionDia dia( this, 0, m_pKPresenterDoc->getAutoFormat() );
    dia.exec();
}

void KPrView::openPopupMenuZoom( const TQPoint & _point )
{
    if(!koDocument()->isReadWrite() || !factory())
        return;
    actionZoomSelectedObject->setEnabled( m_canvas->isOneObjectSelected());
    int nbObj = m_canvas->activePage()->objectList().count();
    actionZoomAllObject->setEnabled( nbObj > 0);
    static_cast<TQPopupMenu*>(factory()->container("zoom_popup",this))->popup(_point);
}

void KPrView::zoomMinus()
{
    //unzoom from 25%
    int zoom = zoomHandler()->zoom() - (int)(zoomHandler()->zoom()*0.25);
    viewZoom( TQString::number(zoom ) );
    m_canvas->setToolEditMode( TEM_MOUSE );
}

void KPrView::zoomPageWidth()
{
    int zoom = tqRound( static_cast<double>(m_canvas->visibleRect().width() * 100 ) /
                       (zoomHandler()->resolutionX() * m_pKPresenterDoc->pageLayout().ptWidth ) );
    viewZoom( TQString::number(zoom ) );
    m_canvas->setToolEditMode( TEM_MOUSE );
}

void KPrView::zoomEntirePage()
{
    viewZoom( TQString::number(getZoomEntirePage() ) );
    m_canvas->setToolEditMode( TEM_MOUSE );
}

void KPrView::zoomPlus()
{
    int zoom = zoomHandler()->zoom() + int( zoomHandler()->zoom() * 0.25 );
    viewZoom( TQString::number( zoom ) );
    m_canvas->setToolEditMode( TEM_MOUSE );
}

int KPrView::getZoomEntirePage() const
{
    double height = zoomHandler()->resolutionY() * m_pKPresenterDoc->pageLayout().ptHeight;
    double width = zoomHandler()->resolutionX() * m_pKPresenterDoc->pageLayout().ptWidth;
    int zoom = TQMIN( tqRound( static_cast<double>(m_canvas->visibleRect().height() * 100 ) / height ),
                     tqRound( static_cast<double>(m_canvas->visibleRect().width() * 100 ) / width ) );
    return zoom;
}

void KPrView::zoomSelectedObject()
{
    if( m_canvas->isOneObjectSelected() )
    {
        KoRect rect = m_canvas->objectRect( false );
        setZoomRect( rect );
    }
}

void KPrView::zoomPageHeight()
{
    int zoom = tqRound( static_cast<double>(m_canvas->visibleRect().height() * 100 ) /
                       (zoomHandler()->resolutionX() * m_pKPresenterDoc->pageLayout().ptHeight ) );
    viewZoom( TQString::number(zoom ) );
    m_canvas->setToolEditMode( TEM_MOUSE );
}

void KPrView::zoomAllObject()
{
    KoRect rect = m_canvas->objectRect( true );
    double height = zoomHandler()->resolutionY() * rect.height();
    double width = zoomHandler()->resolutionX() * rect.width();
    int zoom = TQMIN( tqRound( static_cast<double>(m_canvas->visibleRect().height() * 100 ) / height ),
                     tqRound( static_cast<double>(m_canvas->visibleRect().width() * 100 ) / width ) );
    viewZoom( TQString::number(zoom ) );

    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->scrollTopLeftPoint( zoomHandler()->zoomPoint( rect.topLeft()) );
}

void KPrView::flipHorizontal()
{
    m_canvas->flipObject( true );
}

void KPrView::flipVertical()
{
    m_canvas->flipObject( false );
}

void KPrView::slotObjectEditChanged()
{
    bool state=m_canvas->isOneObjectSelected();
    bool rw = koDocument()->isReadWrite();

    bool isText=!m_canvas->applicableTextInterfaces().isEmpty();
    actionTextFont->setEnabled(isText);
    actionTextFontSize->setEnabled(isText);
    actionTextFontFamily->setEnabled(isText);
    actionTextColor->setEnabled(isText);
    actionTextAlignLeft->setEnabled(isText);
    actionTextAlignCenter->setEnabled(isText);
    actionTextAlignRight->setEnabled(isText);
    actionTextAlignBlock->setEnabled(isText);

    actionFormatBullet->setEnabled(rw && isText );
    actionFormatNumber->setEnabled(rw && isText );

    actionTextDepthPlus->setEnabled(isText);
    actionFormatDefault->setEnabled(isText);
    actionTextDepthMinus->setEnabled(isText);

    actionTextExtentCont2Height->setEnabled(isText);
    actionTextExtendObj2Cont->setEnabled(isText);
    actionTextBold->setEnabled(isText);
    actionTextItalic->setEnabled(isText);
    actionTextUnderline->setEnabled(isText);
    actionFormatStrikeOut->setEnabled(isText);
    actionFormatSuper->setEnabled(isText);
    actionFormatSub->setEnabled(isText);
    actionIncreaseFontSize->setEnabled(isText);
    actionDecreaseFontSize->setEnabled(isText);

    if ( isText )
    {
        KoTextFormat format =*(m_canvas->applicableTextInterfaces().first()->currentFormat());
        showFormat( format );
        const KoParagLayout * paragLayout=m_canvas->applicableTextInterfaces().first()->currentParagLayoutFormat();
        KoParagCounter counter;
        if(paragLayout->counter)
            counter = *(paragLayout->counter);
        int align = paragLayout->alignment;
        if ( align == TQt::AlignAuto )
            align = TQt::AlignLeft; // ## seems hard to detect RTL here
        alignChanged( align );
    }

    KPrTextView *edit=m_canvas->currentTextObjectView();
    bool val=(edit!=0) && isText && !edit->kpTextObject()->isProtectContent();
    actionInsertSpecialChar->setEnabled(val);
    actionInsertComment->setEnabled( val );

    actionInsertLink->setEnabled(val);
    actionFormatParag->setEnabled(isText);
    actionInsertVariable->setEnabled(val);
    actionTextInsertPageNum->setEnabled(val);
    if ( edit )
        actionBrushColor->setEnabled(val);

    bool hasSelection = false ;
    if(edit)
    {
        double leftMargin =edit->currentParagLayout().margins[TQStyleSheetItem::MarginLeft];
        actionTextDepthMinus->setEnabled(val && leftMargin>0);
        hasSelection = edit->textObject()->hasSelection();
        actionEditCut->setEnabled(hasSelection);
    }
    actionCreateStyleFromSelection->setEnabled(edit!=0);

    actionChangeCase->setEnabled( (val && rw && hasSelection ) || (rw && !edit && isText) );

    if(!edit)
    {
        actionEditCopy->setEnabled(state);
        bool headerfooterselected = false;
        if(m_canvas->numberOfObjectSelected()==1)
        {
            KPrObject *obj=m_canvas->getSelectedObj();
            //disable this action when we select a header/footer
            if(obj==m_pKPresenterDoc->header() || obj==m_pKPresenterDoc->footer())
                headerfooterselected=true;
            else
                headerfooterselected=false;
        }

        actionEditCut->setEnabled(state&&!headerfooterselected);
    }
    actionFormatStyleMenu->setEnabled( isText );
    actionFormatStyle->setEnabled(isText);

    state=m_canvas->oneObjectTextExist();
    actionEditFind->setEnabled(state);
    actionEditFindNext->setEnabled( state );
    actionEditFindPrevious->setEnabled( state );
    actionEditReplace->setEnabled(state);

    slotUpdateRuler();
}

void KPrView::duplicateObj()
{
    if (m_canvas->currentTextObjectView() && !m_canvas->isOneObjectSelected() )
        return;

    KPrDuplicatObjDia *dlg= new KPrDuplicatObjDia(this, m_pKPresenterDoc);
    if ( dlg->exec() )
    {
        int nbCopy= dlg->nbCopy();
        double angle = dlg->angle();
        double increaseX = dlg->increaseX();
        double increaseY = dlg->increaseY();
        double moveX = dlg->moveX();
        double moveY = dlg->moveY();
        m_canvas->copyObjs();
        m_canvas->setToolEditMode( TEM_MOUSE );
        deSelectAllObjects();
        TQMimeSource *data = TQApplication::clipboard()->data();
        TQCString clip_str = KoStoreDrag::mimeType("application/x-kpresenter");
        if ( data->provides( clip_str ) )
        {
            m_canvas->activePage()->pasteObjs( data->encodedData(clip_str),
                                               nbCopy, angle, increaseX,increaseY, moveX, moveY );
            m_canvas->setMouseSelectedObject(true);
            emit objectSelectedChanged();
        }
    }
    delete dlg;
}

void KPrView::extraArrangePopup()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    TQPoint pnt( TQCursor::pos() );
    m_arrangeObjectsPopup->popup( pnt );
}

void KPrView::extraSendBackward()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->lowerObjs( false );
}

void KPrView::extraBringForward()
{
    m_canvas->setToolEditMode( TEM_MOUSE );
    m_canvas->raiseObjs( false );
}

void KPrView::applyAutoFormat()
{
    m_pKPresenterDoc->getAutoFormat()->readConfig();
    KMacroCommand *macro = 0L;
    m_switchPage=m_pKPresenterDoc->pageList().findRef(m_canvas->activePage());
    m_initSwitchPage=m_switchPage;
    TQPtrList<KoTextObject> list=m_canvas->activePage()->allTextObjects();

    KCommand * cmd2 = applyAutoFormatToCurrentPage( list );
    if ( cmd2 )
    {
        if ( !macro )
            macro = new KMacroCommand( i18n("Apply Autoformat"));
        macro->addCommand( cmd2 );
    }

    while(switchInOtherPage(i18n( "Do you want to apply autoformat in new slide?")) )
    {
        KCommand * cmd = applyAutoFormatToCurrentPage(m_canvas->activePage()->allTextObjects());
        if ( cmd )
        {
            if ( !macro )
                macro = new KMacroCommand( i18n("Apply Autoformat"));
            macro->addCommand( cmd );
        }
    }
    if ( macro )
        m_pKPresenterDoc->addCommand(macro);
    m_switchPage=-1;
    m_initSwitchPage=-1;
}

bool KPrView::switchInOtherPage( const TQString & text )
{
    //there is not other page
    if(m_pKPresenterDoc->pageList().count()==1)
        return false;
    m_switchPage++;
    if( m_switchPage>=(int)m_pKPresenterDoc->pageList().count())
        m_switchPage=0;
    if( m_switchPage==m_initSwitchPage)
        return false;
    if ( KMessageBox::questionYesNo( this, text) != KMessageBox::Yes )
        return false;
    skipToPage(m_switchPage);
    return true;
}

KCommand * KPrView::applyAutoFormatToCurrentPage( const TQPtrList<KoTextObject> & lst)
{
    KMacroCommand *macro = 0L;
    TQPtrList<KoTextObject> list(lst);
    TQPtrListIterator<KoTextObject> fit(list);
    for ( ; fit.current() ; ++fit )
    {
        KCommand *cmd = m_pKPresenterDoc->getAutoFormat()->applyAutoFormat( fit.current() );
        if ( cmd )
        {
            if ( !macro )
                macro = new KMacroCommand( i18n("Apply Autoformat"));
            macro->addCommand( cmd );
        }
    }
    return macro;
}

void KPrView::createStyleFromSelection()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
    {
        KoStyleCollection* coll = m_pKPresenterDoc->styleCollection();
        KoCreateStyleDia *dia = new KoCreateStyleDia( TQStringList(), this, 0 );
        if ( dia->exec() )
        {
            TQString name = dia->nameOfNewStyle();
            KoParagStyle* style = coll->findStyleByDisplayName( name );
            if ( style ) // update existing style
            {
                // TODO confirmation message box
                edit->updateStyleFromSelection( style );
            }
            else // create new style
            {
                style = edit->createStyleFromSelection( name );
                m_pKPresenterDoc->styleCollection()->addStyle( style );
                m_pKPresenterDoc->updateAllStyleLists();
            }
            showStyle( name );
        }
        delete dia;
    }
}

void KPrView::closeObject()
{
    m_canvas->closeObject(true);
}

void KPrView::viewSnapToGrid()
{
    m_pKPresenterDoc->setSnapToGrid( actionViewSnapToGrid->isChecked() );
    m_pKPresenterDoc->setModified( true );
    m_pKPresenterDoc->updateGridButton();
}

void KPrView::alignVerticalTop()
{
    if ( actionAlignVerticalTop->isChecked() )
        m_canvas->alignVertical(KP_TOP );
    else
        actionAlignVerticalTop->setChecked(true);
}

void KPrView::alignVerticalBottom()
{
    if ( actionAlignVerticalBottom->isChecked() )
        m_canvas->alignVertical(KP_BOTTOM );
    else
        actionAlignVerticalBottom->setChecked(true);
}

void KPrView::alignVerticalCenter()
{
    if ( actionAlignVerticalCenter->isChecked() )
        m_canvas->alignVertical(KP_CENTER );
    else
        actionAlignVerticalCenter->setChecked(true);
}

void KPrView::changeVerticalAlignmentStatus( VerticalAlignmentType _type )
{
    switch( _type )
    {
    case KP_CENTER:
        actionAlignVerticalCenter->setChecked( true );
        break;
    case KP_TOP:
        actionAlignVerticalTop->setChecked( true );
        break;
    case KP_BOTTOM:
        actionAlignVerticalBottom->setChecked( true );
        break;
    }
}

void KPrView::autoSpellCheck()
{
    m_pKPresenterDoc->changeBgSpellCheckingState( actionAllowBgSpellCheck->isChecked() );
}

void KPrView::insertFile(  )
{
    KFileDialog fd( TQString(), TQString(), this, 0, true );
    TQStringList filter;
    filter<<"application/x-kpresenter";
    filter<<"application/vnd.oasis.opendocument.presentation";
    fd.setMimeFilter( filter );
    fd.setCaption(i18n("Insert File"));

    KURL url;
    if ( fd.exec() == TQDialog::Accepted )
    {
        url = fd.selectedURL();
        if( url.isEmpty() )
        {
            KMessageBox::sorry( this,
                                i18n("File name is empty."),
                                i18n("Insert File"));
            return;
        }
        insertFile(url.path());
    }
}

void KPrView::insertFile(const TQString &path)
{
    m_pKPresenterDoc->insertFile(path);
}

void KPrView::importStyle()
{
    KPrImportStyleDia dia( m_pKPresenterDoc, m_pKPresenterDoc->styleCollection(), this );
    if ( dia.exec() && !dia.importedStyles().isEmpty() ) {
        m_pKPresenterDoc->styleCollection()->importStyles( dia.importedStyles() );
        m_pKPresenterDoc->setModified( true );
        m_pKPresenterDoc->updateAllStyleLists();
    }
}

void KPrView::backgroundPicture()
{
    switch( m_canvas->activePage()->getBackType())
    {
    case BT_COLOR:
    case BT_BRUSH:
        break;
    case BT_CLIPART:
    case BT_PICTURE:
        KoPicture picture=m_canvas->activePage()->background()->picture();
        savePicture(picture.getKey().filename(), picture);
        break;
    }
}

void KPrView::testAndCloseAllTextObjectProtectedContent()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit && edit->kpTextObject()->isProtectContent())
    {
        m_canvas->setToolEditMode( TEM_MOUSE );
        deSelectAllObjects();
    }
}

void KPrView::updateBgSpellCheckingState()
{
    actionAllowBgSpellCheck->setChecked( m_pKPresenterDoc->backgroundSpellCheckEnabled() );
}

void KPrView::updateRulerInProtectContentMode()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit && getHRuler()) {
        if ( !edit->kpTextObject()->isProtectContent() )
            getHRuler()->changeFlags(KoRuler::F_INDENTS | KoRuler::F_TABS);
        else
            getHRuler()->changeFlags(0);
        getHRuler()->repaint();
    }
}

void KPrView::slotChangeCutState(bool b)
{
    KPrTextView *edit=m_canvas->currentTextObjectView();

    if ( edit && edit->kpTextObject()->isProtectContent())
        actionEditCut->setEnabled( false );
    else
        actionEditCut->setEnabled( b );
}

void KPrView::updatePresentationButton(bool b)
{
    actionScreenStart->setEnabled( b );
}

void KPrView::refreshGroupButton()
{
    bool state=m_canvas->isOneObjectSelected();
    actionExtraGroup->setEnabled(state && m_canvas->numberOfObjectSelected()>1);
    actionExtraUnGroup->setEnabled(state && m_canvas->haveASelectedGroupObj());
}

void KPrView::closeTextObject()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit)
    {
        m_canvas->setToolEditMode( TEM_MOUSE );
        deSelectAllObjects();
    }
}

void KPrView::deSelectAllObjects()
{
    m_canvas->deSelectAllObj();
}

void KPrView::copyLink()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->copyLink();
}

void KPrView::addToBookmark()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
    {
        KoLinkVariable * var=edit->linkVariable();
        if(var)
            edit->addBookmarks(var->url());
    }
}

void KPrView::removeLink()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->removeLink();
}

void KPrView::insertDirectCursor()
{
#if 0
    insertDirectCursor( actionInsertDirectCursor->isChecked());
#endif
}

void KPrView::insertDirectCursor(bool b)
{
    m_pKPresenterDoc->setInsertDirectCursor(b);
}

void KPrView::updateDirectCursorButton()
{
#if 0
    actionInsertDirectCursor->setChecked(m_pKPresenterDoc->insertDirectCursor());
#endif
}

void KPrView::copyTextOfComment()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        edit->copyTextOfComment();
}

void KPrView::slotAddIgnoreAllWord()
{
    KPrTextView *edit=m_canvas->currentTextObjectView();
    if ( edit )
        m_pKPresenterDoc->addSpellCheckIgnoreWord( edit->currentWordOrSelection() );
}

void KPrView::addWordToDictionary()
{
    KPrTextView* edit = m_canvas->currentTextObjectView();
    if ( edit )
    {
        TQString word = edit->wordUnderCursor( *edit->cursor() );
        if ( !word.isEmpty() )
            m_pKPresenterDoc->addWordToDictionary( word );
    }
}

void KPrView::imageEffect()
{
    if (m_canvas->numberOfObjectSelected() > 0) {
        imageEffectDia = new KPrImageEffectDia(this);

        KPrPixmapObject *object=m_canvas->getSelectedImage();

        imageEffectDia->setPixmap(object->getOriginalPixmap());
        imageEffectDia->setEffect(object->getImageEffect(), object->getIEParam1(), object->getIEParam2(),
                                  object->getIEParam3());

        m_canvas->setToolEditMode(TEM_MOUSE);

        if (imageEffectDia->exec()==TQDialog::Accepted) {
            KCommand *cmd=m_canvas->activePage()->setImageEffect(imageEffectDia->getEffect(), imageEffectDia->getParam1(),
                                                                 imageEffectDia->getParam2(), imageEffectDia->getParam3());
            if ( cmd )
                kPresenterDoc()->addCommand( cmd );
        }

        delete imageEffectDia;
        imageEffectDia = 0L;
    }
}

void KPrView::spellAddAutoCorrect (const TQString & originalword, const TQString & newword)
{
    m_pKPresenterDoc->getAutoFormat()->addAutoFormatEntry( originalword, newword );
}

TQPtrList<TDEAction> KPrView::listOfResultOfCheckWord( const TQString &word )
{
    TQPtrList<TDEAction> listAction;
    DefaultDictionary *dict = m_broker->defaultDictionary();
    TQStringList lst = dict->suggest( word );
    if ( !lst.contains( word ))
    {
        TQStringList::ConstIterator it = lst.begin();
        for ( int i = 0; it != lst.end() ; ++it, ++i )
        {
            if ( !(*it).isEmpty() ) // in case of removed subtypes or placeholders
            {
                TDEAction * act = new TDEAction( (*it));
                connect( act, TQ_SIGNAL(activated()), this, TQ_SLOT(slotCorrectWord()) );
                listAction.append( act );
            }
        }
    }

    return listAction;
}

void KPrView::slotCorrectWord()
{
    TDEAction * act = (TDEAction *)(sender());
    KPrTextView* edit = m_canvas->currentTextObjectView();
    if ( edit )
    {
        edit->selectWordUnderCursor( *(edit->cursor()) );
        m_pKPresenterDoc->addCommand(edit->textObject()->replaceSelectionCommand(
                                         edit->cursor(), act->text(),
                                         i18n("Replace Word"), KoTextDocument::Standard));
    }
}

KCommand * KPrView::getPenCmd( const TQString &name, KoPen pen, LineEnd lb, LineEnd le, int flags )
{
    KMacroCommand * macro = NULL;

    //tz TODO remove macro, add parameter for command name to setPen()
    KCommand * cmd = m_canvas->activePage()->setPen( pen, lb, le, flags );
    if( cmd )
    {
        macro = new KMacroCommand( name );
        macro->addCommand( cmd );
    }

    return macro;
}

void KPrView::initialLayoutOfSplitter()
{
    if( !notebar )
    {
        return;
    }

    TQSplitter* splitterVertical = static_cast<TQSplitter*>( notebar->parent() );
    TQValueList<int> tmpList;
    int noteHeight = height() / 25;
    tmpList << height() - noteHeight << noteHeight;
    splitterVertical->setSizes( tmpList );
}

void KPrView::slotChildActivated(bool a)
{
  KoViewChild* ch = child( (KoView*)sender() );
  if ( !ch )
    return;

  if( !a ) {
    m_canvas->exitEditMode();
  }

  KoView::slotChildActivated( a );
}

Broker *KPrView::broker() const
{
    return m_broker;
}

void KPrView::slotUnitChanged( KoUnit::Unit unit )
{
    h_ruler->setUnit( unit );
    v_ruler->setUnit( unit );
    if ( statusBar() )
    {
        m_sbUnitLabel->setText( ' ' + KoUnit::unitDescription( unit ) + ' ' );
        updateObjectStatusBarItem();
    }
}

void KPrView::documentModified( bool b )
{
    if ( !statusBar() )
        return;

    if ( b )
        m_sbModifiedLabel->setPixmap( TDEGlobal::iconLoader()->loadIcon( "action-modified", TDEIcon::Small ) );
    else
        m_sbModifiedLabel->setText( "   " );
}

void KPrView::loadingFinished()
{
  h_ruler->setPageLayout(m_pKPresenterDoc->pageLayout());
  v_ruler->setPageLayout(m_pKPresenterDoc->pageLayout());
  slotUpdateRuler();
  m_pKPresenterDoc->updateZoomRuler();
  updateSideBar();
}

#include "KPrView.moc"
