// A wrapper for KCharsets
// Copyright (C) 2004 Shintaro Matsuoka <shin@shoegazed.org>

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

#ifndef KONVERSATION_IRCCHARSETS_H
#define KONVERSATION_IRCCHARSETS_H

#include <tqmap.h>
#include <tqstringlist.h>


namespace Konversation
{

    class IRCCharsets
    {
        private:
            IRCCharsets();

        public:
            static IRCCharsets *self();

            /**
             * Lists all available encoding names.
             * e.g. "utf8", "iso 8859-1"
             * Encodings which don't work on IRC are excluded. (e.g. utf16)
             * @note It's guaranteed that the order of this list is same with that of @ref availableEncodingDescriptiveNames() .
             */
            TQStringList availableEncodingShortNames();

            /**
             * Lists all available encoding descriptions.
             * e.g. "Unicode ( utf8 )", "Western European ( iso 8859-1 )"
             * Encodings which don't work on IRC are excluded. (e.g. utf16)
             */
            TQStringList availableEncodingDescriptiveNames();

            int availableEncodingsCount();

            TQString shortNameToDescriptiveName( const TQString& shortName );
            TQString descriptiveNameToShortName( const TQString& descriptiveName );

            /**
             * Converts the ambiguous encoding name to a short encoding name
             * Like : iso8859-9 -> iso 8859-9, iso-8859-9 -> iso 8859-9
             * If the ambiguous name is invalid, returns TQString:null.
             * @return a short encoding name or TQString()
             */
            TQString ambiguousNameToShortName( const TQString& ambiguousName );

            /**
             * Returns the encoding index in the short names list or the descriptions list.
             * If the encoding name is invalid, returns -1.
             * @return an index number of the encoding
             */
            int shortNameToIndex( const TQString& shortName );

            /**
             * Checks if the encoding name is in the short encoding names.
             * @see availableEncodingShortNames()
             */
            bool isValidEncoding( const TQString& shortName );

            /**
             * Returns the short name of the most suitable encoding for this locale.
             * @return a short encoding name
             */
            TQString encodingForLocale();

            TQTextCodec* codecForName( const TQString& shortName );

        private:
            TQMap<TQString,TQString> m_shortNameAliases;

            /**
             * short names list
             * you can get this list with @ref availableEncodingShortNames()
             * e.g. iso 8859-1
             */
            TQStringList m_shortNames;

            /**
             * descriptive names list
             * you can get this list with @ref availableEncodingDescriptiveNames();
             * e.g. Western European ( iso 8859-1 )
             */
            TQStringList m_descriptiveNames;

            /**
             * simplified short names list (for internal use)
             * e.g. iso88591
             * used in @ref ambiguousNameToShortName()
             */
            TQStringList m_simplifiedShortNames;

            static IRCCharsets *s_self;
    };

}
#endif                                            // KONVERSATION_IRCCHARSETS_H
