/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2005 Renchi Raju <renchi@pooh.tam.uiuc.edu>
  Copyright (C) 2006 Peter Simonsson <psn@linux.se>
*/

#include "searchbar.h"

#include <tqcheckbox.h>
#include <tqtimer.h>
#include <tqpalette.h>
#include <tqaccel.h>
#include <tqlabel.h>
#include <tqpixmap.h>
#include <tqobjectlist.h>
#include <tqtoolbutton.h>
#include <tqpopupmenu.h>
#include <tqwidgetstack.h>

#include <kdebug.h>
#include <tdeapplication.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <tdelocale.h>

#define SEARCH_FORWARD_MENU 1
#define MATCH_CASE_MENU 2
#define WHOLE_WORDS_ONLY_MENU 3
#define FROM_CURSOR_MENU 4


SearchBar::SearchBar(TQWidget* parent)
: SearchBarBase(parent)
{
    m_searchFoward = false;
    m_matchCase = false;
    m_wholeWords = false;
    m_fromCursor = false;

    setFocusProxy(m_searchEdit);
    TDEIconLoader* iconLoader = kapp->iconLoader();
    m_closeButton->setIconSet(iconLoader->loadIconSet("window-close", TDEIcon::Toolbar, 16));
    m_findNextButton->setIconSet(iconLoader->loadIconSet("go-up", TDEIcon::Toolbar, 16));
    m_findPreviousButton->setIconSet(iconLoader->loadIconSet("go-down", TDEIcon::Toolbar, 16));
    m_statusPixLabel->hide();
    m_statusTextLabel->hide();

    m_timer = new TQTimer(this);

    TQAccel* accel = new TQAccel(this);
    accel->connectItem( accel->insertItem(TQt::Key_Escape), this, TQ_SLOT(hide()));

    connect(m_timer, TQ_SIGNAL(timeout()), TQ_SLOT(slotFind()));
    connect(m_searchEdit, TQ_SIGNAL(textChanged(const TQString&)), TQ_SLOT(slotTextChanged()));
    connect(m_searchEdit, TQ_SIGNAL(returnPressed()), TQ_SLOT(slotFindNext()));
    connect(m_findNextButton, TQ_SIGNAL(clicked()), TQ_SLOT(slotFindNext()));
    connect(m_findPreviousButton, TQ_SIGNAL(clicked()), TQ_SLOT(slotFindPrevious()));
    connect(m_closeButton, TQ_SIGNAL(clicked()), TQ_SLOT(hide()));
    connect(m_optionsButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(showOptionsMenu()));

    m_optionsMenu = new TQPopupMenu(m_optionsButton, "options_menu");
    m_optionsMenu->setCheckable(true);
    m_optionsMenu->insertItem(i18n("Find Forward"), this, TQ_SLOT(toggleSearchFoward()), 0, SEARCH_FORWARD_MENU);
    m_optionsMenu->insertItem(i18n("Case Sensitive"), this, TQ_SLOT(toggleMatchCase()), 0, MATCH_CASE_MENU);
    m_optionsMenu->insertItem(i18n("Whole Words Only"), this, TQ_SLOT(toggleWholeWords()), 0, WHOLE_WORDS_ONLY_MENU);
    m_optionsMenu->insertItem(i18n("From Cursor"), this, TQ_SLOT(toggleFromCursor()), 0, FROM_CURSOR_MENU);

    m_optionsButton->setPopup(m_optionsMenu);
}

SearchBar::~SearchBar()
{
}

void SearchBar::showEvent(TQShowEvent *e)
{
    SearchBarBase::showEvent(e);
    m_searchEdit->selectAll();
}

bool SearchBar::focusedChild()
{
    TQObjectList *l = queryList("TQWidget", 0,0, true);
    TQObjectListIt it( *l ); 
    TQObject *obj;
    bool has=false;

    while ((obj = it.current()) != 0)
    {
        ++it;
        if (((TQWidget*)obj)->hasFocus())
        {
            has=true;
            break;
        }
    }
    delete l;
    return has;
}

void SearchBar::hide()
{
    m_timer->stop();
    SearchBarBase::hide();

    if (focusedChild())
        emit hidden();
}

void SearchBar::slotTextChanged()
{
    m_timer->start(50, true);
}

void SearchBar::slotFind()
{
    if (m_searchEdit->text().isEmpty())
    {
        m_searchEdit->unsetPalette();
        m_findNextButton->setEnabled(false);
        m_findPreviousButton->setEnabled(false);
        setStatus(TQPixmap(), "");
        return;
    }

    emit signalSearchChanged(m_searchEdit->text());
}

void SearchBar::slotFindNext()
{
    if (m_searchEdit->text().isEmpty())
    {
        m_searchEdit->unsetPalette();
        m_findNextButton->setEnabled(false);
        m_findPreviousButton->setEnabled(false);
        setStatus(TQPixmap(), "");
        return;
    }

    emit signalSearchNext();
}

void SearchBar::slotFindPrevious()
{
    if (m_searchEdit->text().isEmpty())
    {
        m_searchEdit->unsetPalette();
        m_findNextButton->setEnabled(false);
        m_findPreviousButton->setEnabled(false);
        setStatus(TQPixmap(), "");
        return;
    }

    emit signalSearchPrevious();
}

void SearchBar::setHasMatch(bool value)
{
    TQPalette pal = m_searchEdit->palette();
    pal.setColor(TQPalette::Active, TQColorGroup::Base, value ? TQt::green : TQt::red);
    m_searchEdit->setPalette(pal);
    m_findNextButton->setEnabled(value);
    m_findPreviousButton->setEnabled(value);
}

void SearchBar::setStatus(const TQPixmap& pix, const TQString& text)
{
    if(!text.isEmpty()) {
        m_statusPixLabel->show();
        m_statusTextLabel->show();
    } else {
        m_statusPixLabel->hide();
        m_statusTextLabel->hide();
    }

    m_statusPixLabel->setPixmap(pix);
    m_statusTextLabel->setText(text);
}

TQString SearchBar::pattern() const
{
    return m_searchEdit->text();
}

bool SearchBar::searchForward() const
{
    return m_searchFoward;
}

bool SearchBar::caseSensitive() const
{
    return m_matchCase;
}

bool SearchBar::wholeWords() const
{
    return m_wholeWords;
}

bool SearchBar::fromCursor() const
{
    return m_fromCursor;
}

void SearchBar::toggleSearchFoward()
{
    m_searchFoward = !m_searchFoward;
    m_optionsMenu->setItemChecked(SEARCH_FORWARD_MENU, m_searchFoward);
    slotTextChanged();
}

void SearchBar::toggleMatchCase()
{
    m_matchCase = !m_matchCase;
    m_optionsMenu->setItemChecked(MATCH_CASE_MENU, m_matchCase);
    slotTextChanged();
}

void SearchBar::toggleWholeWords()
{
    m_wholeWords = !m_wholeWords;
    m_optionsMenu->setItemChecked(WHOLE_WORDS_ONLY_MENU, m_wholeWords);
    slotTextChanged();
}

void SearchBar::toggleFromCursor()
{
    m_fromCursor = !m_fromCursor;
    m_optionsMenu->setItemChecked(FROM_CURSOR_MENU, m_fromCursor);
    slotTextChanged();
}

void SearchBar::showOptionsMenu()
{
  m_optionsButton->openPopup();
}

#include "searchbar.moc"
