/*
 * kcmlisa.cpp
 *
 * Copyright (c) 2000,2001 Alexander Neundorf <neundorf@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "kcmlisa.h"

#include "findnic.h"
#include "setupwizard.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <time.h>

#include <tqtooltip.h>
#include <tqfile.h>
#include <tqspinbox.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>
#include <tqgrid.h>
#include <tqvbuttongroup.h>

#include <tdeapplication.h>
#include <kprocess.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <keditlistbox.h>
#include <krestrictedline.h>
#include <kdialogbase.h>
#include <tdetempfile.h>

#include <kdebug.h>

LisaSettings::LisaSettings(const TQString& config, TQWidget *parent)
: TDECModule(parent, "kcmlanbrowser")
,m_config(config,false,true)
,m_wizard(0)
,m_configFilename(config)
,m_changed(false)
{
   TQVBoxLayout *layout = new TQVBoxLayout(this, KDialog::marginHint(), KDialog::spacingHint());
   layout->setAutoAdd(true);

   TQWidget *dummy(0);

   TQVButtonGroup *gb=new TQVButtonGroup(i18n("Tell LISa Daemon How to Search for Hosts"),this);
   gb->setInsideSpacing(10);

   m_useNmblookup=new TQCheckBox(i18n("Send &NetBIOS broadcasts using nmblookup"), gb);
   TQToolTip::add(m_useNmblookup,i18n("Only hosts running SMB servers will answer"));

   m_sendPings=new TQCheckBox(i18n("Send &pings (ICMP echo packets)"), gb);
   TQToolTip::add(m_sendPings,i18n("All hosts running TCP/IP will answer"));

   TQHBox *hbox=new TQHBox(gb);
   hbox->setSpacing(10);

   dummy=new TQWidget(hbox);
   dummy->setMinimumWidth(10);
   TQLabel *label=new TQLabel(i18n("To these &IP addresses:"),hbox);
   TQString comment = i18n("Enter all ranges to scan, using the format '192.168.0.1/255.255.255.0;10.0.0.1;255.0.0.0'");
   TQToolTip::add(label,comment);
   m_pingAddresses=new KRestrictedLine(hbox,"a","0123456789.-/;");
   TQToolTip::add(m_pingAddresses, comment);
   label->setBuddy(m_pingAddresses);

   TQGrid *addressesGrid = new TQGrid(2, TQt::Horizontal, this);
   addressesGrid->setSpacing(10);
   layout->setStretchFactor(addressesGrid,0);

   label=new TQLabel(i18n("&Broadcast network address:"),addressesGrid);
   comment=i18n("Your network address/subnet mask (e.g. 192.168.0.0/255.255.255.0;)");
   TQToolTip::add(label, comment);

   m_broadcastNetwork=new KRestrictedLine(addressesGrid,"a","0123456789./;");
   TQToolTip::add(m_broadcastNetwork,comment);
   label->setBuddy(m_broadcastNetwork);

   label=new TQLabel(i18n("&Trusted IP addresses:"),addressesGrid);
   comment = i18n("Usually your network address/subnet mask (e.g. 192.168.0.0/255.255.255.0;)");
   TQToolTip::add(label, comment);

   m_allowedAddresses=new KRestrictedLine(addressesGrid,"a","0123456789./;");
   TQToolTip::add(m_allowedAddresses, comment);
   label->setBuddy(m_allowedAddresses);

   dummy=new TQWidget(this);
   layout->setStretchFactor(dummy,10);

   hbox = new TQHBox(this);
   hbox->setSpacing(10);
//   m_autoSetup=new TQPushButton(i18n("&Guided LISa Setup..."),hbox);
   m_autoSetup=new TQPushButton(i18n("Setup Wizard..."),hbox);
   m_autoSetup->setFixedWidth( m_autoSetup->sizeHint().width() );

   m_suggestSettings=new TQPushButton(i18n("&Suggest Settings"),hbox);

   new TQWidget(hbox);

   m_advancedSettingsButton=new TQPushButton(i18n("Ad&vanced Settings..."),hbox);

   m_lisaAdvancedDlg=new KDialogBase(0,0,true,i18n("Advanced Settings for LISa"),KDialogBase::Close, KDialogBase::Close);
   connect(m_advancedSettingsButton,TQ_SIGNAL(clicked()),m_lisaAdvancedDlg,TQ_SLOT(show()));

   TQVBox *vbox=m_lisaAdvancedDlg->makeVBoxMainWidget();

   m_pingNames=new KEditListBox(i18n("&Additionally Check These Hosts"),vbox,"a",false, KEditListBox::Add|KEditListBox::Remove);
   m_pingNames->setMinimumHeight(180);
   TQToolTip::add(m_pingNames,i18n("The hosts listed here will be pinged"));

   dummy=new TQWidget(vbox);
   dummy->setMinimumHeight(10);
   m_deliverUnnamedHosts=new TQCheckBox(i18n("Show &hosts without DNS names"),vbox);

   TQGrid *advGrid = new TQGrid(2, TQt::Horizontal, vbox);
   advGrid->setSpacing(10);

   label=new TQLabel(i18n("Host list update interval:"),advGrid);
   TQToolTip::add(label,i18n("Search hosts after this number of seconds"));
   m_updatePeriod=new TQSpinBox(30,1800,10,advGrid);
   m_updatePeriod->setSuffix(i18n(" sec"));
   TQToolTip::add(m_updatePeriod,i18n("Search hosts after this number of seconds"));

   m_secondScan=new TQCheckBox(i18n("Always check twice for hosts when searching"),advGrid);
   new TQWidget(advGrid);

   label=new TQLabel(i18n("Wait for replies from hosts after first scan:"),advGrid);
   TQToolTip::add(label,i18n("How long to wait for replies to the ICMP echo requests from hosts"));
   m_firstWait=new TQSpinBox(10,1000,50,advGrid);
   m_firstWait->setSuffix(i18n(" ms"));
   TQToolTip::add(m_firstWait,i18n("How long to wait for replies to the ICMP echo requests from hosts"));

   label=new TQLabel(i18n("Wait for replies from hosts after second scan:"),advGrid);
   TQToolTip::add(label,i18n("How long to wait for replies to the ICMP echo requests from hosts"));
   m_secondWait=new TQSpinBox(0,1000,50,advGrid);
   m_secondWait->setSuffix(i18n(" ms"));
   TQToolTip::add(m_secondWait,i18n("How long to wait for replies to the ICMP echo requests from hosts"));

   label=new TQLabel(i18n("Max. number of ping packets to send at once:"),advGrid);
   m_maxPingsAtOnce=new TQSpinBox(8,1024,5,advGrid);

   dummy=new TQWidget(advGrid);
   dummy->setMinimumHeight(10);

   connect(m_secondScan,TQ_SIGNAL(toggled(bool)),m_secondWait,TQ_SLOT(setEnabled(bool)));
   connect(m_sendPings,TQ_SIGNAL(toggled(bool)),m_pingAddresses,TQ_SLOT(setEnabled(bool)));

   connect(m_pingAddresses,TQ_SIGNAL(textChanged(const TQString&)),this,TQ_SIGNAL(changed()));
   connect(m_allowedAddresses,TQ_SIGNAL(textChanged(const TQString&)),this,TQ_SIGNAL(changed()));
   connect(m_broadcastNetwork,TQ_SIGNAL(textChanged(const TQString&)),this,TQ_SIGNAL(changed()));

   connect(m_pingAddresses,TQ_SIGNAL(returnPressed()),this,TQ_SIGNAL(changed()));
   connect(m_allowedAddresses,TQ_SIGNAL(returnPressed()),this,TQ_SIGNAL(changed()));
   connect(m_broadcastNetwork,TQ_SIGNAL(returnPressed()),this,TQ_SIGNAL(changed()));

   connect(m_sendPings,TQ_SIGNAL(toggled(bool)),this,TQ_SIGNAL(changed()));
   connect(m_firstWait,TQ_SIGNAL(valueChanged(int)),this,TQ_SIGNAL(changed()));
   connect(m_secondWait,TQ_SIGNAL(valueChanged(int)),this,TQ_SIGNAL(changed()));
   connect(m_maxPingsAtOnce,TQ_SIGNAL(valueChanged(int)),this,TQ_SIGNAL(changed()));
   connect(m_secondScan,TQ_SIGNAL(toggled(bool)),this,TQ_SIGNAL(changed()));
   connect(m_deliverUnnamedHosts,TQ_SIGNAL(toggled(bool)),this,TQ_SIGNAL(changed()));
   connect(m_updatePeriod,TQ_SIGNAL(valueChanged(int)),this,TQ_SIGNAL(changed()));
   connect(m_pingNames,TQ_SIGNAL(changed()),this,TQ_SIGNAL(changed()));
   connect(m_useNmblookup,TQ_SIGNAL(toggled(bool)),this,TQ_SIGNAL(changed()));
   connect(m_autoSetup,TQ_SIGNAL(clicked()),this,TQ_SLOT(autoSetup()));
   connect(m_suggestSettings,TQ_SIGNAL(clicked()),this,TQ_SLOT(suggestSettings()));
   connect(this, TQ_SIGNAL(changed()), TQ_SLOT(slotChanged()));
}

void LisaSettings::load()
{
   int secondWait=m_config.readNumEntry("SecondWait",-1);
   if (secondWait<0)
   {
      m_secondWait->setValue(300);
      m_secondScan->setChecked(false);
      m_secondWait->setEnabled(false);
   }
   else
   {
      m_secondWait->setValue(secondWait*10);
      m_secondScan->setChecked(true);
      m_secondWait->setEnabled(true);
   }
   m_deliverUnnamedHosts->setChecked(m_config.readNumEntry("DeliverUnnamedHosts",0));

   m_firstWait->setValue(m_config.readNumEntry("FirstWait",30)*10);
   m_maxPingsAtOnce->setValue(m_config.readNumEntry("MaxPingsAtOnce",256));
   m_updatePeriod->setValue(m_config.readNumEntry("UpdatePeriod",300));
   m_pingAddresses->setText(m_config.readEntry("PingAddresses","192.168.0.0/255.255.255.0;192.168.100.0-192.168.100.254"));
   m_sendPings->setChecked(!m_pingAddresses->text().isEmpty());
   m_allowedAddresses->setText(m_config.readEntry("AllowedAddresses","192.168.0.0/255.255.255.0"));
   m_broadcastNetwork->setText(m_config.readEntry("BroadcastNetwork","192.168.0.0/255.255.255.0"));
   m_pingNames->clear();
   m_pingNames->insertStringList(m_config.readListEntry("PingNames",';'));
   int i=m_config.readNumEntry("SearchUsingNmblookup",1);
   m_useNmblookup->setChecked(i!=0);
   m_changed = false;
}

void LisaSettings::save()
{
   if (!m_changed) return;

   if ( getuid()==0)
   {
      if (m_secondScan->isChecked())
         m_config.writeEntry("SecondWait",(m_secondWait->value()+5)/10);
      else
         m_config.writeEntry("SecondWait",-1);

      if (m_useNmblookup->isChecked())
         m_config.writeEntry("SearchUsingNmblookup",1);
      else
         m_config.writeEntry("SearchUsingNmblookup",0);

      if (m_deliverUnnamedHosts->isChecked())
         m_config.writeEntry("DeliverUnnamedHosts",1);
      else
         m_config.writeEntry("DeliverUnnamedHosts",0);

      m_config.writeEntry("FirstWait",(m_firstWait->value()+5)/10);
      m_config.writeEntry("MaxPingsAtOnce",m_maxPingsAtOnce->value());
      m_config.writeEntry("UpdatePeriod",m_updatePeriod->value());
      m_config.writeEntry("PingAddresses",m_sendPings->isChecked()?m_pingAddresses->text():"");
      m_config.writeEntry("AllowedAddresses",m_allowedAddresses->text());
      m_config.writeEntry("BroadcastNetwork",m_broadcastNetwork->text());
      TQStringList writeStuff;
      for (int i=0; i<m_pingNames->count(); i++)
         writeStuff.append(m_pingNames->text(i));
      m_config.writeEntry("PingNames",writeStuff,';');

      m_config.sync();
      chmod(TQFile::encodeName(m_configFilename),S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
   }
   else
   {
      //ok, now it gets harder
      //we are not root but we want to write into /etc ....
      //any idea how to do it better ?
      KTempFile tmp;

      if (tmp.status() == 0 && tmp.textStream())
      {
         m_tmpFilename = tmp.name();
         TQTextStream &confStream = *(tmp.textStream());
         if (m_secondScan->isChecked())
            confStream<<"SecondWait = "<<(m_secondWait->value()+5)/10<<"\n";
         else
            confStream<<"SecondWait = -1\n";

         if (m_useNmblookup->isChecked())
            confStream<<"SearchUsingNmblookup = 1\n";
         else
            confStream<<"SearchUsingNmblookup = 0\n";

         if (m_deliverUnnamedHosts->isChecked())
            confStream<<"DeliverUnnamedHosts = 1\n";
         else
            confStream<<"DeliverUnnamedHosts = 0\n";

         confStream<<"FirstWait = "<< (m_firstWait->value()+5)/10 <<"\n";
         confStream<<"MaxPingsAtOnce = "<<m_maxPingsAtOnce->value()<<"\n";
         confStream<<"UpdatePeriod = "<<m_updatePeriod->value()<<"\n";
         confStream<<"PingAddresses = "<<m_pingAddresses->text().latin1()<<"\n";
         confStream<<"AllowedAddresses = "<<m_allowedAddresses->text().latin1()<<"\n";
         confStream<<"BroadcastNetwork = "<<m_broadcastNetwork->text().latin1()<<"\n";
         TQString writeStuff;
         for (int i=0; i<m_pingNames->count(); i++)
            writeStuff=writeStuff+m_pingNames->text(i).latin1()+";";

         confStream<<"PingNames = "<<writeStuff.latin1()<<"\n";
         tmp.close();
         TQString suCommand=TQString("cp '%1' '%2'; chmod 644 '%3'; rm -f '%4'").arg(m_tmpFilename).arg(m_configFilename).arg(m_configFilename).arg(m_tmpFilename);
         TDEProcess *proc = new TDEProcess();
         connect(proc, TQ_SIGNAL(processExited(TDEProcess *)), this, TQ_SLOT(saveDone(TDEProcess *)));
         *proc<<"tdesu"<<"-c"<<suCommand;
         TDEApplication::setOverrideCursor(TQt::waitCursor);
         setEnabled(false);
         if ( !proc->start() )
             delete proc;
      }
      else
         KMessageBox::sorry(0,i18n("Saving the results to %1 failed.").arg(m_configFilename));
   }
}

void LisaSettings::suggestSettings()
{
   NICList *nics=findNICs();
   if (nics->count()==0)
   {
      KMessageBox::sorry(0,i18n("No network interface cards found."));
      delete nics;
      return;
   }
   MyNIC *nic=nics->first();
   LisaConfigInfo lci;
   suggestSettingsForNic(nic,lci);
   m_pingAddresses->setText(lci.pingAddresses);
   m_sendPings->setChecked(!m_pingAddresses->text().isEmpty());
   m_broadcastNetwork->setText(lci.broadcastNetwork);
   m_allowedAddresses->setText(lci.allowedAddresses);
   m_secondWait->setValue(lci.secondWait*10);
   m_secondScan->setChecked(lci.secondScan);
   m_secondWait->setEnabled(lci.secondScan);
   m_firstWait->setValue(lci.firstWait*10);
   m_maxPingsAtOnce->setValue(lci.maxPingsAtOnce);
   m_updatePeriod->setValue(lci.updatePeriod);
   m_useNmblookup->setChecked(lci.useNmblookup);
   m_deliverUnnamedHosts->setChecked(lci.unnamedHosts);

   if (nics->count()>1)
   {
      TQString msg(i18n("You have more than one network interface installed.<br>"
                       "Please make sure the suggested settings are correct.<br>"
                       "<br>The following interfaces were found:<br><br>"));
      //not that easy to handle
      for (MyNIC* tmp=nics->first(); tmp!=0; tmp=nics->next())
      {
         msg+="<b>"+tmp->name+": </b>"+tmp->addr+"/"+tmp->netmask+";<br>";
      }
      KMessageBox::information(0,TQString("<html>%1</html>").arg(msg));
   }

   emit changed();
   delete nics;
}

void LisaSettings::autoSetup()
{
   LisaConfigInfo lci;
   if (m_wizard==0)
      m_wizard=new SetupWizard(this,&lci);
   else
      m_wizard->clearAll();
   int result=m_wizard->exec();

   if (result!=TQDialog::Accepted)
      return;

   m_pingAddresses->setText(lci.pingAddresses);
   m_sendPings->setChecked(!m_pingAddresses->text().isEmpty());
   m_broadcastNetwork->setText(lci.broadcastNetwork);
   m_allowedAddresses->setText(lci.allowedAddresses);
   m_secondWait->setValue(lci.secondWait*10);
   m_secondScan->setChecked(lci.secondScan);
   m_secondWait->setEnabled(lci.secondScan);
   m_firstWait->setValue(lci.firstWait*10);
   m_maxPingsAtOnce->setValue(lci.maxPingsAtOnce);
   m_updatePeriod->setValue(lci.updatePeriod);
   m_useNmblookup->setChecked(lci.useNmblookup);
   m_deliverUnnamedHosts->setChecked(lci.unnamedHosts);

   emit changed();
   return;
}

void LisaSettings::saveDone(TDEProcess *proc)
{
   unlink(TQFile::encodeName(m_tmpFilename));
   TDEApplication::restoreOverrideCursor();
   setEnabled(true);
   KMessageBox::information(0,i18n("The configuration has been saved to /etc/lisarc.\n"
                                   "Make sure that the LISa daemon is started,\n e.g. using an init script when booting.\n"
                                   "You can find examples and documentation at http://lisa-home.sourceforge.net ."));

   delete(proc);
}

void LisaSettings::slotChanged()
{
   m_changed = true;
}

#include "kcmlisa.moc"

