/***************************************************************************
 *   Copyright (C) 2004 by Paulo Moura Guedes                              *
 *   moura@kdewebdev.org                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/
#ifndef XSL_H
#define XSL_H

#include <tqobject.h>

class XSLTPrivate;

/**
    @author Paulo Moura Guedes <moura@kdewebdev.org>
    
    Taken from kopetexsl. Kudos to the Kopete team.
 * 
 * This class provides an easy to use interface to basic
 * libxslt transformations.
*/
class XSLT : public TQObject
{
    TQ_OBJECT
    

    TQ_PROPERTY( Flags flags READ flagsProperty WRITE setFlagsProperty )
    TQ_PROPERTY( bool isValid READ isValid )

    TQ_SETS( Flags )

public:
    /**
     * Special flags to be used during the transformation process. Passed
     * into the engine as processing instructions.
     */
    enum Flags
    {
        TransformAllMessages = 1
    };

    /**
     * Constructor.
     *
     * Constructs a new XSLT parser using the provided XSLT document
     */
    XSLT( const TQString &xsltDocument, TQObject *parent = 0L );

    virtual ~XSLT();

    /**
     * Set the XSLT document
     *
     * @return an ORed set of @ref Flags, or 0 if none
     */
    void setXSLT( const TQString &document );

    /**
     * Transforms the XML string using the XSLT document, synchronously
     *
     * @param xmlString The source XML
     * @return The result of the transformation
     */
    TQString transform( const TQString &xmlString );

    /**
     * Transforms the XML string using the XSLT document, asynchronously
     *
     * @param xmlString The source XML
     * @param target The TQObject that contains the slot to be executed when processing is complete
     * @param slotCompleted A slot that accepts a TQVariant & paramater, that is the result
     * of the transformation
     */
    void transformAsync( const TQString &xmlString, TQObject *target, const char *slotCompleted );

    /**
     * Check whether the XSLT document is valid
     *
     * @return Whether the document represents a valid XSLT stylesheet
     */
    bool isValid() const;

    /**
     * @return An ORed list of Flags that the current stylesheet provides via processing instructions
     */
    unsigned int flags() const;

    /**
     * Sets flags to be used for the transformation.
     *
     * @param flags An ORed list of flags
     */
    void setFlags( unsigned int flags );

    inline XSLT::Flags flagsProperty() const { return (XSLT::Flags)flags(); }
    inline void setFlagsProperty( XSLT::Flags newflags ) { setFlags((XSLT::Flags)newflags); }

private:
    XSLTPrivate *d;
};

#endif
