/* This file is part of the KDE project
   Copyright (C) 2002 Ariya Hidayat <ariya@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KPrTransEffectDia.h"

#include "global.h"
#include "KPrBackground.h"
#include "KPrObject.h"
#include "KPrTextObject.h"
#include "KPrView.h"
#include "KPrDocument.h"
#include "KPrCanvas.h"
#include "KPrPage.h"
#include "KPrPageEffects.h"
#include "KPrSoundPlayer.h"

#include <tqsplitter.h>
#include <tqheader.h>
#include <tqwmatrix.h>
#include <tqvbox.h>
#include <tqlayout.h>
#include <tqimage.h>
#include <tqpushbutton.h>
#include <tqcheckbox.h>
#include <tqslider.h>
#include <tqtooltip.h>
#include <tqwhatsthis.h>
#include <tqframe.h>
#include <tqlabel.h>
#include <tqcombobox.h>

#include <kdebug.h>
#include <kurlrequester.h>
#include <tdelocale.h>
#include <knuminput.h>
#include <kiconloader.h>
#include <kurl.h>
#include <tdestandarddirs.h>
#include <tdefiledialog.h>


KPrEffectPreview::KPrEffectPreview( TQWidget *parent, KPrDocument *_doc, KPrView *_view )
    : TQLabel( parent ), doc( _doc ), view( _view ), m_pageEffect( 0 )
{
    setFrameStyle( StyledPanel | Sunken );
}

void KPrEffectPreview::setPixmap( const TQPixmap& pixmap )
{
    // find the right size
    TQRect rect = pixmap.rect();
    int w = rect.width();
    int h = rect.height();
    if ( w > h ) {
        w = 297;
        h = 210;
    }
    else if ( w < h ) {
        w = 210;
        h = 297;
    }
    else if ( w == h ) {
        w = 297;
        h = 297;
    }

    setMinimumSize( w, h );

    // create smaller pixmap for preview
    const TQImage img( pixmap.convertToImage().smoothScale( w, h, TQImage::ScaleFree ) );
    m_pixmap.convertFromImage( img );

    TQLabel::setPixmap( m_pixmap );
}

void KPrEffectPreview::run( PageEffect effect, EffectSpeed speed )
{
    TQRect rect = m_pixmap.rect();
    m_target.resize( rect.size() );
    m_target.fill( TQt::black );

    //kPchangePages( this, m_pixmap, m_target, effect, speed );
    if ( m_pageEffect )
    {
        m_pageEffectTimer.stop();
        TQObject::disconnect( &m_pageEffectTimer, TQ_SIGNAL( timeout() ), this, TQ_SLOT( slotDoPageEffect() ) );

        m_pageEffect->finish();

        delete m_pageEffect;
        m_pageEffect = 0;
        TQLabel::repaint();
    }

    m_pageEffect = new KPrPageEffects( this, m_target, effect, speed );
    if ( m_pageEffect->doEffect() )
    {
        delete m_pageEffect;
        m_pageEffect = 0;
        TQLabel::update();
    }
    else
    {
        connect( &m_pageEffectTimer, TQ_SIGNAL( timeout() ), TQ_SLOT( slotDoPageEffect() ) );
        m_pageEffectTimer.start( 50, true );
    }

    //TQLabel::update();
}


void KPrEffectPreview::slotDoPageEffect()
{
    if ( m_pageEffect->doEffect() )
    {
        m_pageEffectTimer.stop();
        TQObject::disconnect( &m_pageEffectTimer, TQ_SIGNAL( timeout() ), this, TQ_SLOT( slotDoPageEffect() ) );
        delete m_pageEffect;
        m_pageEffect = 0;
        TQLabel::update();
    }
    else
    {
        m_pageEffectTimer.start( 50, true );
    }
}


KPrTransEffectDia::KPrTransEffectDia( TQWidget *parent, const char *name,
                                    KPrDocument *_doc, KPrView *_view )
    : KDialogBase( parent, name, true, "", KDialogBase::User1|Ok|Cancel ),
      doc( _doc ), view( _view ), soundPlayer( 0 )
{
    enableButtonSeparator( true );

    TQWidget *page = new TQWidget( this );
    setMainWidget(page);

    TQBoxLayout *topLayout = new TQHBoxLayout( page, KDialog::marginHint(), KDialog::spacingHint() );
    TQWidget* leftpart = new TQWidget( page );
    topLayout->addWidget( leftpart );
    TQWidget* rightpart = new TQWidget( page );
    topLayout->addWidget( rightpart );

    // right-side of the dialog, for showing effect preview

    TQVBoxLayout *rightlayout = new TQVBoxLayout( rightpart, KDialog::marginHint(), KDialog::spacingHint() );
    rightlayout->setAutoAdd( true );

    effectPreview = new KPrEffectPreview( rightpart, doc, view );

    int pgnum = view->getCurrPgNum() - 1;  // getCurrPgNum() is 1-based
    KPrPage* pg = doc->pageList().at( pgnum );

    // pixmap for effect preview
    TQRect rect= pg->getZoomPageRect();
    TQPixmap pix( rect.size() );
    pix.fill( TQt::white );
    view->getCanvas()->drawPageInPix( pix, pgnum, 100 );
    effectPreview->setPixmap( pix );

    pageEffect = pg->getPageEffect();
    speed = pg->getPageEffectSpeed();

    TQVBoxLayout *leftlayout = new TQVBoxLayout( leftpart, KDialog::marginHint(), KDialog::spacingHint() );
    leftlayout->setAutoAdd( true );

    new TQLabel( i18n("Effect:"), leftpart );

    effectList = new TQListBox( leftpart );
    effectList->insertItem( i18n( "No Effect" ) );
    effectList->insertItem( i18n( "Close Horizontal" ) );
    effectList->insertItem( i18n( "Close Vertical" ) );
    effectList->insertItem( i18n( "Close From All Directions" ) );
    effectList->insertItem( i18n( "Open Horizontal" ) );
    effectList->insertItem( i18n( "Open Vertical" ) );
    effectList->insertItem( i18n( "Open From All Directions" ) );
    effectList->insertItem( i18n( "Interlocking Horizontal 1" ) );
    effectList->insertItem( i18n( "Interlocking Horizontal 2" ) );
    effectList->insertItem( i18n( "Interlocking Vertical 1" ) );
    effectList->insertItem( i18n( "Interlocking Vertical 2" ) );
    effectList->insertItem( i18n( "Surround 1" ) );
    effectList->insertItem( i18n( "Fly Away 1" ) );
    effectList->insertItem( i18n( "Blinds Horizontal" ) );
    effectList->insertItem( i18n( "Blinds Vertical" ) );
    effectList->insertItem( i18n( "Box In" ) );
    effectList->insertItem( i18n( "Box Out" ) );
    effectList->insertItem( i18n( "Checkerboard Across" ) );
    effectList->insertItem( i18n( "Checkerboard Down" ) );
    effectList->insertItem( i18n( "Cover Down" ) );
    effectList->insertItem( i18n( "Uncover Down" ) );
    effectList->insertItem( i18n( "Cover Up" ) );
    effectList->insertItem( i18n( "Uncover Up" ) );
    effectList->insertItem( i18n( "Cover Left" ) );
    effectList->insertItem( i18n( "Uncover Left" ) );
    effectList->insertItem( i18n( "Cover Right" ) );
    effectList->insertItem( i18n( "Uncover Right" ) );
    effectList->insertItem( i18n( "Cover Left-Up" ) );
    effectList->insertItem( i18n( "Uncover Left-Up" ) );
    effectList->insertItem( i18n( "Cover Left-Down" ) );
    effectList->insertItem( i18n( "Uncover Left-Down" ) );
    effectList->insertItem( i18n( "Cover Right-Up" ) );
    effectList->insertItem( i18n( "Uncover Right-Up" ) );
    effectList->insertItem( i18n( "Cover Right-Bottom" ) );
    effectList->insertItem( i18n( "Uncover Right-Bottom" ) );
    effectList->insertItem( i18n( "Dissolve" ) );
    effectList->insertItem( i18n( "Strips Left-Up" ) );
    effectList->insertItem( i18n( "Strips Left-Down" ) );
    effectList->insertItem( i18n( "Strips Right-Up" ) );
    effectList->insertItem( i18n( "Strips Right-Down" ) );
    effectList->insertItem( i18n( "Melting" ) );
    effectList->insertItem( i18n( "Random Transition" ) );
    effectList->setCurrentItem( static_cast<int>( pageEffect ) );

    // workaround, because Random Effect is always negative
    if( pageEffect == PEF_RANDOM )
        effectList->setCurrentItem( effectList->count()-1 );

    connect( effectList, TQ_SIGNAL(highlighted(int)), this, TQ_SLOT(effectChanged(int)) );
    connect( effectList, TQ_SIGNAL( doubleClicked ( TQListBoxItem *) ), this, TQ_SLOT( effectChanged()) );

    new TQLabel( i18n("Speed:"), leftpart );

    TQWidget* sp = new TQWidget( leftpart );
    TQBoxLayout* speedLayout = new TQHBoxLayout( sp, KDialog::marginHint(), KDialog::spacingHint() );
    speedLayout->setAutoAdd( true );

    speedCombo = new TQComboBox( sp );
    speedCombo->insertItem(i18n("Slow") );
    speedCombo->insertItem(i18n("Medium") );
    speedCombo->insertItem(i18n("Fast") );


    speedCombo->setCurrentItem( speed );

    connect( speedCombo, TQ_SIGNAL(activated(int)), this, TQ_SLOT(speedChanged(int)) );


    TQWidget* previewgrp = new TQWidget( leftpart );
    TQBoxLayout* previewLayout = new TQHBoxLayout( previewgrp, KDialog::marginHint(), KDialog::spacingHint() );
    previewLayout->setAutoAdd( true );

    automaticPreview = new TQCheckBox( i18n( "Automatic preview" ), previewgrp );
    automaticPreview->setChecked( true );

    TQWidget* previewspacer = new TQWidget( previewgrp );
    previewspacer->setSizePolicy( TQSizePolicy( TQSizePolicy::Expanding,
                                               TQSizePolicy::Expanding ) );

    previewButton = new TQPushButton( previewgrp );
    previewButton->setText( i18n("Preview") );
    connect( previewButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(preview()) );

    TQFrame* line = new TQFrame( leftpart );
    line->setFrameStyle( TQFrame::HLine | TQFrame::Sunken );

    soundFileName = pg->getPageSoundFileName();
    soundEffect = pg->getPageSoundEffect();

    checkSoundEffect = new TQCheckBox( i18n( "Sound effect" ), leftpart );
    checkSoundEffect->setChecked( soundEffect );
    connect( checkSoundEffect, TQ_SIGNAL( clicked() ), this, TQ_SLOT( soundEffectChanged() ) );

    TQWidget* soundgrp = new TQWidget( leftpart );
    TQBoxLayout* soundLayout = new TQHBoxLayout( soundgrp, KDialog::marginHint(), KDialog::spacingHint() );
    soundLayout->setAutoAdd( true );

    lSoundEffect = new TQLabel( i18n( "File name:" ), soundgrp );
    requester = new KURLRequester( soundgrp );
    requester->setURL( soundFileName );
    connect( requester, TQ_SIGNAL( openFileDialog( KURLRequester * ) ),
             this, TQ_SLOT( slotRequesterClicked( KURLRequester * ) ) );
    connect( requester, TQ_SIGNAL( textChanged( const TQString& ) ),
             this, TQ_SLOT( slotSoundFileChanged( const TQString& ) ) );

    buttonTestPlaySoundEffect = new TQPushButton( soundgrp );
    buttonTestPlaySoundEffect->setPixmap( BarIcon("1rightarrow", TDEIcon::SizeSmall) );
    TQToolTip::add( buttonTestPlaySoundEffect, i18n("Play") );

    connect( buttonTestPlaySoundEffect, TQ_SIGNAL( clicked() ), this, TQ_SLOT( playSound() ) );

    buttonTestStopSoundEffect = new TQPushButton( soundgrp );
    buttonTestStopSoundEffect->setPixmap( BarIcon("media-playback-stop", TDEIcon::SizeSmall) );
    TQToolTip::add( buttonTestStopSoundEffect, i18n("Stop") );

    connect( buttonTestStopSoundEffect, TQ_SIGNAL( clicked() ), this, TQ_SLOT( stopSound() ) );

    soundEffect = pg->getPageSoundEffect();
    setButtonText(KDialogBase::User1,i18n( "Apply &Global" ));

    slideTime = pg->getPageTimer();

    new TQLabel( i18n("Automatically advance to the next slide after:"), rightpart );

    timeSlider = new KIntNumInput( slideTime, rightpart );
    timeSlider->setRange( 1, 600, 1 );
    timeSlider->setSuffix( i18n( " seconds" ) );
    connect( timeSlider, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(timeChanged(int)) );

    TQWidget* rspacer = new TQWidget( rightpart );
    rspacer->setSizePolicy( TQSizePolicy( TQSizePolicy::Minimum, TQSizePolicy::Expanding ) );

    TQWidget* lspacer = new TQWidget( leftpart );
    lspacer->setMinimumSize( 10, spacingHint() );

    soundEffectChanged();
}

void KPrTransEffectDia::preview()
{
    if( pageEffect==PEF_NONE)
        return;
    effectPreview->run( pageEffect, speed );
}

void KPrTransEffectDia::effectChanged()
{
    effectChanged( effectList->currentItem() );
}

void KPrTransEffectDia::effectChanged( int index )
{
    if( effectList->currentText() == i18n( "Random Transition" ) )
        pageEffect = PEF_RANDOM;
    else
        pageEffect =  static_cast<PageEffect>( index );

    if( automaticPreview->isChecked() ) preview();
}

void KPrTransEffectDia::speedChanged( int value )
{
    speed = static_cast<EffectSpeed>(value);
}

void KPrTransEffectDia::timeChanged( int value )
{
    if( value <= 0 ) value = 1;
    slideTime = value;
}

void KPrTransEffectDia::soundEffectChanged()
{
    soundEffect = checkSoundEffect->isChecked();

    lSoundEffect->setEnabled( checkSoundEffect->isChecked() );
    requester->setEnabled( checkSoundEffect->isChecked() );

    if ( !requester->url().isEmpty() ) {
        buttonTestPlaySoundEffect->setEnabled( checkSoundEffect->isChecked() );
        buttonTestStopSoundEffect->setEnabled( checkSoundEffect->isChecked() );
    }
    else {
        buttonTestPlaySoundEffect->setEnabled( false );
        buttonTestStopSoundEffect->setEnabled( false );
    }
}

static TQString getSoundFileFilter()
{
    TQStringList fileList;
    fileList << "wav" << "au" << "mp3" << "mp1" << "mp2" << "mpg" << "dat"
             << "mpeg" << "ogg" << "cdda" << "cda " << "vcd" << "null";
    fileList.sort();

    bool comma = false;
    TQString full, str;
    TQStringList::ConstIterator end( fileList.end() );
    for ( TQStringList::ConstIterator it = fileList.begin(); it != end; ++it ) {
        if ( comma )
            str += '\n';
        comma = true;
        str += TQString( i18n( "*.%1|%2 Files" ) ).arg( *it ).arg( (*it).upper() );

        full += TQString( "*.") + (*it) + ' ';
    }

    str = full + '|' + i18n( "All Supported Files" ) + '\n' + str;
    str += "\n*|" + i18n( "All Files" );

    return str;
}

void KPrTransEffectDia::slotRequesterClicked( KURLRequester * )
{
    TQString filter = getSoundFileFilter();
    requester->fileDialog()->setFilter( filter );

    // find the first "sound"-resource that contains files
    TQStringList soundDirs = TDEGlobal::dirs()->resourceDirs( "sound" );
    if ( !soundDirs.isEmpty() ) {
        KURL soundURL;
        TQDir dir;
        dir.setFilter( TQDir::Files | TQDir::Readable );
        TQStringList::ConstIterator it = soundDirs.begin();
        while ( it != soundDirs.end() ) {
            dir = *it;
            if ( dir.isReadable() && dir.count() > 2 ) {
                soundURL.setPath( *it );
                requester->fileDialog()->setURL( soundURL );
                break;
            }
            ++it;
        }
    }
}

void KPrTransEffectDia::slotSoundFileChanged( const TQString& text )
{
    soundFileName = text;

    buttonTestPlaySoundEffect->setEnabled( !text.isEmpty() );
    buttonTestStopSoundEffect->setEnabled( !text.isEmpty() );
}

void KPrTransEffectDia::playSound()
{
    delete soundPlayer;
    soundPlayer = new KPrSoundPlayer( requester->url() );
    soundPlayer->play();

    buttonTestPlaySoundEffect->setEnabled( false );
    buttonTestStopSoundEffect->setEnabled( true );
}

void KPrTransEffectDia::stopSound()
{
    if ( soundPlayer ) {
        soundPlayer->stop();
        delete soundPlayer;
        soundPlayer = 0;

        buttonTestPlaySoundEffect->setEnabled( true );
        buttonTestStopSoundEffect->setEnabled( false );
    }
}

void KPrTransEffectDia::slotOk()
{
    // TODO: only if changed. And pass flags for which settings changed
    emit apply( false );
}

void KPrTransEffectDia::slotUser1()
{
    // TODO: only if changed. And pass flags for which settings changed
    emit apply( true );
}

#include "KPrTransEffectDia.moc"
