#ifndef _KVI_CONFIG_H_INCLUDED_
#define _KVI_CONFIG_H_INCLUDED_

//=============================================================================
//
//   File : kvi_config.h (/usr/build/NEW_kvirc/kvirc/kvilib/kvi_config.h)
//   Last major modification : Thu Jan 14 1999 18:01:22 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
//
//=============================================================================

#include "kvi_string.h"
#include "kvi_settings.h"
#include "kvi_heapobject.h"
#include "kvi_pointerhashtable.h"
#include "kvi_valuelist.h"

#include <tqcolor.h>
#include <tqfont.h>
#include <tqrect.h>
#include <tqstringlist.h>

#define KVI_CONFIG_DEFAULT_GROUP "KVIrc"

class KviPixmap;
class KviMsgType;

typedef KviPointerHashTable<TQString,TQString> KviConfigGroup;
typedef KviPointerHashTableIterator<TQString,TQString> KviConfigGroupIterator;
typedef KviPointerHashTableIterator<TQString,KviConfigGroup> KviConfigIterator;

class KVILIB_API KviConfig : public KviHeapObject
{
public:
	enum FileMode { Read = 1 , Write = 2 , ReadWrite = 3 };
public:
	KviConfig(const TQString &filename,FileMode f/* = ReadWrite*/,bool bLocal8Bit = false);
	KviConfig(const char *filename,FileMode f/* = ReadWrite*/,bool bLocal8Bit = false);
	~KviConfig();
private:
	bool                         m_bLocal8Bit;
	KviPointerHashTable<TQString,KviConfigGroup>      * m_pDict;
	TQString                      m_szFileName;
	bool                         m_bDirty;
	TQString                      m_szStrBuffer;
	TQString                      m_szGroup;
	bool                         m_bPreserveEmptyGroups;
	bool                         m_bReadOnly;
private:
	bool load();
	bool save();
	KviConfigGroup * getCurrentGroup();
public:
	//
	// Useful when saving...
	// Normally this class does not save empty groups
	// and setGroup() is never a config modification.
	// If the group is not existing it will be effectively
	// created only at the first attempt to read from it or write to it.
	// With this flag set to true the KviConfig class will
	// write down also the empty groups , and calls to setGroup()
	// will create the groups if not existing yet (and set the config data
	// as modified).
	//
	void preserveEmptyGroups(bool bPreserve){ m_bPreserveEmptyGroups = bPreserve; };
	const TQString & fileName(){ return m_szFileName; };
	bool readOnly(){ return m_bReadOnly; };
	void setReadOnly(bool bReadOnly){ m_bReadOnly = bReadOnly; };
	bool dirty(){ return m_bDirty; };
	//
	// This sets the save path for the config file
	// In this way you can load a system-wide read-only config file
	// as default configuration, alter its settings and save it to the
	// user local configuration directory
	void setSavePath(const TQString & savePath){ m_szFileName = savePath; };
	KviPointerHashTable<TQString,KviConfigGroup> *dict(){ return m_pDict; };

	void clearDirtyFlag(){ m_bDirty = false; };
	void clear();
	void clearGroup(const TQString & szGroup);
	void clearKey(const TQString & szKey);
	unsigned int groupsCount(){ return m_pDict->count(); };
	bool sync(){ return save(); };
	bool hasKey(const TQString & szKey);
	bool hasGroup(const TQString & szGroup);
	void setGroup(const TQString & szGroup);
	//void getContentsString(KviStr &buffer);
	const TQString & group(){ return m_szGroup; };
	void writeEntry(const TQString & szKey,const TQString & szValue);
	void writeEntry(const TQString & szKey,const char * szValue)
		{ writeEntry(szKey,TQString::fromUtf8(szValue)); };
	TQString readEntry(const TQString & szKey,const TQString & szDefault = TQString());
	// HACK for KviOptions.. (FIXME)
	TQString readKviStrEntry(const TQString &szKey,const KviStr &szDefault)
		{ return readEntry(szKey,szDefault.ptr()); };
	//void writeEntry(const char *szKey,KviStr &szValue);
	//const char * readEntry(const char *szKey,KviStr &szDefault);
	void writeEntry(const TQString & szKey,const KviPixmap &pixmap);
	KviPixmap readPixmapEntry(const TQString & szKey,const KviPixmap &pixDef);
	void writeEntry(const TQString & szKey,const KviMsgType &msg);
	KviMsgType readMsgTypeEntry(const TQString & szKey,const KviMsgType &msgDef);
	void writeEntry(const TQString & szKey,const TQColor &clr);
	TQColor readColorEntry(const TQString & szKey,const TQColor &clr);
	void writeEntry(const TQString & szKey,TQFont &fnt);
	TQFont readFontEntry(const TQString & szKey,const TQFont &fnt);
	void writeEntry(const TQString & szKey,bool bTrue);
	bool readBoolEntry(const TQString & szKey,bool bTrue);
	TQRect readRectEntry(const TQString & szKey,const TQRect &rct);
	void writeEntry(const TQString & szKey,const TQRect &rct);
	TQStringList readStringListEntry(const TQString & szKey,const TQStringList &list);
	void writeEntry(const TQString & szKey,const TQStringList &list);
	KviValueList<int> readIntListEntry(const TQString & ,const KviValueList<int> &list);
	void writeEntry(const TQString & szKey,const KviValueList<int> &list);
	TQString readTQStringEntry(const TQString & szKey,const TQString &szDefault = TQString())
		{ return readEntry(szKey,szDefault); };
	//void writeEntry(const TQString & szKey,const TQString &szValue);
	//void writeEntry(const char *szKey,unsigned long lValue);
	//unsigned long readULongEntry(const char *szKey,unsigned long lDefault);
	//void writeEntry(const char *szKey,long lValue);
	//long readLongEntry(const char *szKey,long lDefault);
	void writeEntry(const TQString & szKey,int iValue);
	int readIntEntry(const TQString & szKey,int iDefault);
	void writeEntry(const TQString & szKey,unsigned short int usValue);
	unsigned short int readUShortEntry(const TQString & szKey,unsigned short int usDefault);
	void writeEntry(const TQString & szKey,unsigned int iValue);
	unsigned int readUIntEntry(const TQString & szKey,unsigned int iDefault);
	void writeEntry(const TQString & szKey,char iValue);
	char readCharEntry(const TQString & szKey,char iDefault);
	void writeEntry(const TQString & szKey,unsigned char iValue);
	unsigned char readUCharEntry(const TQString & szKey,unsigned char iDefault);

	static void getFontProperties(KviStr & buffer,TQFont *fnt);
	static void setFontProperties(KviStr & str,TQFont *fnt);

#ifdef COMPILE_ON_WINDOWS
	// On windows we need to override new and delete operators
	// to ensure that always the right new/delete pair is called for an object instance
	// This bug is present in all the classes exported by a module that
	// can be instantiated/destroyed from external modules.
	// (this is a well known bug described in Q122675 of MSDN)
	void       * operator new(size_t tSize);
	void         operator delete(void * p);
#endif
};

#endif //!_KVI_CONFIG_H_INCLUDED_
