/***************************************************************************
 *   Copyright (C) 2007 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "list_container.h"

//----------------------------------------------------------------------------
PopupContainer::PopupContainer(const TQString &title, TQWidget *parent, const char *name)
  : TDEPopupMenu(parent, name)
{
  if ( !title.isEmpty() ) insertTitle(title);
}

ListContainer *PopupContainer::appendBranch(const TQString &title)
{
  PopupContainer *branch = new PopupContainer(title, this);
  insertItem(title, branch);
  return branch;
}

ListContainer *PopupContainer::appendBranch(const TQPixmap &pixmap, const TQString &title)
{
  PopupContainer *branch = new PopupContainer(title, this);
  insertItem(pixmap, title, branch);
  return branch;
}

void PopupContainer::appendItem(const TQPixmap &icon, const TQString &label, uint id, ItemState state)
{
  insertItem(icon, label, id);
  switch (state) {
    case Normal:    break;
    case Checked:   setItemChecked(id, true); break;
    case UnChecked: setItemChecked(id, false); break;
    case Disabled:  setItemEnabled(id, false); break;
  }
}

//----------------------------------------------------------------------------
ListViewItemContainer::ListViewItemContainer(const TQString &title, TDEListView *parent)
  : TDEListViewItem(parent, title), _parent(0), _column(0)
{
  _ids = new TQMap<const TQListViewItem *, uint>;
}

ListViewItemContainer::ListViewItemContainer(const TQString &title, ListViewItemContainer *parent)
  : TDEListViewItem(parent, title), _parent(parent), _column(0)
{
  _ids = parent->_ids;
}

ListViewItemContainer::~ListViewItemContainer()
{
  if ( _parent==0 ) delete _ids;
}

ListContainer *ListViewItemContainer::appendBranch(const TQString &title)
{
  ListViewItemContainer *branch = new ListViewItemContainer(title, this);
  branch->setColumn(_column);
  branch->setSelectable(false);
  // append instead of prepend
  TQListViewItem *litem=firstChild();
  while ( litem && litem->nextSibling() ) litem = litem->nextSibling();
  if (litem) branch->moveItem(litem);
  return branch;
}

void ListViewItemContainer::appendItem(const TQPixmap &icon, const TQString &title, uint id, ItemState state)
{
  TQListViewItem *item = 0;
  if ( state==Normal || state==Disabled ) {
    item = new TDEListViewItem(this);
    item->setText(_column, title);
  } else {
    item = new TQCheckListItem(this, title, TQCheckListItem::CheckBox);
    static_cast<TQCheckListItem *>(item)->setState(state==Checked ? TQCheckListItem::On : TQCheckListItem::Off);
  }
  item->setPixmap(_column, icon);
  item->setSelectable(state==Normal);
  // append instead of prepend
  TQListViewItem *litem=firstChild();
  while ( litem && litem->nextSibling() ) litem = litem->nextSibling();
  if (litem) item->moveItem(litem);
  (*_ids)[item] = id;
}

int ListViewItemContainer::id(const TQListViewItem *item) const
{
  return (_ids->contains(item) ? int((*_ids)[item]) : -1);
}

#include "list_container.moc"
