/***************************************************************************
    begin                : Fri Sep 13 2002
    copyright            : (C) 2002 - 2004 by Scott Wheeler
    email                : wheeler@kde.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <tdeaction.h>
#include <kdebug.h>

#include <tqlayout.h>
#include <tqevent.h>

#include "playlistsplitter.h"
#include "searchwidget.h"
#include "playlistsearch.h"
#include "actioncollection.h"
#include "tageditor.h"
#include "collectionlist.h"
#include "playermanager.h"
#include "nowplaying.h"

using namespace ActionCollection;

////////////////////////////////////////////////////////////////////////////////
// public methods
////////////////////////////////////////////////////////////////////////////////

PlaylistSplitter::PlaylistSplitter(TQWidget *parent, const char *name) :
    TQSplitter(TQt::Horizontal, parent, name),
    m_newVisible(0),
    m_playlistBox(0),
    m_searchWidget(0),
    m_playlistStack(0),
    m_editor(0)
{
    setupActions();
    setupLayout();
    readConfig();

    m_editor->slotUpdateCollection();
    m_editor->setupObservers();
}

PlaylistSplitter::~PlaylistSplitter()
{
    saveConfig();

    // Since we want to ensure that the shutdown process for the PlaylistCollection
    // (a base class for PlaylistBox) has a chance to write the playlists to disk
    // before they are deleted we're explicitly deleting the PlaylistBox here.

    delete m_playlistBox;
}

bool PlaylistSplitter::eventFilter(TQObject *, TQEvent *event)
{
    if(event->type() == FocusUpEvent::id) {
        m_searchWidget->setFocus();
        return true;
    }
    return false;
}

////////////////////////////////////////////////////////////////////////////////
// public slots
////////////////////////////////////////////////////////////////////////////////

void PlaylistSplitter::setFocus()
{
    m_searchWidget->setFocus();
}

void PlaylistSplitter::slotFocusCurrentPlaylist()
{
    Playlist *playlist = m_playlistBox->visiblePlaylist();

    if(playlist) {
        playlist->setFocus();
        playlist->TDEListView::selectAll(false);

        // Select the top visible (and matching) item.

        PlaylistItem *item = static_cast<PlaylistItem *>(playlist->itemAt(TQPoint(0, 0)));

        if(!item)
            return;

        // A little bit of a hack to make TQListView repaint things properly.  Switch
        // to single selection mode, set the selection and then switch back.

        playlist->setSelectionMode(TQListView::Single);

        playlist->markItemSelected(item, true);
        playlist->setCurrentItem(item);

        playlist->setSelectionMode(TQListView::Extended);
    }
}

////////////////////////////////////////////////////////////////////////////////
// private members
////////////////////////////////////////////////////////////////////////////////

Playlist *PlaylistSplitter::visiblePlaylist() const
{
    return m_newVisible ? m_newVisible : m_playlistBox->visiblePlaylist();
}

void PlaylistSplitter::setupActions()
{
    TDEToggleAction *showSearch =
        new TDEToggleAction(i18n("Show &Search Bar"), "filefind", 0, ActionCollection::actions(), "showSearch");
    showSearch->setCheckedState(i18n("Hide &Search Bar"));

    new TDEAction(i18n("Edit Track Search"), "edit_clear", "F6", this, TQ_SLOT(setFocus()), ActionCollection::actions(), "editTrackSearch");
}

void PlaylistSplitter::setupLayout()
{
    setOpaqueResize(false);

    // Create a splitter to go between the playlists and the editor.

    TQSplitter *editorSplitter = new TQSplitter(TQt::Vertical, this, "editorSplitter");

    // Create the playlist and the editor.

    TQWidget *top = new TQWidget(editorSplitter);
    TQVBoxLayout *topLayout = new TQVBoxLayout(top);

    m_playlistStack = new TQWidgetStack(top, "playlistStack");
    m_playlistStack->installEventFilter(this);

    connect(m_playlistStack, TQ_SIGNAL(aboutToShow(TQWidget *)), this, TQ_SLOT(slotPlaylistChanged(TQWidget *)));

    m_editor = new TagEditor(editorSplitter, "tagEditor");

    // Make the editor as small as possible (or at least as small as recommended)

    editorSplitter->setResizeMode(m_editor, TQSplitter::FollowSizeHint);

    // Create the PlaylistBox

    m_playlistBox = new PlaylistBox(this, m_playlistStack, "playlistBox");

    connect(m_playlistBox->object(), TQ_SIGNAL(signalSelectedItemsChanged()),
            this, TQ_SLOT(slotPlaylistSelectionChanged()));
    connect(m_playlistBox, TQ_SIGNAL(signalPlaylistDestroyed(Playlist *)),
            m_editor, TQ_SLOT(slotPlaylistDestroyed(Playlist *)));

    moveToFirst(m_playlistBox);

    connect(CollectionList::instance(), TQ_SIGNAL(signalCollectionChanged()),
            m_editor, TQ_SLOT(slotUpdateCollection()));

    NowPlaying *nowPlaying = new NowPlaying(top, m_playlistBox);

    // Create the search widget -- this must be done after the CollectionList is created.

    m_searchWidget = new SearchWidget(top, "searchWidget");
    connect(m_searchWidget, TQ_SIGNAL(signalQueryChanged()),
            this, TQ_SLOT(slotShowSearchResults()));
    connect(m_searchWidget, TQ_SIGNAL(signalDownPressed()),
            this, TQ_SLOT(slotFocusCurrentPlaylist()));
    connect(m_searchWidget, TQ_SIGNAL(signalAdvancedSearchClicked()),
            m_playlistBox->object(), TQ_SLOT(slotCreateSearchPlaylist()));
    connect(m_searchWidget, TQ_SIGNAL(signalShown(bool)),
            m_playlistBox->object(), TQ_SLOT(slotSetSearchEnabled(bool)));
    connect(action<TDEToggleAction>("showSearch"), TQ_SIGNAL(toggled(bool)),
            m_searchWidget, TQ_SLOT(setEnabled(bool)));

    topLayout->addWidget(nowPlaying);
    topLayout->addWidget(m_searchWidget);
    topLayout->addWidget(m_playlistStack);

    // Show the collection on startup.
    m_playlistBox->setSelected(0, true);
}

void PlaylistSplitter::readConfig()
{
    TDEConfigGroup config(TDEGlobal::config(), "Splitter");

    TQValueList<int> splitterSizes = config.readIntListEntry("PlaylistSplitterSizes");
    if(splitterSizes.isEmpty()) {
        splitterSizes.append(100);
        splitterSizes.append(640);
    }
    setSizes(splitterSizes);

    bool showSearch = config.readBoolEntry("ShowSearch", true);
    action<TDEToggleAction>("showSearch")->setChecked(showSearch);
    m_searchWidget->setShown(showSearch);
}

void PlaylistSplitter::saveConfig()
{
    TDEConfigGroup config(TDEGlobal::config(), "Splitter");
    config.writeEntry("PlaylistSplitterSizes", sizes());
    config.writeEntry("ShowSearch", action<TDEToggleAction>("showSearch")->isChecked());
}

void PlaylistSplitter::slotShowSearchResults()
{
    PlaylistList playlists;
    playlists.append(visiblePlaylist());
    PlaylistSearch search = m_searchWidget->search(playlists);
    visiblePlaylist()->setSearch(search);
}

void PlaylistSplitter::slotPlaylistSelectionChanged()
{
    m_editor->slotSetItems(visiblePlaylist()->selectedItems());
}

void PlaylistSplitter::slotPlaylistChanged(TQWidget *w)
{
    Playlist *p = dynamic_cast<Playlist *>(w);

    if(!p)
        return;

    m_newVisible = p;
    m_searchWidget->setSearch(p->search());
    m_newVisible = 0;
}

#include "playlistsplitter.moc"
