/*
    test_cryptoconfig.cpp

    This file is part of libkleopatra's test suite.
    Copyright (c) 2004 KlarÃ¤lvdalens Datakonsult AB

    Libkleopatra is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License,
    version 2, as published by the Free Software Foundation.

    Libkleopatra is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

    In addition, as a special exception, the copyright holders give
    permission to link the code of this program with any edition of
    the TQt library by Trolltech AS, Norway (or with modified versions
    of TQt that use the same license as TQt), and distribute linked
    combinations including the two.  You must obey the GNU General
    Public License in all respects for all of the code used other than
    TQt.  If you modify this file, you may extend this exception to
    your version of the file, but you are not obligated to do so.  If
    you do not wish to do so, delete this exception statement from
    your version.
*/

#include <backends/qgpgme/qgpgmecryptoconfig.h>

#include <tdeapplication.h>
#include <tdeaboutdata.h>
#include <tdecmdlineargs.h>
#include <iostream>
using namespace std;

#include <stdlib.h>
#include <assert.h>

int main( int argc, char** argv ) {

  TDEAboutData aboutData( "test_cryptoconfig", "CryptoConfig Test", "0.1" );
  TDECmdLineArgs::init( argc, argv, &aboutData );
  TDEApplication app( false, false );

  Kleo::CryptoConfig * config = new QGpgMECryptoConfig();

  // Dynamic querying of the options
  cout << "Components:" << endl;
  TQStringList components = config->componentList();

  for( TQStringList::Iterator compit = components.begin(); compit != components.end(); ++compit ) {
    cout << "Component " << (*compit).local8Bit() << ":" << endl;
    const Kleo::CryptoConfigComponent* comp = config->component( *compit );
    assert( comp );
    TQStringList groups = comp->groupList();
    for( TQStringList::Iterator groupit = groups.begin(); groupit != groups.end(); ++groupit ) {
      const Kleo::CryptoConfigGroup* group = comp->group( *groupit );
      assert( group );
      cout << " Group " << (*groupit).local8Bit() << ": descr=\"" << group->description().local8Bit() << "\""
           << " level=" << group->level() << endl;
      TQStringList entries = group->entryList();
      for( TQStringList::Iterator entryit = entries.begin(); entryit != entries.end(); ++entryit ) {
        const Kleo::CryptoConfigEntry* entry = group->entry( *entryit );
        assert( entry );
        cout << "  Entry " << (*entryit).local8Bit() << ":"
             << " descr=\"" << entry->description().local8Bit() << "\""
             << " " << ( entry->isSet() ? "is set" : "is not set" );
        if ( !entry->isList() )
          switch( entry->argType() ) {
          case Kleo::CryptoConfigEntry::ArgType_None:
            break;
          case Kleo::CryptoConfigEntry::ArgType_Int:
            cout << " int value=" << entry->intValue();
            break;
          case Kleo::CryptoConfigEntry::ArgType_UInt:
            cout << " uint value=" << entry->uintValue();
            break;
          case Kleo::CryptoConfigEntry::ArgType_LDAPURL:
          case Kleo::CryptoConfigEntry::ArgType_URL:
            cout << " URL value=" << entry->urlValue().prettyURL().local8Bit();
            // fallthrough
          case Kleo::CryptoConfigEntry::ArgType_Path:
            // fallthrough
          case Kleo::CryptoConfigEntry::ArgType_DirPath:
            // fallthrough
          case Kleo::CryptoConfigEntry::ArgType_String:

            cout << " string value=" << entry->stringValue().local8Bit();
            break;
          }
        else // lists
        {
          switch( entry->argType() ) {
          case Kleo::CryptoConfigEntry::ArgType_None: {
            cout << " set " << entry->numberOfTimesSet() << " times";
            break;
          }
          case Kleo::CryptoConfigEntry::ArgType_Int: {
            assert( entry->isOptional() ); // empty lists must be allowed (see issue121)
            TQValueList<int> lst = entry->intValueList();
            TQString str;
            for( TQValueList<int>::Iterator it = lst.begin(); it != lst.end(); ++it ) {
              str += TQString::number( *it );
            }
            cout << " int values=" << str.local8Bit();
            break;
          }
          case Kleo::CryptoConfigEntry::ArgType_UInt: {
            assert( entry->isOptional() ); // empty lists must be allowed (see issue121)
            TQValueList<uint> lst = entry->uintValueList();
            TQString str;
            for( TQValueList<uint>::Iterator it = lst.begin(); it != lst.end(); ++it ) {
              str += TQString::number( *it );
            }
            cout << " uint values=" << str.local8Bit();
            break;
          }
          case Kleo::CryptoConfigEntry::ArgType_LDAPURL:
          case Kleo::CryptoConfigEntry::ArgType_URL: {
              assert( entry->isOptional() ); // empty lists must be allowed (see issue121)
              KURL::List urls = entry->urlValueList();
              cout << " url values=" << urls.toStringList().join(" ").local8Bit() << "\n    ";
          }
            // fallthrough
          case Kleo::CryptoConfigEntry::ArgType_Path:
            // fallthrough
          case Kleo::CryptoConfigEntry::ArgType_DirPath:
            // fallthrough
          case Kleo::CryptoConfigEntry::ArgType_String: {
            assert( entry->isOptional() ); // empty lists must be allowed (see issue121)
            TQStringList lst = entry->stringValueList();
            cout << " string values=" << lst.join(" ").local8Bit();
            break;
          }
          }
        }
        cout << endl;
      }
      // ...
    }
  }

  {
    // Static querying of a single boolean option
    static const char* s_groupName = "Monitor";
    static const char* s_entryName = "quiet";
    Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
    if ( entry ) {
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_None );
      bool val = entry->boolValue();
      cout << "quiet option initially: " << ( val ? "is set" : "is not set" ) << endl;

      entry->setBoolValue( !val );
      assert( entry->isDirty() );
      config->sync( true );

      // Clear cached values!
      config->clear();

      // Check new value
      Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
      assert( entry );
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_None );
      cout << "quiet option now: " << ( val ? "is set" : "is not set" ) << endl;
      assert( entry->boolValue() == !val );

      // Set to default
      entry->resetToDefault();
      assert( entry->boolValue() == false ); // that's the default
      assert( entry->isDirty() );
      assert( !entry->isSet() );
      config->sync( true );
      config->clear();

      // Check value
      entry = config->entry( "dirmngr", s_groupName, s_entryName );
      assert( !entry->isDirty() );
      assert( !entry->isSet() );
      cout << "quiet option reset to default: " << ( entry->boolValue() ? "is set" : "is not set" ) << endl;
      assert( entry->boolValue() == false );

      // Reset old value
      entry->setBoolValue( val );
      assert( entry->isDirty() );
      config->sync( true );

      cout << "quiet option reset to initial: " << ( val ? "is set" : "is not set" ) << endl;
    }
    else
      cout << "Entry 'dirmngr/" << s_groupName << "/" << s_entryName << "' not found" << endl;
  }

  {
    // Static querying and setting of a single int option
    static const char* s_groupName = "LDAP";
    static const char* s_entryName = "ldaptimeout";
    Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
    if ( entry ) {
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_UInt );
      uint val = entry->uintValue();
      cout << "LDAP timeout initially: " << val << " seconds." << endl;

      // Test setting the option directly, then querying again
      //system( "echo 'ldaptimeout:0:101' | gpgconf --change-options dirmngr" );
      // Now let's do it with the C++ API instead
      entry->setUIntValue( 101 );
      assert( entry->isDirty() );
      config->sync( true );

      // Clear cached values!
      config->clear();

      // Check new value
      Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
      assert( entry );
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_UInt );
      cout << "LDAP timeout now: " << entry->uintValue() << " seconds." << endl;
      assert( entry->uintValue() == 101 );

      // Set to default
      entry->resetToDefault();
      assert( entry->uintValue() == 100 );
      assert( entry->isDirty() );
      assert( !entry->isSet() );
      config->sync( true );
      config->clear();

      // Check value
      entry = config->entry( "dirmngr", s_groupName, s_entryName );
      assert( !entry->isDirty() );
      assert( !entry->isSet() );
      cout << "LDAP timeout reset to default, " << entry->uintValue() << " seconds." << endl;
      assert( entry->uintValue() == 100 );

      // Reset old value
      entry->setUIntValue( val );
      assert( entry->isDirty() );
      config->sync( true );

      cout << "LDAP timeout reset to initial " << val << " seconds." << endl;
    }
    else
      cout << "Entry 'dirmngr/" << s_groupName << "/" << s_entryName << "' not found" << endl;
  }

  {
    // Static querying and setting of a single string option
    static const char* s_groupName = "Debug";
    static const char* s_entryName = "log-file";
    Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
    if ( entry ) {
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_Path );
      TQString val = entry->stringValue();
      cout << "Log-file initially: " << val.local8Bit() << endl;

      // Test setting the option, sync'ing, then querying again
      entry->setStringValue( TQString::fromUtf8( "/tmp/test:%e5Ã¤" ) );
      assert( entry->isDirty() );
      config->sync( true );

      // Let's see how it prints it
      system( "gpgconf --list-options dirmngr | grep log-file" );

      // Clear cached values!
      config->clear();

      // Check new value
      Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
      assert( entry );
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_Path );
      cout << "Log-file now: " << entry->stringValue().local8Bit() << endl;
      assert( entry->stringValue() == TQString::fromUtf8( "/tmp/test:%e5Ã¤" ) ); // (or even with %e5 decoded)

      // Reset old value
#if 0
      TQString arg( val );
      if ( !arg.isEmpty() )
        arg.prepend( '"' );
      TQCString sys;
      sys.sprintf( "echo 'log-file:%s' | gpgconf --change-options dirmngr", arg.local8Bit().data() );
      system( sys.data() );
#endif
      entry->setStringValue( val );
      assert( entry->isDirty() );
      config->sync( true );

      cout << "Log-file reset to initial " << val.local8Bit() << endl;
    }
    else
      cout << "Entry 'dirmngr/" << s_groupName << "/" << s_entryName << "' not found" << endl;
  }

  {
    // Static querying and setting of the LDAP URL list option
    static const char* s_groupName = "LDAP";
    static const char* s_entryName = "LDAP Server";
    Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
    if ( entry ) {
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_LDAPURL );
      assert( entry->isList() );
      KURL::List val = entry->urlValueList();
      cout << "URL list initially: " << val.toStringList().join(", ").local8Bit() << endl;

      // Test setting the option, sync'ing, then querying again
      KURL::List lst;
      // We use non-empty paths to workaround a bug in KURL (tdelibs-3.2)
      lst << KURL( "ldap://a:389/?b" );
      // Test with query containing a litteral ':' (KURL supports this)
      // and a ' ' (KURL will escape it, see issue119)
      lst << KURL( "ldap://foo:389/?a:b c" );
      lst << KURL( "ldap://server:389/?a%3db,c=DE" ); // the query contains a litteral ','
      //cout << " trying to set: " << lst.toStringList().join(", ").local8Bit() << endl;
      assert( lst[0].query() == "?b" );
      assert( lst[1].query() == "?a:b%20c" ); // see, the space got escaped
      entry->setURLValueList( lst );
      assert( entry->isDirty() );
      config->sync( true );

      // Let's see how it prints it
      system( "gpgconf --list-options dirmngr | grep 'LDAP Server'" );

      // Clear cached values!
      config->clear();

      // Check new value
      Kleo::CryptoConfigEntry* entry = config->entry( "dirmngr", s_groupName, s_entryName );
      assert( entry );
      assert( entry->argType() == Kleo::CryptoConfigEntry::ArgType_LDAPURL );
      assert( entry->isList() );
      // Get raw a:b:c:d:e form
      TQStringList asStringList = entry->stringValueList();
      assert( asStringList.count() == 3 );
      cout << "asStringList[0]=" << asStringList[0].local8Bit() << endl;
      cout << "asStringList[1]=" << asStringList[1].local8Bit() << endl;
      cout << "asStringList[2]=" << asStringList[2].local8Bit() << endl;
      assert( asStringList[0] == "a:389:::b" );
      assert( asStringList[1] == "foo:389:::a%3ab c" ); // the space must be decoded (issue119)
      assert( asStringList[2] == "server:389:::a=b,c=DE" ); // all decoded
      // Get KURL form
      KURL::List newlst = entry->urlValueList();
      cout << "URL list now: " << newlst.toStringList().join(", ").local8Bit() << endl;
      assert( newlst.count() == 3 );
      //cout << "newlst[0]=" << newlst[0].url().local8Bit() << endl;
      //cout << "lst[0]=" << lst[0].url().local8Bit() << endl;
      assert( newlst[0] == lst[0] );
      assert( newlst[1] == lst[1] );
      assert( newlst[2].url() == "ldap://server:389/?a=b,c=DE" ); // != lst[2] due to the encoded =

      // Reset old value
      entry->setURLValueList( val );
      assert( entry->isDirty() );
      config->sync( true );

      cout << "URL list reset to initial: " << val.toStringList().join(", ").local8Bit() << endl;
    }
    else
      cout << "Entry 'dirmngr/" << s_groupName << "/" << s_entryName << "' not found" << endl;
  }

  cout << "Done." << endl;
}
