/*
    This file is part of libkcal.

    Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "calendarlocal.h"
#include "icalformat.h"
#include "qtopiaformat.h"

#include <tdeaboutdata.h>
#include <tdeapplication.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <tdecmdlineargs.h>
#include <tdeglobal.h>
#include <tdeconfig.h>
#include <tdestandarddirs.h>

#include <iostream>

using namespace KCal;

static const TDECmdLineOptions options[] =
{
  {"q", 0, 0 },
  {"qtopia2icalendar", I18N_NOOP("Convert TQtopia calendar file to iCalendar"), 0 },
  {"i", 0, 0 },
  {"icalendar2qtopia", I18N_NOOP("Convert iCalendar to iCalendar"), 0 },
  {"o", 0, 0},
  {"output <file>", I18N_NOOP("Output file"), 0 },
  {"+input", I18N_NOOP("Input file"), 0 },
  TDECmdLineLastOption
};

int main(int argc,char **argv)
{
  TDEAboutData aboutData("convertqtopia",I18N_NOOP("TQtopia calendar file converter"),"0.1");
  aboutData.addAuthor("Cornelius Schumacher", 0, "schumacher@kde.org");

  TDECmdLineArgs::init(argc,argv,&aboutData);
  TDECmdLineArgs::addCmdLineOptions( options );

  TDEApplication app;

  TDECmdLineArgs *args = TDECmdLineArgs::parsedArgs();

  bool sourceTQtopia = false;
  bool sourceIcalendar = false;

  if ( args->isSet( "qtopia2icalendar" ) ) {
    sourceTQtopia = true;
  }

  if ( args->isSet( "icalendar2qtopia" ) ) {
    sourceIcalendar = true;
  }

  if ( sourceTQtopia && sourceIcalendar ) {
    TDECmdLineArgs::usage(
        i18n("Please specify only one of the conversion options.") );
  }
  if ( !sourceTQtopia && !sourceIcalendar ) {
    TDECmdLineArgs::usage(
        i18n("You have to specify one conversion option.") );
  }

  if ( args->count() != 1 ) {
    TDECmdLineArgs::usage( i18n("Error: No input file.") );
  }

  TQString inputFile = args->arg( 0 );

  TQString outputFile;
  if ( args->isSet("output") ) outputFile = args->getOption( "output" );

  kdDebug(5800) << "Input File: '" << inputFile << "'" << endl;
  kdDebug(5800) << "Output File: '" << outputFile << "'" << endl;

  if ( sourceTQtopia ) {
    CalendarLocal cal;
    
    TQtopiaFormat qtopiaFormat;
    qtopiaFormat.load( &cal, inputFile );

    ICalFormat icalendarFormat;
    if ( outputFile.isEmpty() ) {
      TQString out = icalendarFormat.toString( &cal );
      std::cout << out.local8Bit() << std::endl;
    } else {
      bool success = icalendarFormat.save( &cal, outputFile );
      if ( !success ) {
        std::cerr << i18n( "Error saving to '%1'." ).arg( outputFile ).local8Bit()
                  << std::endl;
        return 1;
      }
    }
  }
  
  if ( sourceIcalendar ) {
    std::cerr << "Not implemented yet." << std::endl;
    return 1;
  }
}
