/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>
                2002-2004 by Stanislav Visnovsky
                            <visnovsky@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  In addition, as a special exception, the copyright holders give
  permission to link the code of this program with any edition of
  the TQt library by Trolltech AS, Norway (or with modified versions
  of TQt that use the same license as TQt), and distribute linked
  combinations including the two.  You must obey the GNU General
  Public License in all respects for all of the code used other than
  TQt. If you modify this file, you may extend this exception to
  your version of the file, but you are not obligated to do so.  If
  you do not wish to do so, delete this exception statement from
  your version.

**************************************************************************** */

#include "kbabel.h"

#include "kbabelsettings.h"
#include "kbprojectsettings.h"
#include "kbabelpref.h"
#include "projectpref.h"
#include "kbabelsplash.h"
#include "regexpextractor.h"
#include "toolaction.h"
#include "commentview.h"
#include "contextview.h"
#include "charselectview.h"
#include "taglistview.h"
#include "sourceview.h"

#include <tqdragobject.h>
#include <tqlineedit.h>
#include <tqpopupmenu.h>
#include <tqhbox.h>
#include <tqwhatsthis.h>
#include <tqsize.h>
#include <tqtextcodec.h>
#include <tqtooltip.h>
#include <tqtimer.h>

#include <dcopclient.h>
#include <kdatatool.h>
#include <tdepopupmenu.h>
#include <kstatusbar.h>
#include <tdestdaccel.h>
#include <kedittoolbar.h>
#include <tdeglobal.h>
#include <kled.h>
#include <tdelocale.h>
#include <kiconloader.h>
#include <tdetoolbar.h>
#include <tdefiledialog.h>
#include <tdeconfig.h>
#include <kurl.h>
#include <kdialogbase.h>
#include <kprogress.h>
#include <kpushbutton.h>
#include <tdemessagebox.h>
#include <twin.h>
#include <tdeaction.h>
#include <kstdaction.h>
#include <tdespelldlg.h>
#include <ksqueezedtextlabel.h>
#include <kurldrag.h>

#include "resources.h"
#include "kbcatalog.h"
#include "dictionarymenu.h"
#include "kbabeldictbox.h"
#include "kbmailer.h"
#include "kbbookmarkhandler.h"
#include "kbprojectmanager.h"
#include "projectpref.h"
#include "projectwizard.h"

#include "version.h"

#define ID_STATUS_TOTAL 1
#define ID_STATUS_CURRENT 2
#define ID_STATUS_FUZZY 3
#define ID_STATUS_UNTRANS 4
#define ID_STATUS_EDITMODE 5
#define ID_STATUS_READONLY 6
#define ID_STATUS_CURSOR 7

// maximum number of recent files
#define MAX_RECENT 10

using namespace KBabel;

TQPtrList<KBabelPreferences> KBabelMW::prefDialogs;

class MyKProgress: public KProgress
{
public:
    MyKProgress( TQWidget *parent, const char *name ) : KProgress( parent, name )
    {
        setSizePolicy( TQSizePolicy::Minimum, TQSizePolicy::Preferred );
    }
    TQSize sizeHint() const { return TQSize( 1, 1);}
};

KBabelMW::KBabelMW(TQString projectFile)
        : KDockMainWindow (), m_charselectorview(0)
{
    if ( projectFile.isEmpty() )
    	projectFile = KBabel::ProjectManager::defaultProjectName();
    _project = ProjectManager::open(projectFile);
    
    if ( _project == NULL ) // FIXME should not happen anymore
    {
	KMessageBox::error( this, i18n("Cannot open project file\n%1").arg(projectFile)
            , i18n("Project File Error"));
	_project = ProjectManager::open(KBabel::ProjectManager::defaultProjectName());
    }
    
    KBCatalog* catalog=new KBCatalog(projectFile);
    init(catalog);
}

KBabelMW::KBabelMW(KBCatalog* catalog, TQString projectFile)
        : KDockMainWindow (), m_charselectorview(0)
{
    if ( projectFile.isEmpty() )
    	projectFile = KBabel::ProjectManager::defaultProjectName();
    _project = ProjectManager::open(projectFile);

    if ( _project == NULL )
    {
	KMessageBox::error( this, i18n("Cannot open project file\n%1").arg(projectFile)
            , i18n("Project File Error"));
	_project = ProjectManager::open(KBabel::ProjectManager::defaultProjectName());
    }
    
    init(catalog);
}

void KBabelMW::init(KBCatalog* catalog)
{
    _config = TDESharedConfig::openConfig( "kbabelrc" );

    _toolsShortcuts.clear();

    _fuzzyLed=0;
    _untransLed=0;
    _errorLed=0;

    _projectDialog=0;

    _prefDialog=0;
    prefDialogs.setAutoDelete(true);

    _statusbarTimer = new TQTimer(this, "statusbartimer");
    connect(_statusbarTimer,TQ_SIGNAL(timeout()),this
        ,TQ_SLOT(clearStatusbarMsg()));

    // FIXME:
    Q_ASSERT(_project);

    m_view=new KBabelView(catalog,this, _project);

    setXMLFile ("kbabelui.rc");

    createGUI (0);

    // accept dnd
    setAcceptDrops(true);


    // setup our menubars and toolbars
    setupStatusBar();
    setupActions();
    stateChanged( "fileopened" , StateReverse );
    stateChanged( "readonly", StateNoReverse );

        TQPopupMenu* popup;
        popup = (TQPopupMenu*)(factory()->container("rmb_edit", this));
        if(popup)
        {
                m_view->setRMBEditMenu(popup);
        }
        popup = (TQPopupMenu*)(factory()->container("rmb_search", this));
        if(popup)
        {
                m_view->setRMBSearchMenu(popup);
        }


    connect(catalog,TQ_SIGNAL(signalUndoAvailable(bool)),this
            ,TQ_SLOT(enableUndo(bool)));
    connect(catalog,TQ_SIGNAL(signalRedoAvailable(bool)),this
            ,TQ_SLOT(enableRedo(bool)));
    connect(catalog,TQ_SIGNAL(signalNumberOfFuzziesChanged(uint)),this
            ,TQ_SLOT(setNumberOfFuzzies(uint)));
    connect(catalog,TQ_SIGNAL(signalNumberOfUntranslatedChanged(uint)),this
            ,TQ_SLOT(setNumberOfUntranslated(uint)));
    connect(catalog,TQ_SIGNAL(signalTotalNumberChanged(uint)),this
            ,TQ_SLOT(setNumberOfTotal(uint)));
    connect(catalog,TQ_SIGNAL(signalProgress(int)),_progressBar,TQ_SLOT(setProgress(int)));
    connect(catalog,TQ_SIGNAL(signalClearProgressBar()),this,TQ_SLOT(clearProgressBar()));
    connect(catalog,TQ_SIGNAL(signalResetProgressBar(TQString,int))
           ,this,TQ_SLOT(prepareProgressBar(TQString,int)));
    connect(catalog,TQ_SIGNAL(signalFileOpened(bool)),this,TQ_SLOT(enableDefaults(bool)));
    connect(catalog,TQ_SIGNAL(signalFileOpened(bool)),m_view,TQ_SLOT(newFileOpened(bool)));
    connect(catalog,TQ_SIGNAL(signalModified(bool)),this,TQ_SLOT(showModified(bool)));

    // allow the view to change the statusbar and caption
    connect(m_view, TQ_SIGNAL(signalChangeStatusbar(const TQString&)),
            this,   TQ_SLOT(changeStatusbar(const TQString&)));
    connect(m_view, TQ_SIGNAL(signalChangeCaption(const TQString&)),
            this,   TQ_SLOT(changeCaption(const TQString&)));
    connect(m_view,TQ_SIGNAL(signalFirstDisplayed(bool, bool)),this
           ,TQ_SLOT(firstEntryDisplayed(bool, bool)));
    connect(m_view,TQ_SIGNAL(signalLastDisplayed(bool, bool)),this
           ,TQ_SLOT(lastEntryDisplayed(bool, bool)));
    connect(m_view,TQ_SIGNAL(signalFuzzyDisplayed(bool)),this
           ,TQ_SLOT(fuzzyDisplayed(bool)));
    connect(m_view,TQ_SIGNAL(signalUntranslatedDisplayed(bool)),this
           ,TQ_SLOT(untranslatedDisplayed(bool)));
    connect(m_view,TQ_SIGNAL(signalFaultyDisplayed(bool)),this
           ,TQ_SLOT(faultyDisplayed(bool)));
    connect(m_view,TQ_SIGNAL(signalDisplayed(const KBabel::DocPosition&)),this
           ,TQ_SLOT(displayedEntryChanged(const KBabel::DocPosition&)));
    connect(m_view,TQ_SIGNAL(signalFuzzyAfterwards(bool)),this
            ,TQ_SLOT(hasFuzzyAfterwards(bool)));
    connect(m_view,TQ_SIGNAL(signalFuzzyInFront(bool)),this
            ,TQ_SLOT(hasFuzzyInFront(bool)));
    connect(m_view,TQ_SIGNAL(signalUntranslatedAfterwards(bool)),this
            ,TQ_SLOT(hasUntranslatedAfterwards(bool)));
    connect(m_view,TQ_SIGNAL(signalUntranslatedInFront(bool)),this
            ,TQ_SLOT(hasUntranslatedInFront(bool)));
    connect(m_view,TQ_SIGNAL(signalErrorAfterwards(bool)),this
            ,TQ_SLOT(hasErrorAfterwards(bool)));
    connect(m_view,TQ_SIGNAL(signalErrorInFront(bool)),this
            ,TQ_SLOT(hasErrorInFront(bool)));
    connect(m_view,TQ_SIGNAL(signalBackHistory(bool)),this
            ,TQ_SLOT(enableBackHistory(bool)));
    connect(m_view,TQ_SIGNAL(signalForwardHistory(bool)),this
            ,TQ_SLOT(enableForwardHistory(bool)));


    connect(m_view,TQ_SIGNAL(ledColorChanged(const TQColor&)),this
                  ,TQ_SLOT(setLedColor(const TQColor&)));


    connect(m_view,TQ_SIGNAL(signalSearchActive(bool)),this,TQ_SLOT(enableStop(bool)));

    connect(m_view,TQ_SIGNAL(signalProgress(int)),_progressBar,TQ_SLOT(setProgress(int)));
    connect(m_view,TQ_SIGNAL(signalClearProgressBar()),this,TQ_SLOT(clearProgressBar()));
    connect(m_view,TQ_SIGNAL(signalResetProgressBar(TQString,int))
           ,this,TQ_SLOT(prepareProgressBar(TQString,int)));

    connect(m_view,TQ_SIGNAL(signalDictionariesChanged())
           , this, TQ_SLOT(buildDictMenus()));
    connect(m_view,TQ_SIGNAL(signalCursorPosChanged(int,int)), this
            , TQ_SLOT(updateCursorPosition(int,int)));

    if(!catalog->currentURL().isEmpty())
    {
       enableDefaults(catalog->isReadOnly());
       setNumberOfFuzzies(catalog->numberOfFuzzies());
       setNumberOfUntranslated(catalog->numberOfUntranslated());
       setNumberOfTotal(catalog->numberOfEntries());

       enableUndo(catalog->isUndoAvailable());
       enableUndo(catalog->isRedoAvailable());

       m_view->emitEntryState();

       changeCaption(catalog->currentURL().prettyURL() );
    }

    mailer = new KBabelMailer( this, _project );

    bmHandler = new KBabelBookmarkHandler((TQPopupMenu*)factory()->container("bookmark", this));
    // the earlier created TDEAction for "clear_bookmarks" is now reconnected
    TDEAction* action = actionCollection()->action("clear_bookmarks");
    if (action) {
      action->disconnect(TQ_SIGNAL(activated()));
      connect(action, TQ_SIGNAL(activated()),
              bmHandler, TQ_SLOT(slotClearBookmarks()));
    }
    connect(bmHandler, TQ_SIGNAL(signalBookmarkSelected(int)),
            this, TQ_SLOT(slotOpenBookmark(int)));
    connect(m_view, TQ_SIGNAL(signalNewFileOpened(KURL)),
            bmHandler, TQ_SLOT(slotClearBookmarks()));

    _config = TDESharedConfig::openConfig( "kbabelrc" );
    restoreSettings();

    _config->setGroup("KBabel");

    if(!_config->hasKey("Version"))
    {
      TQString encodingStr;
      switch(catalog->saveSettings().encoding)
      {
         case KBabel::ProjectSettingsBase::UTF8:
            encodingStr=TQTextCodec::codecForName("UTF-8")->name();
            break;
         case KBabel::ProjectSettingsBase::UTF16:
            encodingStr=TQTextCodec::codecForName("UTF-16")->name();
            break;
         default:
            encodingStr=TQTextCodec::codecForLocale()->name();
      }

      if( KBabelSplash::instance ) KBabelSplash::instance->close(); //close splash screen window, if there is one

      KMessageBox::information(0,i18n("You have not run KBabel before. "
                   "To allow KBabel to work correctly you must enter some "
                   "information in the preferences dialog first.\n"
                   "The minimum requirement is to fill out the Identity page.\n"
                   "Also check the encoding on the Save page, which is currently "
                   "set to %1. You may want to change this setting "
                   "according to the settings of your language team.").arg(encodingStr));

      TQTimer::singleShot(1,this,TQ_SLOT(projectConfigure()));
    }

    _config->writeEntry("Version",VERSION);
    _config->sync();

}

KBabelMW::~KBabelMW()
{
   if(_prefDialog)
   {
      prefDialogs.remove(_prefDialog);
   }
   if(_projectDialog)
   {
      delete _projectDialog;
   }
   delete mailer;
   delete bmHandler;
}


void KBabelMW::setSettings(SaveSettings saveOpts,IdentitySettings idOpts)
{
   m_view->updateSettings();
   m_view->catalog()->setSettings(saveOpts);
   m_view->catalog()->setSettings(idOpts);

   if(_fuzzyLed)
   {
      _fuzzyLed->setColor(KBabelSettings::ledColor());
   }
   if(_untransLed)
   {
      _untransLed->setColor(KBabelSettings::ledColor());
   }
   if(_errorLed)
   {
      _errorLed->setColor(KBabelSettings::ledColor());
   }

}

void KBabelMW::updateSettings()
{
   m_view->updateSettings();

   if(_fuzzyLed)
   {
      _fuzzyLed->setColor(KBabelSettings::ledColor());
   }
   if(_untransLed)
   {
      _untransLed->setColor(KBabelSettings::ledColor());
   }
   if(_errorLed)
   {
      _errorLed->setColor(KBabelSettings::ledColor());
   }

}


void KBabelMW::setupActions()
{
   TDEAction* action;

   // the file menu
   action = KStdAction::open(this, TQ_SLOT(fileOpen()), actionCollection());

   a_recent = KStdAction::openRecent(this, TQ_SLOT(openRecent(const KURL&)), actionCollection());
   a_recent->setMaxItems(MAX_RECENT);
   
   action = KStdAction::revert(m_view,TQ_SLOT(revertToSaved()),actionCollection());
   action=KStdAction::save(this, TQ_SLOT(fileSave()), actionCollection());
   action = KStdAction::saveAs(this, TQ_SLOT(fileSaveAs()), actionCollection());
   action = new TDEAction(i18n("Save Sp&ecial..."), 0, this, TQ_SLOT(fileSaveSpecial()),
         actionCollection(), "save_special" );
   action = new TDEAction(i18n("Set &Package..."), 0, m_view, TQ_SLOT(setFilePackage()),
         actionCollection(), "set_package" );

   action = KStdAction::mail(this, TQ_SLOT(fileMail()), actionCollection());

   action = new TDEAction(i18n("&New View"), 0, this, TQ_SLOT(fileNewView()),
         actionCollection(), "file_new_view");

   action = new TDEAction(i18n("New &Window"), 0, this, TQ_SLOT(fileNewWindow()),
         actionCollection(), "file_new_window");
   action->setShortcut( TDEStdAccel::openNew() );

   action = KStdAction::quit(this, TQ_SLOT(quit()), actionCollection());



   // the edit menu
   action = KStdAction::undo(m_view, TQ_SLOT(undo()), actionCollection());
   action = KStdAction::redo(m_view, TQ_SLOT(redo()), actionCollection());
   action = KStdAction::cut(m_view, TQ_SIGNAL(signalCut()), actionCollection());
   action = KStdAction::copy(m_view, TQ_SIGNAL(signalCopy()), actionCollection());
   action = KStdAction::paste(m_view, TQ_SIGNAL(signalPaste()), actionCollection());
   action = KStdAction::selectAll(m_view, TQ_SIGNAL(signalSelectAll()), actionCollection());
   action = KStdAction::find(m_view, TQ_SLOT(find()), actionCollection());
   action = KStdAction::findNext(m_view, TQ_SLOT(findNext()), actionCollection());
   action = KStdAction::findPrev(m_view, TQ_SLOT(findPrev()), actionCollection());
   action = KStdAction::replace(m_view, TQ_SLOT(replace()), actionCollection());

   action = KStdAction::clear( m_view, TQ_SLOT(clear()), actionCollection(), "clear" );

   action = new TDEAction(i18n("Cop&y Msgid to Msgstr"), "msgid2msgstr", CTRL+Key_Space, m_view
                 ,TQ_SLOT(msgid2msgstr()), actionCollection(), "msgid2msgstr");
   action = new TDEAction(i18n("Copy Searc&h Result to Msgstr"), "search2msgstr",
                        CTRL+ALT+Key_Space, m_view, TQ_SLOT(search2msgstr()),
                        actionCollection(), "search2msgstr");
   action = new TDEAction(i18n("Copy Msgstr to Other &Plurals"), Key_F11, m_view
                   ,TQ_SLOT(plural2msgstr()), actionCollection(), "plural2msgstr");
   action = new TDEAction(i18n("Copy Selected Character to Msgstr"), Key_F10, m_charselectorview
                   ,TQ_SLOT(emitChar()), actionCollection(), "char2msgstr");

   a_unsetFuzzy = new TDEAction(i18n("To&ggle Fuzzy Status"), "togglefuzzy", CTRL+Key_U, m_view
                 , TQ_SLOT(removeFuzzyStatus()), actionCollection(), "edit_toggle_fuzzy");
   action = new TDEAction(i18n("&Edit Header..."), 0, m_view, TQ_SLOT(editHeader()),
         actionCollection(), "edit_edit_header");

   action = new TDEAction(i18n("&Insert Next Tag"), "insert_tag", CTRL+ALT+Key_N
           , m_view, TQ_SLOT(insertNextTag())
           , actionCollection(),"insert_next_tag");
   connect(m_view,TQ_SIGNAL(signalNextTagAvailable(bool)),action
           ,TQ_SLOT(setEnabled(bool)));
   action = new TDEAction(i18n("Insert Next Tag From Msgid P&osition"), "insert_tag", CTRL+Key_M
           , m_view, TQ_SLOT(insertNextTagMsgid())
           , actionCollection(),"insert_next_tag_msgid");
   connect(m_view,TQ_SIGNAL(signalNextTagAvailable(bool)),action
           ,TQ_SLOT(setEnabled(bool)));
   TDEActionMenu *actionMenu= new TDEActionMenu(i18n("Inser&t Tag"), "insert_tag"
         , actionCollection(),"insert_tag");
   m_view->setTagsMenu(actionMenu->popupMenu());
   connect(m_view,TQ_SIGNAL(signalTagsAvailable(bool)),actionMenu
           ,TQ_SLOT(setEnabled(bool)));
   connect(actionMenu,TQ_SIGNAL(activated()),m_view,TQ_SLOT(insertNextTag()));

   action = new TDEAction(i18n("Show Tags Menu"),CTRL+Key_Less
           , m_view, TQ_SLOT(showTagsMenu()), actionCollection(),"show_tags_menu");
   action->setEnabled(false);

   connect(m_view,TQ_SIGNAL(signalTagsAvailable(bool)),action
           ,TQ_SLOT(setEnabled(bool)));

   action = new TDEAction(i18n("Move to Next Tag"), 0, CTRL+ALT+Key_M
           , m_view, TQ_SLOT(skipToNextTag())
           , actionCollection(),"move_to_next_tag");

   action = new TDEAction(i18n("Move to Previous Tag"), 0, CTRL+ALT+Key_B
           , m_view, TQ_SLOT(skipToPreviousTag())
           , actionCollection(),"move_to_prev_tag");

   action = new TDEAction(i18n("Insert Next Argument"), "insert_arg", CTRL+ALT+Key_G
           , m_view, TQ_SLOT(insertNextArg())
           , actionCollection(),"insert_next_arg");
   connect(m_view,TQ_SIGNAL(signalNextArgAvailable(bool)),action
           ,TQ_SLOT(setEnabled(bool)));
   actionMenu= new TDEActionMenu(i18n("Inser&t Argument"), "insert_arg"
         , actionCollection(),"insert_arg");
   m_view->setArgsMenu(actionMenu->popupMenu());
   connect(m_view,TQ_SIGNAL(signalArgsAvailable(bool)),actionMenu
           ,TQ_SLOT(setEnabled(bool)));
   connect(actionMenu,TQ_SIGNAL(activated()),m_view,TQ_SLOT(insertNextArg()));

   action = new TDEAction(i18n("Show Arguments Menu"),CTRL+Key_Percent
           , m_view, TQ_SLOT(showArgsMenu()), actionCollection(),"show_args_menu");
   action->setEnabled(false);

   connect(m_view,TQ_SIGNAL(signalArgsAvailable(bool)),action
           ,TQ_SLOT(setEnabled(bool)));

    // next, the go-menu
    action = new TDEAction(i18n("&Previous"), "go-previous",
                 TDEStdAccel::shortcut(TDEStdAccel::Prior), m_view , TQ_SLOT(gotoPrev()),
                 actionCollection(),"go_prev_entry");
   action = new TDEAction(i18n("&Next"), "go-next",
                 TDEStdAccel::shortcut(TDEStdAccel::Next), m_view , TQ_SLOT(gotoNext()),
                 actionCollection(),"go_next_entry");
   action = KStdAction::goTo(m_view, TQ_SLOT(gotoEntry()), actionCollection());
   action->setShortcut( TDEStdAccel::gotoLine());
   action = KStdAction::firstPage(m_view, TQ_SLOT(gotoFirst()),actionCollection());
   action->setText(i18n("&First Entry"));
   action->setShortcut(CTRL+ALT+Key_Home);
   action = KStdAction::lastPage(m_view, TQ_SLOT(gotoLast()),actionCollection());
   action->setText(i18n("&Last Entry"));
   action->setShortcut(CTRL+ALT+Key_End);
   a_prevFoU = new TDEAction(i18n("P&revious Fuzzy or Untranslated"),"prevfuzzyuntrans",
         CTRL+SHIFT+Key_Prior, m_view,
         TQ_SLOT(gotoPrevFuzzyOrUntrans()),actionCollection(), "go_prev_fuzzyUntr");
   a_nextFoU = new TDEAction(i18n("N&ext Fuzzy or Untranslated"),"nextfuzzyuntrans",
         CTRL+SHIFT+Key_Next, m_view,
         TQ_SLOT(gotoNextFuzzyOrUntrans()),actionCollection(), "go_next_fuzzyUntr");
   a_prevFuzzy = new TDEAction(i18n("Pre&vious Fuzzy"),"prevfuzzy",
         CTRL+Key_Prior, m_view,
         TQ_SLOT(gotoPrevFuzzy()),actionCollection(), "go_prev_fuzzy");
   a_nextFuzzy = new TDEAction(i18n("Ne&xt Fuzzy"), "nextfuzzy",
         CTRL+Key_Next, m_view,
         TQ_SLOT(gotoNextFuzzy()),actionCollection(), "go_next_fuzzy");
   a_prevUntrans = new TDEAction(i18n("Prev&ious Untranslated"), "prevuntranslated",
         ALT+Key_Prior, m_view,
         TQ_SLOT(gotoPrevUntranslated()),actionCollection(), "go_prev_untrans");
   a_nextUntrans = new TDEAction(i18n("Nex&t Untranslated"), "nextuntranslated",
         ALT+Key_Next, m_view,
         TQ_SLOT(gotoNextUntranslated()),actionCollection(), "go_next_untrans");
   action = new TDEAction(i18n("Previo&us Error"), "preverror",
          SHIFT+Key_Prior, m_view,
         TQ_SLOT(gotoPrevError()),actionCollection(), "go_prev_error");
   action = new TDEAction(i18n("Next Err&or"), "nexterror",
         SHIFT+Key_Next, m_view,
         TQ_SLOT(gotoNextError()),actionCollection(), "go_next_error");
   action = new TDEAction(i18n("&Back in History"), "back", ALT+Key_Left, m_view,
         TQ_SLOT(backHistory()),actionCollection(), "go_back_history");
   action = new TDEAction(i18n("For&ward in History"), "forward", ALT+Key_Right, m_view,
         TQ_SLOT(forwardHistory()),actionCollection(), "go_forward_history");

    // the search menu
   actionMenu=new TDEActionMenu(i18n("&Find Text"),
         "transsearch",actionCollection(),"dict_search_all");
   connect(actionMenu,TQ_SIGNAL(activated()),m_view,TQ_SLOT(startSearch()));
   dictMenu = new DictionaryMenu(actionMenu->popupMenu(),actionCollection(),this);
   connect(dictMenu,TQ_SIGNAL(activated(const TQString)), m_view
           , TQ_SLOT(startSearch(const TQString)));

   actionMenu=new TDEActionMenu(i18n("F&ind Selected Text"),
         "transsearch",actionCollection(),"dict_search_selected");
   connect(actionMenu,TQ_SIGNAL(activated()),m_view,TQ_SLOT(startSelectionSearch()));
   selectionDictMenu = new DictionaryMenu(actionMenu->popupMenu(),actionCollection(),this);
   connect(selectionDictMenu,TQ_SIGNAL(activated(const TQString)), m_view
           , TQ_SLOT(startSelectionSearch(const TQString)));

   actionMenu=new TDEActionMenu(i18n("&Edit Dictionary"),
         "transsearch",actionCollection(),"dict_edit");
   editDictMenu = new DictionaryMenu(actionMenu->popupMenu(),actionCollection(),this);
   connect(editDictMenu,TQ_SIGNAL(activated(const TQString)), m_view
           , TQ_SLOT(editDictionary(const TQString)));


   actionMenu=new TDEActionMenu(i18n("Con&figure Dictionary"),
         "transsearch",actionCollection(),"dict_configure");
   configDictMenu = new DictionaryMenu(actionMenu->popupMenu(),actionCollection(),this);
   connect(configDictMenu,TQ_SIGNAL(activated(const TQString)), m_view
           , TQ_SLOT(configureDictionary(const TQString)));

   actionMenu=new TDEActionMenu(i18n("About Dictionary"), "transsearch",
         actionCollection(), "dict_about");
   aboutDictMenu = new DictionaryMenu(actionMenu->popupMenu(),actionCollection(),this);
   connect(aboutDictMenu,TQ_SIGNAL(activated(const TQString)), m_view
           , TQ_SLOT(aboutDictionary(const TQString)));

   buildDictMenus();

    // the project menu
   action = new TDEAction(i18n("&New..."), "document-new"
           , this, TQ_SLOT(projectNew()),actionCollection()
           ,"project_new");

   action = new TDEAction(i18n("&Open..."), "document-open"
           , this, TQ_SLOT(projectOpen()),actionCollection()
           ,"project_open");

   action = new TDEAction(i18n("C&lose"), "window-close"
           , this, TQ_SLOT(projectClose()),actionCollection()
           ,"project_close");
   action->setEnabled (_project->filename() != KBabel::ProjectManager::defaultProjectName() );

   action = new TDEAction(i18n("&Configure..."), "configure"
           , this, TQ_SLOT(projectConfigure()),actionCollection()
           ,"project_settings");

   a_recentprojects = new TDERecentFilesAction(i18n("Open &Recent"), 0, this, TQ_SLOT(projectOpenRecent(const KURL&)), actionCollection(), "recent_projects");

    // the tools menu
   action = new TDEAction(i18n("&Spell Check..."), "tools-check-spelling", CTRL+Key_I
           , m_view, TQ_SLOT(spellcheckCommon()),actionCollection()
           ,"spellcheck_common");
   action = new TDEAction(i18n("&Check All..."), "spellcheck_all", 0
           , m_view, TQ_SLOT(spellcheckAll()),actionCollection()
           ,"spellcheck_all");
   action = new TDEAction(i18n("C&heck From Cursor Position..."), "spellcheck_from_cursor", 0
           , m_view, TQ_SLOT(spellcheckFromCursor()),actionCollection()
           ,"spellcheck_from_cursor");
   action = new TDEAction(i18n("Ch&eck Current..."), "spellcheck_actual", 0
           , m_view, TQ_SLOT(spellcheckCurrent()),actionCollection()
           ,"spellcheck_current");
   action = new TDEAction(i18n("Check Fro&m Current to End of File..."), 0
           , m_view, TQ_SLOT(spellcheckFromCurrent()),actionCollection()
           ,"spellcheck_from_current");
   action = new TDEAction(i18n("Chec&k Selected Text..."), "spellcheck_selected", 0
           , m_view, TQ_SLOT(spellcheckMarked()),actionCollection()
           ,"spellcheck_marked");

   TDEToggleAction *toggleAction;

   toggleAction = new TDEToggleAction(i18n("&Diffmode"), "autodiff", 0
           ,actionCollection(), "diff_toggleDiff");
   connect(toggleAction,TQ_SIGNAL(toggled(bool)), m_view, TQ_SLOT(toggleAutoDiff(bool)));
   connect(m_view,TQ_SIGNAL(signalDiffEnabled(bool)), toggleAction
               , TQ_SLOT(setChecked(bool)));
   toggleAction->setChecked(m_view->autoDiffEnabled());

   action = new TDEAction(i18n("&Show Diff"), "diff", Key_F5
           , m_view, TQ_SLOT(diff()),actionCollection()
           ,"diff_diff");
   action = new TDEAction(i18n("S&how Original Text"), "contents", Key_F6
           , m_view, TQ_SLOT(diffShowOrig()),actionCollection()
           ,"diff_showOrig");

   action = new TDEAction(i18n("&Open File for Diff"), "document-open" ,0
           , m_view, TQ_SLOT(openDiffFile()),actionCollection()
           ,"diff_openFile");

   action = new TDEAction(i18n("&Rough Translation..."), 0
           , m_view, TQ_SLOT(roughTranslation()),actionCollection()
           ,"rough_translation");

   action = new TDEAction(i18n("&Catalog Manager..."),"catalogmanager", 0 , this,
         TQ_SLOT(openCatalogManager()),actionCollection(), "open_catalog_manager");

   new TDEAction( i18n("Toggle Edit Mode"), 0, Key_Insert,this,TQ_SLOT(toggleEditMode()), actionCollection(), "toggle_insert_mode");
   
   new TDEAction( i18n("&Word Count"), 0, m_view, TQ_SLOT(wordCount()), actionCollection(), "word_count");   

    // next, the settings menu
    createStandardStatusBarAction();

    KStdAction::configureToolbars(this,TQ_SLOT(optionsEditToolbars()),actionCollection());

    KStdAction::keyBindings(guiFactory(),TQ_SLOT(configureShortcuts()),actionCollection());
    KStdAction::preferences(this,TQ_SLOT(optionsPreferences()),actionCollection());

    setStandardToolBarMenuEnabled ( true );

   action = new TDEAction(i18n("&Stop Searching"), "process-stop",Key_Escape, m_view,
         TQ_SLOT(stopSearch()),actionCollection(), "stop_search");
   action->setEnabled(false);

   new TDEAction(i18n("&Gettext Info"), 0, this,
           TQ_SLOT(gettextHelp()), actionCollection(), "help_gettext");


   // the bookmarks menu

   action = KStdAction::addBookmark(this, TQ_SLOT(slotAddBookmark()),
      actionCollection(), "add_bookmark");
   action->setEnabled(false);
   // this action is now connected to dummySlot(), and later reconnected
   // to bmHandler after that object actually is created
   new TDEAction(i18n("Clear Bookmarks"), 0, this, TQ_SLOT(dummySlot()),
      actionCollection(), "clear_bookmarks");

   setupDynamicActions();

   createGUI(0);

   TQPopupMenu *popup = static_cast<TQPopupMenu*>(factory()->container("settings",this));
   popup->insertItem( i18n("&Views"), dockHideShowMenu(), -1, 0 );
}


void KBabelMW::setupStatusBar()
{
    statusBar()->insertItem(i18n("Current: 0"),ID_STATUS_CURRENT);
    statusBar()->insertItem(i18n("Total: 0"),ID_STATUS_TOTAL);
    statusBar()->insertItem(i18n("Fuzzy: 0"),ID_STATUS_FUZZY);
    statusBar()->insertItem(i18n("Untranslated: 0"),ID_STATUS_UNTRANS);

    if(KBabelSettings::ledInStatusbar())
    {
       TQColor ledColor=KBabelSettings::ledColor();
       TQHBox* statusBox = new TQHBox(statusBar(),"statusBox");
       statusBox->setSpacing(2);
       new TQLabel(" "+i18n("Status: "),statusBox);
       _fuzzyLed = new KLed(ledColor,KLed::Off,KLed::Sunken,KLed::Rectangular
                   ,statusBox);
       _fuzzyLed->setFixedSize(15,12);
       new TQLabel(i18n("fuzzy")+" ",statusBox);
       _untransLed = new KLed(ledColor,KLed::Off,KLed::Sunken,KLed::Rectangular
                   ,statusBox);
       _untransLed->setFixedSize(15,12);
       new TQLabel(i18n("untranslated")+" ",statusBox);
       _errorLed = new KLed(ledColor,KLed::Off,KLed::Sunken,KLed::Rectangular
                   ,statusBox);
       _errorLed->setFixedSize(15,12);
       new TQLabel(i18n("faulty")+" ",statusBox);

       statusBox->setFixedWidth(statusBox->sizeHint().width());
       statusBar()->addWidget(statusBox);
    }

    statusBar()->insertItem(i18n("INS"),ID_STATUS_EDITMODE);

    statusBar()->insertItem(i18n("RW"),ID_STATUS_READONLY);

    statusBar()->insertItem(i18n("Line: %1 Col: %2").arg(1).arg(1)
            ,ID_STATUS_CURSOR);

    TQHBox* progressBox = new TQHBox(statusBar(),"progressBox");
    progressBox->setSpacing(2);
    _progressLabel = new KSqueezedTextLabel( "", progressBox );
    _progressBar=new MyKProgress(progressBox,"progressbar");
    _progressBar->hide();
    progressBox->setStretchFactor(_progressBar,1);

    statusBar()->addWidget(progressBox,1);
    statusBar()->setMinimumHeight(progressBox->sizeHint().height());

    TQWhatsThis::add(statusBar(),
       i18n("<qt><p><b>Statusbar</b></p>\n\
<p>The statusbar displays some information about the opened file,\n\
like the total number of entries and the number of fuzzy and untranslated\n\
messages. Also the index and the status of the currently displayed entry is shown.</p></qt>"));

}

void KBabelMW::setupDynamicActions()
{
   // dynamic validation tools
   TQValueList<KDataToolInfo> tools = ToolAction::validationTools();

   TQPtrList<TDEAction> actions = ToolAction::dataToolActionList(
      tools, m_view, TQ_SLOT(validateUsingTool( const KDataToolInfo &, const TQString & )),
      "validate", false, actionCollection() );

   TDEActionMenu* m_menu = new TDEActionMenu(i18n("&Validation"), actionCollection(), "dynamic_validation_tools");

   TDEAction* ac = new TDEAction(i18n("Perform &All Checks"), CTRL+Key_E , m_view,
         TQ_SLOT(checkAll()),actionCollection(), "check_all");
   ac->setEnabled(false);
   m_menu->insert(ac);

   m_menu->insert( new TDEActionSeparator() );

   ac = new TDEAction(i18n("C&heck Syntax"), CTRL+Key_T , m_view,
         TQ_SLOT(checkSyntax()),actionCollection(), "check_syntax");
   ac->setEnabled(false);
   m_menu->insert(ac);

   for( ac = actions.first(); ac ; ac = actions.next() )
   {
        m_menu->insert(ac);
   }

   // dynamic modify tools

   // query available tools
   TQValueList<KDataToolInfo> allTools = KDataToolInfo::query
        ("CatalogItem", "application/x-kbabel-catalogitem", TDEGlobal::instance());

   // skip read-only tools for single items
   TQValueList<KDataToolInfo> modifyTools;

   TQValueList<KDataToolInfo>::ConstIterator entry = allTools.begin();
   for( ; entry != allTools.end(); ++entry )
   {
        if( !(*entry).isReadOnly() )
        {
            modifyTools.append( (*entry) );
        }
   }

   // create corresponding actions
   actions = ToolAction::dataToolActionList(
      modifyTools, m_view, TQ_SLOT(modifyUsingTool( const KDataToolInfo &, const TQString & )),
      "validate", true, actionCollection() );

   // skip validation actions
   for( ac = actions.first(); ac ; ac = actions.next() )
   {
        m_menu->insert(ac);
   }

   // insert tools
   m_menu = new TDEActionMenu(i18n("&Modify"), actionCollection(), "dynamic_modify_tools");
   for( ac = actions.first(); ac ; ac = actions.next() )
   {
        m_menu->insert(ac);
   }

   // query available tools for whole catalog
   allTools = KDataToolInfo::query
        ("Catalog", "application/x-kbabel-catalog", TDEGlobal::instance());

   // skip read-only tools
   entry = allTools.begin();
   for( ; entry != allTools.end(); ++entry )
   {
        if( !(*entry).isReadOnly() )
        {
            modifyTools.append( (*entry) );
        }
   }

   // create corresponding actions
   actions = ToolAction::dataToolActionList(
      modifyTools, m_view, TQ_SLOT(modifyUsingTool( const KDataToolInfo &, const TQString & )),
      "validate", true, actionCollection() );

   // skip validation actions
   for( ac = actions.first(); ac ; ac = actions.next() )
   {
        m_menu->insert(ac);
   }

   // create corresponding actions
   actions = ToolAction::dataToolActionList(
      modifyTools, m_view, TQ_SLOT(modifyCatalogUsingTool( const KDataToolInfo &, const TQString & )),
      "validate", true, actionCollection() );

   // insert tools
   m_menu = new TDEActionMenu(i18n("&Modify"), actionCollection(), "dynamic_modify_tools");
   for( ac = actions.first(); ac ; ac = actions.next() )
   {
        m_menu->insert(ac);
   }
}

void KBabelMW::saveSettings()
{
    {
        saveMainWindowSettings(_config, "View");
        writeDockConfig (_config, "View");
    }

    {
       a_recent->saveEntries(_config);
       a_recentprojects->saveEntries(_config,"Project");
    }

    _config->sync();
}

void KBabelMW::restoreSettings()
{
    {
       applyMainWindowSettings(_config, "View");
       TDEConfigGroupSaver saver(_config,"View");

       _config->setGroup("View");
       m_view->restoreView(_config);

        readDockConfig (_config, "View");
    }

    {
       a_recent->loadEntries(_config);

       a_recentprojects->loadEntries(_config, "Project");
    }
}



void KBabelMW::saveProperties(TDEConfig *config)
{
   m_view->saveSession(config);
}

void KBabelMW::readProperties(TDEConfig *config)
{
   m_view->restoreSession(config);

   // need to ensure that the windows is propertly setup also
   // for new views-only
   if(!m_view->currentURL().isEmpty())
   {
       KBCatalog* catalog=m_view->catalog();
       enableDefaults(catalog->isReadOnly());
       setNumberOfFuzzies(catalog->numberOfFuzzies());
       setNumberOfUntranslated(catalog->numberOfUntranslated());
       setNumberOfTotal(catalog->numberOfEntries());

       enableUndo(catalog->isUndoAvailable());
       enableUndo(catalog->isRedoAvailable());

       m_view->emitEntryState();

       changeCaption(catalog->currentURL().prettyURL() );
    }
}

bool KBabelMW::queryClose()
{
   if(m_view->isSearching())
   {
       connect(m_view,TQ_SIGNAL(signalSearchActive(bool)),this,TQ_SLOT(quit()));
       m_view->stopSearch();
       return false;
   }

   if(m_view->catalog()->isActive())
   {
       // stop the activity and try again
       m_view->catalog()->stop();
       TQTimer::singleShot(0, this, TQ_SLOT( close() ));
       return false;
   }

   if(m_view->isModified())
   {
      switch(KMessageBox::warningYesNoCancel(this,
      i18n("The document contains unsaved changes.\n\
Do you want to save your changes or discard them?"),i18n("Warning"),
      KStdGuiItem::save(),KStdGuiItem::discard()))
      {
         case KMessageBox::Yes:
         {
            return m_view->saveFile();
         }
         case KMessageBox::No:
            return true;
         default:
            return false;
      }
   }

  return true;
}

bool KBabelMW::queryExit()
{
   saveSettings();
   _config->setGroup("View");
   m_view->saveView(_config);

   m_view->saveSettings();
   return true;
}

void KBabelMW::quit()
{
    close();
}


void KBabelMW::dragEnterEvent(TQDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(KURLDrag::canDecode(event));
}

void KBabelMW::dropEvent(TQDropEvent *event)
{
    KURL::List uri;
    // see if we can decode a URI.. if not, just ignore it
    if (KURLDrag::decode(event, uri))
    {
       m_view->processUriDrop(uri,mapToGlobal(event->pos()));
    }
}

void KBabelMW::wheelEvent(TQWheelEvent *e)
{
    m_view->wheelEvent(e);
}

void KBabelMW::openRecent(const KURL& url)
{
   KBabelView *view = KBabelView::viewForURL(url,TQString());
   if(view)
   {
       KWin::activateWindow(view->topLevelWidget()->winId());
       return;
   }

   m_view->open(url);
}

void KBabelMW::open(const KURL& url)
{
   open(url,TQString(),false);
}

void KBabelMW::open(const KURL& url, const TQString package, bool newWindow)
{
   kdDebug(KBABEL) << "opening file with project:" << _project->filename() << endl;
   kdDebug(KBABEL) << "URL:" << url.prettyURL() << endl;
   KBabelView *view = KBabelView::viewForURL(url, _project->filename());
   if(view)
   {
        kdDebug(KBABEL) << "there is a such view" << endl;
       KWin::activateWindow(view->topLevelWidget()->winId());
       return;
   }

   addToRecentFiles(url);

   if(newWindow)
   {
        kdDebug(KBABEL) << "creating new window"<< endl;
      fileNewWindow()->open(url, package,false);
   }
   else
   {
        m_view->open(url,package);
   }
}

void KBabelMW::openTemplate(const KURL& openURL,const KURL& saveURL,const TQString& package, bool newWindow)
{
   if(newWindow)
   {
      fileNewWindow()->openTemplate(openURL,saveURL,package,false);
   }
   else
   {
      m_view->openTemplate(openURL,saveURL);
      m_view->catalog()->setPackage(package);
   }
}

void KBabelMW::fileOpen()
{
    m_view->open();

    KURL url=m_view->currentURL();
    addToRecentFiles(url);
}


void KBabelMW::addToRecentFiles(KURL url)
{
   if( url.isValid() && ! url.isEmpty() )
        a_recent->addURL(url);
}

void KBabelMW::fileSave()
{
    // do it asynchronously due to tdelibs bug
    TQTimer::singleShot( 0, this, TQ_SLOT( fileSave_internal() ));
}

void KBabelMW::fileSave_internal()
{
    // this slot is called whenever the File->Save menu is selected,
    // the Save shortcut is pressed (usually CTRL+S) or the Save toolbar
    // button is clicked

    if(!m_view->isModified())
    {
       statusBar()->message(i18n("There are no changes to save."),2000);
    }
    else
    {
       // disable save
       TDEAction* saveAction=(TDEAction*)actionCollection()->action( KStdAction::name( KStdAction::Save) );
       saveAction->setEnabled(false);

       m_view->saveFile();

       KURL url=m_view->currentURL();

       DCOPClient *client = tdeApp->dcopClient();
       TQByteArray data;
       TQDataStream arg(data, IO_WriteOnly);
       arg << ((url.directory(false)+url.fileName()).utf8())  ;
       if( !client->send( "catalogmanager-*", "CatalogManagerIFace", "updatedFile(TQCString)", data ))
        kdDebug(KBABEL) << "Unable to send file update info via DCOP" << endl;

       // reenable save action
       saveAction->setEnabled(true);
    }
}

void KBabelMW::fileSaveAs()
{
    m_view->saveFileAs();
    KURL url=m_view->currentURL();

    DCOPClient *client = tdeApp->dcopClient();
    TQByteArray data;
    TQDataStream arg(data, IO_WriteOnly);
    arg << ((url.directory(false)+url.fileName()).utf8())  ;
    if( !client->send( "catalogmanager-*", "CatalogManagerIFace", "updatedFile(TQCString)", data ))
        kdDebug(KBABEL) << "Unable to send file update info via DCOP" << endl;
}

void KBabelMW::fileSaveSpecial()
{
    if( !m_view->saveFileSpecial() ) return;

    KURL url=m_view->currentURL();

    DCOPClient *client = tdeApp->dcopClient();
    TQByteArray data;
    TQDataStream arg(data, IO_WriteOnly);
    arg << ((url.directory(false)+url.fileName()).utf8())  ;
    if( !client->send( "catalogmanager-*", "CatalogManagerIFace", "updatedFile(TQCString)", data ))
        kdDebug(KBABEL) << "Unable to send file update info via DCOP" << endl;
}

void KBabelMW::fileMail()
{
    if( m_view->isModified() ) fileSave();
    mailer->sendOneFile( m_view->currentURL() );
}

void KBabelMW::fileNewView()
{
   KBabelMW* b=new KBabelMW(m_view->catalog(),_project->filename());
   b->updateSettings();
   b->initBookmarks(bmHandler->bookmarks());
   b->show();
}

KBabelMW* KBabelMW::fileNewWindow()
{
   KBabelMW* b=new KBabelMW(_project->filename());
   b->setSettings(m_view->catalog()->saveSettings(),m_view->catalog()->identitySettings());
   b->show();

   return b;
}

void KBabelMW::toggleEditMode()
{
   bool ovr=!m_view->isOverwriteMode();

   m_view->setOverwriteMode(ovr);

   if(ovr)
      statusBar()->changeItem(i18n("OVR"),ID_STATUS_EDITMODE);
   else
      statusBar()->changeItem(i18n("INS"),ID_STATUS_EDITMODE);

}

void KBabelMW::optionsShowStatusbar(bool on)
{
   if(on)
   {
      statusBar()->show();
   }
   else
   {
      statusBar()->hide();
   }
}

void KBabelMW::optionsEditToolbars()
{
   saveMainWindowSettings( TDEGlobal::config(), "View" );
   KEditToolbar dlg(actionCollection());
   connect(&dlg, TQ_SIGNAL(newToolbarConfig()), this, TQ_SLOT(newToolbarConfig()));
   dlg.exec();
}

void KBabelMW::newToolbarConfig()
{
    createGUI(0);
    applyMainWindowSettings( TDEGlobal::config(), "View" );
}

void KBabelMW::optionsPreferences()
{
    if(!_prefDialog)
    {
        _prefDialog = new KBabelPreferences(m_view->dictionaries());
        prefDialogs.append(_prefDialog);

        connect(_prefDialog,TQ_SIGNAL(settingsChanged())
                ,m_view,TQ_SLOT(updateSettings()));
    }

    int prefHeight=_prefDialog->height();
    int prefWidth=_prefDialog->width();
    int width=this->width();
    int height=this->height();

    int x=width/2-prefWidth/2;
    int y=height/2-prefHeight/2;

    _prefDialog->move(mapToGlobal(TQPoint(x,y)));

    if(!_prefDialog->isVisible())
    {
       _prefDialog->show();
    }

    _prefDialog->raise();
    KWin::activateWindow(_prefDialog->winId());
}

void KBabelMW::setLedColor(const TQColor& color)
{
   if(_fuzzyLed)
   {
      _fuzzyLed->setColor(color);
   }
   if(_untransLed)
   {
      _untransLed->setColor(color);
   }
   if(_errorLed)
   {
      _errorLed->setColor(color);
   }
}

void KBabelMW::openCatalogManager()
{
    TQCString service;
    TQString result;

    DCOPClient * client = tdeApp->dcopClient();

    // find out, if there is a running catalog manager
    QCStringList apps = client->registeredApplications();
    for( QCStringList::Iterator it = apps.begin() ; it != apps.end() ; ++it )
    {
        TQString clientID = *it;
        if( clientID.startsWith("catalogmanager") )
        {
            service = *it;
            break;
        }
    }

    // if there is no running catalog manager, start one
    if( service.isEmpty() )
    {
        TQString prog = "catalogmanager";
        TQString url = "";
        if( tdeApp->startServiceByDesktopName(prog,url, &result,&service))
        {
             KMessageBox::error(this, i18n("Unable to use TDELauncher to start "
                "Catalog Manager. You should check the installation of TDE.\n"
                "Please start Catalog Manager manually."));
             return;
        }
    }

    // set preferred window
    TQByteArray data;
    TQDataStream arg(data, IO_WriteOnly);
    arg << (this->winId()) ;
    if( !client->send( service, "CatalogManagerIFace", "setPreferredWindow( WId )", data )) kdDebug(KBABEL) << "Unable to set preferred window via DCOP" << endl;
}



void KBabelMW::firstEntryDisplayed(bool firstEntry, bool firstForm)
{
   TDEAction* firstAction=(TDEAction*)actionCollection()->action(KStdAction::stdName(KStdAction::FirstPage));
   TDEAction* prevAction=(TDEAction*)actionCollection()->action("go_prev_entry");

   firstAction->setEnabled(!firstEntry);
   prevAction->setEnabled(!(firstEntry && firstForm));

}

void KBabelMW::lastEntryDisplayed(bool lastEntry, bool lastForm)
{
   TDEAction* lastAction=(TDEAction*)actionCollection()->action(KStdAction::stdName(KStdAction::LastPage));
   TDEAction* nextAction=(TDEAction*)actionCollection()->action("go_next_entry");

   lastAction->setEnabled(!lastEntry);
   nextAction->setEnabled(!(lastEntry && lastForm));
}

void KBabelMW::fuzzyDisplayed(bool flag)
{
    if(!_fuzzyLed)
       return;

    if(flag)
    {
       if(_fuzzyLed->state()==KLed::Off)
       {
          _fuzzyLed->on();
       }
    }
    else
    {
       if(_fuzzyLed->state()==KLed::On)
           _fuzzyLed->off();
    }
}

void KBabelMW::untranslatedDisplayed(bool flag)
{
    if(!_untransLed)
       return;

    // do not allow fuzzy toggle for untranslated
    TDEAction *action=actionCollection()->action("edit_toggle_fuzzy");
    if(action)
        action->setEnabled(!flag);


    if(flag)
    {
       if(_untransLed->state()==KLed::Off)
          _untransLed->on();
    }
    else
    {
       if(_untransLed->state()==KLed::On)
          _untransLed->off();
    }
}


void KBabelMW::faultyDisplayed(bool flag)
{
    if(!_errorLed)
       return;

    if(flag)
    {
       if(_errorLed->state()==KLed::Off)
          _errorLed->on();
    }
    else
    {
       if(_errorLed->state()==KLed::On)
          _errorLed->off();
    }
}


void KBabelMW::displayedEntryChanged(const KBabel::DocPosition& pos)
{
   statusBar()->changeItem(i18n("Current: %1").arg(pos.item+1),ID_STATUS_CURRENT);
  _currentIndex = pos.item;
}

void KBabelMW::setNumberOfTotal(uint number)
{
   statusBar()->changeItem(i18n("Total: %1").arg(number),ID_STATUS_TOTAL);
}

void KBabelMW::setNumberOfFuzzies(uint number)
{
   statusBar()->changeItem(i18n("Fuzzy: %1").arg(number),ID_STATUS_FUZZY);
}

void KBabelMW::setNumberOfUntranslated(uint number)
{
   statusBar()->changeItem(i18n("Untranslated: %1").arg(number),ID_STATUS_UNTRANS);
}

void KBabelMW::hasFuzzyAfterwards(bool flag)
{
   a_nextFuzzy->setEnabled(flag);

   // check if there is  a fuzzy or untranslated afterwards
   if( flag || a_nextUntrans->isEnabled() )
   {
       a_nextFoU->setEnabled(true);
   }
   else
   {
       a_nextFoU->setEnabled(false);
   }

}

void KBabelMW::hasFuzzyInFront(bool flag)
{
   a_prevFuzzy->setEnabled(flag);

   // check if there is  a fuzzy or untranslated in front
   if( flag || a_prevUntrans->isEnabled() )
   {
       a_prevFoU->setEnabled(true);
   }
   else
   {
       a_prevFoU->setEnabled(false);
   }
}

void KBabelMW::hasUntranslatedAfterwards(bool flag)
{
   a_nextUntrans->setEnabled(flag);

   // check if there is a fuzzy or untranslated afterwards
   if( flag || a_nextFuzzy->isEnabled() )
   {
       a_nextFoU->setEnabled(true);
   }
   else
   {
       a_nextFoU->setEnabled(false);
   }
}

void KBabelMW::hasUntranslatedInFront(bool flag)
{
   a_prevUntrans->setEnabled(flag);

   // check if there is  a fuzzy or translated in front
   if( flag || a_prevFuzzy->isEnabled() )
   {
       a_prevFoU->setEnabled(true);
   }
   else
   {
       a_prevFoU->setEnabled(false);
   }
}



void KBabelMW::hasErrorAfterwards(bool flag)
{
   TDEAction* action=actionCollection()->action("go_next_error");
   action->setEnabled(flag);
}

void KBabelMW::hasErrorInFront(bool flag)
{
   TDEAction* action=actionCollection()->action("go_prev_error");
   action->setEnabled(flag);
}


void KBabelMW::enableBackHistory(bool on)
{
   TDEAction* action=actionCollection()->action("go_back_history");
   action->setEnabled(on);
}

void KBabelMW::enableForwardHistory(bool on)
{
   TDEAction* action=actionCollection()->action("go_forward_history");
   action->setEnabled(on);
}


void KBabelMW::prepareProgressBar(TQString msg,int max)
{
   if(_statusbarTimer->isActive())
        _statusbarTimer->stop();

   _progressBar->show();
   _progressLabel->setText(" "+msg);
   _progressBar->setTotalSteps(max);
   _progressBar->setProgress(0);

}

void KBabelMW::clearProgressBar()
{
   _progressBar->setProgress(0);
   _progressBar->hide();
   _progressLabel->setText("      ");
}


void KBabelMW::changeStatusbar(const TQString& text)
{
    // display the text on the statusbar
    _progressLabel->setText(" "+text);

    if(_statusbarTimer->isActive())
        _statusbarTimer->stop();

    _statusbarTimer->start(5000,true);
}

void KBabelMW::clearStatusbarMsg()
{
    _progressLabel->setText("");
}

void KBabelMW::changeCaption(const TQString& text)
{
    // display the text on the caption
    setCaption(text + ( _project->filename () != KBabel::ProjectManager::defaultProjectName() ?
            " (" + _project->name() + ")" : "" /* KDE 3.4: i18n("(No project)")*/ )
        ,m_view->isModified());
}


void KBabelMW::showModified(bool on)
{
    // display the text on the caption
    setCaption(m_view->catalog()->package(),on);

    TDEAction *action=actionCollection()->action(
            KStdAction::stdName(KStdAction::Save));
    action->setEnabled(on);

    action=actionCollection()->action(KStdAction::stdName(KStdAction::Revert));
    action->setEnabled(on);
}


void KBabelMW::enableDefaults(bool readOnly)
{
    stateChanged( "readonly", readOnly ? StateNoReverse : StateReverse );
    stateChanged( "fileopened", StateNoReverse );

    if(readOnly)
       statusBar()->changeItem(i18n("RO"),ID_STATUS_READONLY);
    else
       statusBar()->changeItem(i18n("RW"),ID_STATUS_READONLY);
}

void KBabelMW::enableUndo(bool on)
{
   TDEAction* action=actionCollection()->action(KStdAction::stdName(KStdAction::Undo));
   action->setEnabled(on);
}

void KBabelMW::enableRedo(bool on)
{
   TDEAction* action=actionCollection()->action(KStdAction::stdName(KStdAction::Redo));
   action->setEnabled(on);
}

void KBabelMW::enableStop(bool flag)
{
   TDEAction* action=(TDEAction*)actionCollection()->action("stop_search");
   action->setEnabled(flag);
}

void KBabelMW::gettextHelp()
{
    TQString error;
    TDEApplication::startServiceByDesktopName("khelpcenter",
                TQString("info:/gettext"), &error);

    if(!error.isEmpty())
    {
        KMessageBox::sorry(this,i18n("An error occurred while "
                "trying to open the gettext info page:\n%1").arg(error));
    }
}

void KBabelMW::buildDictMenus()
{
   TQPtrList<ModuleInfo> dictList = m_view->dictionaries();
   dictList.setAutoDelete(true);

   dictMenu->clear();
   selectionDictMenu->clear();
   configDictMenu->clear();
   editDictMenu->clear();
   aboutDictMenu->clear();

   ModuleInfo *info;
   for(info = dictList.first(); info !=0; info = dictList.next())
   {
      TQString accel="Ctrl+Alt+%1";
      dictMenu->add(info->name,info->id, accel);

      accel=TQString("Ctrl+%1");
      selectionDictMenu->add(info->name,info->id, accel);

      configDictMenu->add(info->name,info->id);
      aboutDictMenu->add(info->name,info->id);

      if(info->editable)
      {
         dictMenu->add(info->name,info->id);
      }
   }
}

void KBabelMW::updateCursorPosition(int line, int col)
{
    statusBar()->changeItem(i18n("Line: %1 Col: %2").arg(line+1).arg(col+1)
            ,ID_STATUS_CURSOR);
}


KBabelMW *KBabelMW::winForURL(const KURL& url, TQString project)
{
    KBabelMW *kb=0;

    KBabelView *v = KBabelView::viewForURL(url,project);
    if(v)
    {
        TQObject *p = v->parent();
        while(p && !p->inherits("KBabelMW"))
        {
            p = p->parent();
        }

        if(p)
            kb = static_cast<KBabelMW*>(p);
    }

    return kb;
}

KBabelMW *KBabelMW::emptyWin(TQString project)
{
    KBabelMW *kb=0;

    KBabelView *v = KBabelView::emptyView(project);
    if(v)
    {
        TQObject *p = v->parent();
        while(p && !p->inherits("KBabelMW"))
        {
            p = p->parent();
        }

        if(p)
            kb = static_cast<KBabelMW*>(p);
    }

    return kb;
}

void KBabelMW::spellcheckMoreFiles(TQStringList filelist)
{
    if( filelist.isEmpty() ) return;
    _toSpellcheck = filelist;
    connect( m_view, TQ_SIGNAL( signalSpellcheckDone(int) ), this, TQ_SLOT( spellcheckDone(int)));
    spellcheckDone( KS_IGNORE ); // use something else than KS_STOP
}

void KBabelMW::spellcheckDone( int result)
{
    if( _toSpellcheck.isEmpty() || result == KS_STOP)
    {
        disconnect( m_view, TQ_SIGNAL( signalSpellcheckDone(int)), this, TQ_SLOT(spellcheckDone( int)));
        KMessageBox::information( this, i18n("MessageBox text", "Spellchecking of multiple files is finished."),
            i18n("MessageBox caption", "Spellcheck Done"));
    }
    else
    {
        TQString file = _toSpellcheck.first();
        _toSpellcheck.pop_front();
        if( m_view->isModified() ) fileSave();
        open(KURL( file ), TQString(), false);
        kdDebug(KBABEL) << "Starting another spellcheck" << endl;
        TQTimer::singleShot( 1, m_view, TQ_SLOT(spellcheckAllMulti()));
    }
}

void KBabelMW::initBookmarks(TQPtrList<KBabelBookmark> list)
{
  bmHandler->setBookmarks(list);
}

void KBabelMW::slotAddBookmark()
{
    bmHandler->addBookmark(_currentIndex+1, m_view->catalog()->msgid(_currentIndex).first());
}

void KBabelMW::slotOpenBookmark(int index)
{
  DocPosition pos;
  pos.item=index-1;
  pos.form=0;
  m_view->gotoEntry(pos);
}

void KBabelMW::projectNew()
{
    KBabel::Project::Ptr p = KBabel::ProjectWizard::newProject();
    if( p )
    {
        _project = p;
        m_view->useProject(p);
        changeProjectActions(p->filename());
    }
}

void KBabelMW::projectOpen()
{
    TQString oldproject = m_view->project();
    if( oldproject == KBabel::ProjectManager::defaultProjectName() )
    {
        oldproject = TQString();
    }
    const TQString file = KFileDialog::getOpenFileName(oldproject, TQString(), this);
    if (file.isEmpty())
    {
        return;
    }
    
    projectOpen (file);
}

void KBabelMW::projectOpenRecent(const KURL& url)
{
    projectOpen (url.path());
    KBabel::Project::Ptr p = KBabel::ProjectManager::open(url.path());
    if( p )
    {
        _project = p;
        m_view->useProject(p);
        changeProjectActions(url.path());
    }
}

void KBabelMW::projectOpen(const TQString& file)
{
    TQString oldproject = m_view->project();
    if( oldproject == KBabel::ProjectManager::defaultProjectName() )
    {
        oldproject = "";
    }
    if (file.isEmpty())
    {
        return;
    }
    KBabel::Project::Ptr p = KBabel::ProjectManager::open(file);
    if( p )
    {
        _project = p;
        m_view->useProject(p);
        changeProjectActions(file);
    }
    else 
    {
	KMessageBox::error( this, i18n("Cannot open project file\n%1").arg(file)
            , i18n("Project File Error"));
	_project = ProjectManager::open(KBabel::ProjectManager::defaultProjectName());
	m_view->useProject(_project);
	changeProjectActions(KBabel::ProjectManager::defaultProjectName());
    }
}

void KBabelMW::projectClose()
{
    TQString defaultProject = KBabel::ProjectManager::defaultProjectName();
    m_view->useProject( KBabel::ProjectManager::open(defaultProject) );
    _project = ProjectManager::open(defaultProject);
    changeProjectActions(defaultProject);
}

void KBabelMW::changeProjectActions(const TQString& project)
{
    bool def = (project == KBabel::ProjectManager::defaultProjectName());

    TDEAction* saveAction=(TDEAction*)actionCollection()->action( "project_close" );
    saveAction->setEnabled( ! def );
    
    if (!def)
    {
	addToRecentProjects(project);
    }
    
    // if there is a project dialog, delete it (we have a different project now
    if (_projectDialog) 
    {
	delete _projectDialog;
	_projectDialog = NULL;
    }
}

void KBabelMW::projectConfigure()
{
    if(!_projectDialog)
    {
        _projectDialog = new ProjectDialog(_project);
	connect (_projectDialog, TQ_SIGNAL (settingsChanged())
	    , m_view, TQ_SLOT (updateProjectSettings()));
    }

    int prefHeight=_projectDialog->height();
    int prefWidth=_projectDialog->width();
    int width=this->width();
    int height=this->height();

    int x=width/2-prefWidth/2;
    int y=height/2-prefHeight/2;

    _projectDialog->move(mapToGlobal(TQPoint(x,y)));

    if(!_projectDialog->isVisible())
    {
       _projectDialog->show();
    }

    _projectDialog->raise();
    KWin::activateWindow(_projectDialog->winId());
}

void KBabelMW::addToRecentProjects(KURL url)
{
   if( url.isValid() && ! url.isEmpty() )
	a_recentprojects->addURL(url);
}


#include "kbabel.moc"
