/*
 * Copyright (C) 2006 Hans van Leeuwen <hanz@hanz.nl>
 * Copyright (C) 2008 Armin Berres <trigger@space-based.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <iostream>
#include <tdewallet.h>
#include <kpassdlg.h>
#include <tdeaboutdata.h>
#include <tdeapplication.h>
#include <tdecmdlineargs.h>
#include <tdelocale.h>

static TDECmdLineOptions options[] =
{
    { "+[dialog]", I18N_NOOP( "Dialog message. Leave undefined for default message" ), 0 },
	TDECmdLineLastOption
};

int main(int argc, char **argv)
{

        TDEAboutData about (
                "tdesshaskpass",                                                                  // appName
                I18N_NOOP("tdesshaskpass"),                                                       // programName
                "0.4.1",                                                                        // version
                I18N_NOOP("TDE version of ssh-askpass"),                                        // shortDescription
                TDEAboutData::License_GPL,                                                      // licenseType
                "(c) 2006 Hans van Leeuwen\n(c) 2008 Armin Berres",                             // copyrightStatement statement
                I18N_NOOP("tdesshaskpass allows you to interactively prompt users for a passphrase for ssh-add"), // text
                "https://mirror.git.trinitydesktop.org/gitea/TDE/tdesshaskpass",                // homePageAddress
                "trigger@space-based.de"                                                        // bugsEmailAddress
        );
        about.addAuthor("Armin Berres", 0, "trigger@space-based.de");
        about.addAuthor("Hans van Leeuwen", 0, "hanz@hanz.nl");

	TDECmdLineArgs::init(argc, argv, &about);
	TDECmdLineArgs::addCmdLineOptions( options );
	TDECmdLineArgs *args = TDECmdLineArgs::parsedArgs();

	TDEApplication app;


	// Disable Session Management and DCOP. We don't need it.
	app.disableSessionManagement();
	app.disableAutoDcopRegistration();


	// Declare variables
	TQString walletFolder = about.appName();
	TQString dialog = I18N_NOOP("Please enter password");	// Default dialog text
	TQString keyFile;
	TQString password;

	
	// Parse commandline arguments
	if ( args->count() > 0 ) {
		dialog = args->arg(0);
		keyFile = dialog.section(" ", -2).remove(":");
	}
	args->clear();


	// Open TDEWallet to see if a password was previously stored.
	TDEWallet::Wallet *wallet = TDEWallet::Wallet::openWallet( TDEWallet::Wallet::NetworkWallet(), 0 );

	if ( wallet && wallet->hasFolder(walletFolder) ) {
		wallet->setFolder(walletFolder);

		TQString retrievedPass;

		wallet->readPassword(keyFile, retrievedPass);

		if (!retrievedPass.isNull())
		{
			password = retrievedPass;
		}
	}


	// Password could not be retrieved from wallet. Open password dialog
	if (password.isNull())
	{
		// create the password dialog, but only show "Enable Keep" button, if the wallet is opened
		KPasswordDialog *kpd = new KPasswordDialog(KPasswordDialog::Password, wallet, 0);
		kpd->setPrompt(dialog);
		kpd->setCaption(i18n("tdesshaskpass"));
		kpd->setAllowEmptyPasswords(false);
		// We don't want to dump core when the password dialog is shown, because it could contain the entered password.
		kpd->disableCoreDumps();

		if ( kpd->exec() == KDialog::Accepted ) {
			password = kpd->password();
		}

		// If "Enable Keep" is enabled, open/create a folder in TDEWallet and store the password.
		if (!password.isNull() && wallet && kpd->keep())
		{
			if ( !wallet->hasFolder( walletFolder ) ) {
				wallet->createFolder(walletFolder);
			}
			wallet->setFolder(walletFolder);
			wallet->writePassword(keyFile, password);
		}

		delete kpd;
	}

	// Close the wallet if it is opened.
	if (wallet) {
		TDEWallet::Wallet::closeWallet( TDEWallet::Wallet::NetworkWallet(), false );
	}

	// Finally return the password if one has been entered
	if(!password.isNull())
	{
		std::cout << password.local8Bit();
		return 0;
	}
	else
	{
		return 1;
	}
}
