#!/bin/bash

# Function to show help
showHelp() {
    echo "Usage: $(basename "$0") [options] <inputdir> [outputdir]"
    echo
    echo "Options:"
    echo "  -i, --inplace    Process files in place (overwriting original theme)"
    echo "  -p, --package    Create deKorator theme archive"
    echo "  -h, --help       Show this help message"
    exit 0
}

# Init vars
inplace=false
package=false
inputDir=""
outputDir=""

# Parse options
while [[ $# -gt 0 ]]; do
    case "$1" in
        -i|--inplace)
            inplace=true
            shift
            ;;
        -p|--package)
            package=true
            shift
            ;;
        -h|--help)
            showHelp
            ;;
        -*)
            echo "Unknown option: $1"
            echo ""
            showHelp
            ;;
        *)
            if [[ -z "$inputDir" ]]; then
                inputDir="$1"
            elif [[ -z "$outputDir" ]]; then
                outputDir="$1"
            else
                echo "Too many arguments."
                echo ""
                showHelp
            fi
            shift
            ;;
    esac
done

# Check for input dir
if [[ -z "$inputDir" ]]; then
    echo "Error: No input directory specified."
    echo ""
    showHelp
fi
inputDir=$(echo "$inputDir" | sed -e 's/^[[:space:]]*//;s/[[:space:]]*$//;s:/*$::')

# Check for inplace
if [[ "$inplace" == true ]]; then
    outputDir="$inputDir"
fi

# Check output dir
if [[ -z "$outputDir" ]]; then
    echo "Error: No output directory specified."
    echo ""
    showHelp
fi
outputDir=$(echo "$outputDir" | sed -e 's/^[[:space:]]*//;s/[[:space:]]*$//;s:/*$::')

# Check if directory ends with -theme
if ! [[ "$(basename "$outputDir")" == *-theme ]]; then
    if [[ "$inplace" == true ]]; then
        echo "Warning, directory name doesn't end in '-theme'. Keep this in mind for packaging."
    else
        echo "-> Renaming '${outputDir}' to '${outputDir}-theme' to facilitate packaging."
        outputDir="${outputDir}-theme"
    fi
fi

# Check output dir existence
if ! [[ -d "$outputDir" ]]; then
    echo "Creating directory $outputDir..."
    mkdir -p "$outputDir/buttons/hover"
    mkdir -p "$outputDir/buttons/normal"
    mkdir -p "$outputDir/buttons/press"
    mkdir -p "$outputDir/deco"
    mkdir -p "$outputDir/masks"
fi

# Check files.
echo "Checking files..."
# Is new style?
if [[ -f "${inputDir}/deco/leftUpperCornerFrameBg.png" ]]; then
    echo "-> Found leftUpperCornerFrameBg.png, theme seems to already use new filenames."
    echo "   Nothing changed, quit."
    exit 2
fi

partList='bottomLeftFrameBg.png-leftLowerCornerFrameBg.png bottomRightFrameBg.png-rightLowerCornerFrameBg.png leftBottomFrameBg.png-bottomLeftFrameBg.png rightBottomFrameBg.png-bottomRightFrameBg.png leftBottomShadedFrameBg.png-bottomShadedLeftFrameBg.png leftButtonsBg.png-buttonsLeftBg.png leftTitleBg.png-titleLeftBg.png topRightFrameBg.png-rightUpperFrameBg.png topLeftFrameBg.png-leftUpperFrameBg.png leftTopFrameBg.png-topLeftFrameBg.png midBottomFrameBg.png-bottomMidFrameBg.png midBottomShadedFrameBg.png-bottomShadedMidFrameBg.png midLeftFrameBg.png-leftMidFrameBg.png midRightFrameBg.png-rightMidFrameBg.png midTitleBg.png-titleMidBg.png midTopFrameBg.png-topMidFrameBg.png rightBottomShadedFrameBg.png-bottomShadedRightFrameBg.png rightButtonsBg.png-buttonsRightBg.png rightTitleBg.png-titleRightBg.png rightTopFrameBg.png-topRightFrameBg.png topLeftCornerBg.png-leftUpperCornerFrameBg.png topRightCornerBg.png-rightUpperCornerFrameBg.png'

echo "Copying images..."
for part in $partList; do
    oldpart="$(echo $part | cut -d '-' -f1)"
    newpart="$(echo $part | cut -d '-' -f2)"
    if [[ -f "${inputDir}/deco/${oldpart}" ]]; then
        echo "    ${oldpart}   ->   ${newpart}"
        if [[ "$inplace" == true ]]; then mv "${inputDir}/deco/${oldpart}" "${outputDir}/deco/${newpart}" 
        else cp "${inputDir}/deco/${oldpart}" "${outputDir}/deco/${newpart}"
        fi
    else
        echo "[!] File ${oldpart} not found."
    fi
done

# If not inplace copy buttons and masks
if [[ "$inplace" == false ]]; then
    echo "Copying buttons and masks..."
    cp -r "${inputDir}/buttons" "${outputDir}/"
    cp -r "${inputDir}/masks" "${outputDir}/"
fi

# Packaging theme
if [[ "$package" == true ]]; then
    echo "Packaging theme..."
    themeName="${outputDir%-theme}"
    if [[ "$outputDir" == */* ]]; then themeParentPath="${outputDir%/*}"
    else themeParentPath="."
    fi
    tar -cvzf "${themeParentPath}/${themeName}.tgz" "${outputDir}"
fi

echo " -> All done."