/* This file is part of the KDE project
   Copyright (C) 2005 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kexismalltoolbutton.h"

#include <tqtooltip.h>
#include <tqwhatsthis.h>
#include <tqstyle.h>

#include <kiconloader.h>
#include <tdeglobalsettings.h>

#include <core/kexi.h>

KexiSmallToolButton::KexiSmallToolButton(TQWidget* parent, const TQString& text,
	const TQString& icon, const char* name)
 : TQToolButton(parent, name)
{
	init();
	update(text, SmallIconSet(icon));
}

KexiSmallToolButton::KexiSmallToolButton(TQWidget* parent, const TQString& text,
	const TQIconSet& iconSet, const char* name)
 : TQToolButton(parent, name)
{
	init();
	update(text, iconSet);
}

KexiSmallToolButton::KexiSmallToolButton(TQWidget* parent, TDEAction* action)
 : TQToolButton(parent, action->name())
 , m_action(action)
{
	init();
	connect(this, TQ_SIGNAL(clicked()), action, TQ_SLOT(activate()));
	connect(action, TQ_SIGNAL(enabled(bool)), this, TQ_SLOT(setEnabled(bool)));
	updateAction();
}

KexiSmallToolButton::~KexiSmallToolButton()
{
}

void KexiSmallToolButton::updateAction()
{
	if (!m_action)
		return;
	update(m_action->text(), m_action->iconSet(TDEIcon::Small));
	setAccel(m_action->shortcut());
	TQToolTip::add(this, m_action->toolTip());
	TQWhatsThis::add(this, m_action->whatsThis());
}

void KexiSmallToolButton::init()
{
	setPaletteBackgroundColor(palette().active().background());
	setSizePolicy(TQSizePolicy::Fixed, TQSizePolicy::Preferred);
	TQFont f(TDEGlobalSettings::toolBarFont());
	f.setPixelSize(Kexi::smallFont().pixelSize());
	setFont(f);
	setAutoRaise(true);
}

void KexiSmallToolButton::update(const TQString& text, const TQIconSet& iconSet, bool tipToo)
{
	int width = 0;
	if (text.isEmpty()) {
		width = 10;
		setUsesTextLabel(false);
	}
	else {
		width += TQFontMetrics(font()).width(text+" ");
		setUsesTextLabel(true);
		setTextPosition(TQToolButton::Right);
		TQToolButton::setTextLabel(text, tipToo);
	}
	if (!iconSet.isNull()) {
		width += IconSize(TDEIcon::Small);
		TQToolButton::setIconSet(iconSet);
	}
	setFixedWidth( width );
}

void KexiSmallToolButton::setIconSet( const TQIconSet& iconSet )
{
	update(textLabel(), iconSet);
}

void KexiSmallToolButton::setIconSet( const TQString& icon )
{
	setIconSet( SmallIconSet(icon) );
}

void KexiSmallToolButton::setTextLabel( const TQString & newLabel, bool tipToo )
{
	Q_UNUSED( tipToo );

	update(newLabel, iconSet());
}

void KexiSmallToolButton::drawButton( TQPainter *_painter )
{
	TQToolButton::drawButton(_painter);
	if (TQToolButton::popup()) {
		TQStyle::SFlags arrowFlags = TQStyle::Style_Default;
		if (isDown())
			arrowFlags |= TQStyle::Style_Down;
		if (isEnabled())
			arrowFlags |= TQStyle::Style_Enabled;
		style().drawPrimitive(TQStyle::PE_ArrowDown, _painter,
			TQRect(width()-7, height()-7, 5, 5), colorGroup(),
			arrowFlags, TQStyleOption() );
	}
}

#include "kexismalltoolbutton.moc"
