// Copyright (c) 2003-2004 Rob Kaper <cap@capsi.com>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License version 2.1 as published by the Free Software Foundation.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; see the file COPYING.LIB.  If not, write to
// the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.

#include <iostream>

#include <tqheader.h>
#include <tqlayout.h>
#include <tqdatetime.h>

#include <tdelocale.h>
#include <tdelistview.h>
#include <kdialogbase.h>
#include <tdefiledialog.h>
#include <kiconloader.h>
#include <kpushbutton.h>
#include <kstringhandler.h>

#include "event.h"
#include "eventlogwidget.moc"

EventLog::EventLog()
{
}

void EventLog::addEvent(const TQString &description, const TQString &icon)
{
	Event *event = new Event(TQDateTime::currentDateTime(), description, icon);
	m_events.append(event);
	emit newEvent(event);
}

TQPtrList<Event> EventLog::events()
{
	return m_events;
}

EventLogWidget::EventLogWidget(EventLog *eventLog, TQWidget *parent, const char *name)
	: TQWidget(parent, name,
	  WType_Dialog | WStyle_Customize | WStyle_DialogBorder | WStyle_Title |
	  WStyle_Minimize | WStyle_ContextHelp )
{
	m_eventLog = eventLog;

	connect(m_eventLog, TQ_SIGNAL(newEvent(Event *)), this, TQ_SLOT(addEvent(Event *)));

	setCaption(i18n("Event Log"));

	TQVBoxLayout *listCompBox = new TQVBoxLayout(this, KDialog::marginHint());

	m_eventList = new TDEListView(this, "eventList");
	listCompBox->addWidget(m_eventList);

	m_eventList->addColumn(i18n("Date/Time"));
	m_eventList->addColumn(i18n("Description"));
	m_eventList->header()->setClickEnabled( false );

	TQHBoxLayout *actionBox = new TQHBoxLayout(this, 0, KDialog::spacingHint());
	listCompBox->addItem(actionBox);

	actionBox->addItem(new TQSpacerItem(20, 20, TQSizePolicy::Expanding, TQSizePolicy::Minimum));

	m_saveButton = new KPushButton(BarIcon("document-save", TDEIcon::SizeSmall), i18n("&Save As..."), this);
	actionBox->addWidget(m_saveButton);

	connect(m_saveButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(save()));

	// Populate
	TQPtrList<Event> events = m_eventLog->events();
	for (TQPtrListIterator<Event> it( events ); (*it) ; ++it)
		addEvent( (*it) );
}

void EventLogWidget::addEvent(Event *event)
{
	// FIXME: allow a way to view non-squeezed message
	// FIXME: allow a way to show older messages

	if ( m_eventList->childCount() >= 25 )
		delete m_eventList->firstChild();

	TQString description = KStringHandler::rsqueeze( event->description(), 200 );
	TDEListViewItem *item = new TDEListViewItem(m_eventList, event->dateTime().toString("yyyy-MM-dd hh:mm:ss zzz"), description);
	if (event->icon().isEmpty())
		item->setPixmap(1, TQPixmap(SmallIcon("atlantik")));
	else
		item->setPixmap(1, TQPixmap(SmallIcon(event->icon())));

	m_eventList->ensureItemVisible(item);
}

void EventLogWidget::closeEvent(TQCloseEvent *e)
{
	e->accept();
}

void EventLogWidget::save()
{
	TQFile file( KFileDialog::getSaveFileName() );
	if ( file.open( IO_WriteOnly ) )
	{
		TQTextStream stream(&file);

		stream << i18n( "Atlantik log file, saved at %1." ).arg( TQDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") ) << endl;

		TQPtrList<Event> events = m_eventLog->events();
		for (TQPtrListIterator<Event> it( events ); (*it) ; ++it)
			stream << (*it)->dateTime().toString("yyyy-MM-dd hh:mm:ss") << " " << (*it)->description() << endl;
		file.close();
	}
}
