/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "viewpropertiesdialog.h"
#include <tdelocale.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqvbox.h>
#include <tqbuttongroup.h>
#include <tqcheckbox.h>
#include <tqradiobutton.h>
#include <tqpushbutton.h>
#include <tqsizepolicy.h>
#include <tqgroupbox.h>
#include <tqcombobox.h>
#include <kiconloader.h>
#include <tdemessagebox.h>
#include <assert.h>

#include "viewproperties.h"
#include "dolphinview.h"

ViewPropertiesDialog::ViewPropertiesDialog(DolphinView* dolphinView) :
    KDialogBase(Plain, i18n("View Properties"),
                Ok|Apply|Cancel, Ok),
    m_isDirty(false),
    m_dolphinView(dolphinView)
{
    assert(dolphinView != 0);

    const int margin = KDialog::marginHint();
    const TQSizePolicy sizePolicy(TQSizePolicy::Preferred, TQSizePolicy::Fixed);

    const KURL& url = dolphinView->url();
    m_viewProps = new ViewProperties(url);
    m_viewProps->setAutoSaveEnabled(false);

    TQVBoxLayout* topLayout = new TQVBoxLayout(plainPage(), 0, spacingHint());

    // create 'Properties' group containing view mode, sorting, sort order and show hidden files
    TQGroupBox* propsGroup = new TQGroupBox(2, TQt::Horizontal, i18n("Properties"), plainPage());
    propsGroup->setSizePolicy(sizePolicy);
    propsGroup->setMargin(margin);

    new TQLabel(i18n("View mode:"), propsGroup);
    m_viewMode = new TQComboBox(propsGroup);
    m_viewMode->insertItem(SmallIcon("view_icon"), i18n("Icons"));
    m_viewMode->insertItem(SmallIcon("view_text"), i18n("Details"));
    m_viewMode->insertItem(SmallIcon("gvdirpart"), i18n("Previews"));
    const int index = static_cast<int>(m_viewProps->viewMode());
    m_viewMode->setCurrentItem(index);

    new TQLabel(i18n("Sorting:"), propsGroup);
    m_sorting = new TQComboBox(propsGroup);
    m_sorting->insertItem("By Name");
    m_sorting->insertItem("By Size");
    m_sorting->insertItem("By Date");
    int sortingIdx = 0;
    switch (m_viewProps->sorting()) {
        case DolphinView::SortByName: sortingIdx = 0; break;
        case DolphinView::SortBySize: sortingIdx = 1; break;
        case DolphinView::SortByDate: sortingIdx = 2; break;
        default: break;
    }
    m_sorting->setCurrentItem(sortingIdx);

    new TQLabel(i18n("Sort order:"), propsGroup);
    m_sortOrder = new TQComboBox(propsGroup);
    m_sortOrder->insertItem(i18n("Ascending"));
    m_sortOrder->insertItem(i18n("Descending"));
    const int sortOrderIdx = (m_viewProps->sortOrder() == TQt::Ascending) ? 0 : 1;
    m_sortOrder->setCurrentItem(sortOrderIdx);

    m_showHiddenFiles = new TQCheckBox(i18n("Show hidden files"), propsGroup);
    m_showHiddenFiles->setChecked(m_viewProps->isShowHiddenFilesEnabled());

    // create 'Apply view properties to:' group
    TQButtonGroup* buttonGroup = new TQButtonGroup(3,
                                                 TQt::Vertical,
                                                 i18n("Apply view properties to:"),
                                                 plainPage());
    buttonGroup->setSizePolicy(sizePolicy);
    buttonGroup->setMargin(margin);

    m_applyToCurrentFolder = new TQRadioButton(i18n("Current folder"), buttonGroup);
    buttonGroup->insert(m_applyToCurrentFolder);

    m_applyToSubFolders = new TQRadioButton(i18n("Current folder including all sub folders"), buttonGroup);
    buttonGroup->insert(m_applyToSubFolders);

    m_applyToAllFolders = new TQRadioButton(i18n("All folders"), buttonGroup);
    buttonGroup->insert(m_applyToAllFolders);

    if (m_viewProps->isValidForSubDirs()) {
        m_applyToSubFolders->setChecked(true);
    }
    else {
        m_applyToCurrentFolder->setChecked(true);
    }

    topLayout->addWidget(propsGroup);
    topLayout->addWidget(buttonGroup);

    connect(m_viewMode, TQ_SIGNAL(activated(int)),
            this, TQ_SLOT(slotViewModeChanged(int)));
    connect(m_sorting, TQ_SIGNAL(activated(int)),
            this, TQ_SLOT(slotSortingChanged(int)));
    connect(m_sortOrder, TQ_SIGNAL(activated(int)),
            this, TQ_SLOT(slotSortOrderChanged(int)));
    connect(m_showHiddenFiles, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotShowHiddenFilesChanged()));
    connect(m_applyToCurrentFolder, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotApplyToCurrentFolder()));
    connect(m_applyToSubFolders, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotApplyToSubFolders()));
    connect(m_applyToAllFolders, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotApplyToAllFolders()));
}

ViewPropertiesDialog::~ViewPropertiesDialog()
{
    m_isDirty = false;
    delete m_viewProps;
    m_viewProps = 0;
}

void ViewPropertiesDialog::slotOk()
{
    applyViewProperties();
    KDialogBase::slotOk();
}

void ViewPropertiesDialog::slotApply()
{
    applyViewProperties();
    KDialogBase::slotApply();
}

void ViewPropertiesDialog::slotViewModeChanged(int index)
{
    assert((index >= 0) && (index <= 2));
    m_viewProps->setViewMode(static_cast<DolphinView::Mode>(index));
    m_isDirty = true;
}

void ViewPropertiesDialog::slotSortingChanged(int index)
{
    DolphinView::Sorting sorting = DolphinView::SortByName;
    switch (index) {
        case 1: sorting = DolphinView::SortBySize; break;
        case 2: sorting = DolphinView::SortByDate; break;
        default: break;
    }
    m_viewProps->setSorting(sorting);
    m_isDirty = true;
}

void ViewPropertiesDialog::slotSortOrderChanged(int index)
{
    TQt::SortOrder sortOrder = (index == 0) ? TQt::Ascending : TQt::Descending;
    m_viewProps->setSortOrder(sortOrder);
    m_isDirty = true;
}

void ViewPropertiesDialog::slotShowHiddenFilesChanged()
{
    const bool show = m_showHiddenFiles->isChecked();
    m_viewProps->setShowHiddenFilesEnabled(show);
    m_isDirty = true;
}

void ViewPropertiesDialog::slotApplyToCurrentFolder()
{
    m_viewProps->setValidForSubDirs(false);
    m_isDirty = true;
}

void ViewPropertiesDialog::slotApplyToSubFolders()
{
    m_viewProps->setValidForSubDirs(true);
    m_isDirty = true;
}

void ViewPropertiesDialog::slotApplyToAllFolders()
{
    m_isDirty = true;
}

void ViewPropertiesDialog::applyViewProperties()
{
    if (m_applyToAllFolders->isChecked()) {
        if (m_isDirty) {
            const TQString text(i18n("The view properties of all folders will be replaced. Do you want to continue?"));
            if (KMessageBox::questionYesNo(this, text) == KMessageBox::No) {
                return;
            }
        }

        ViewProperties props(TQDir::homeDirPath());
        props.setViewMode(m_viewProps->viewMode());
        props.setSorting(m_viewProps->sorting());
        props.setSortOrder(m_viewProps->sortOrder());
        props.setShowHiddenFilesEnabled(m_viewProps->isShowHiddenFilesEnabled());
        props.setValidForSubDirs(true);
    }
    else if (m_applyToSubFolders->isChecked() && m_isDirty) {
        const TQString text(i18n("The view properties of all sub folders will be replaced. Do you want to continue?"));
        if (KMessageBox::questionYesNo(this, text) == KMessageBox::No) {
            return;
        }
    }

    m_viewProps->save();
    m_dolphinView->setViewProperties(*m_viewProps);
    m_isDirty = false;
}

#include "viewpropertiesdialog.moc"
