/*
 *  Copyright (C) 2001-2003, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef KJSEMBEDJSCONSOLEWIDGET_H
#define KJSEMBEDJSCONSOLEWIDGET_H

#include <tqframe.h>
#include <kjsembed/global.h>

class TQPushButton;
class TQHBox;

class KLineEdit;
class TDEPopupTitle;
class TDEProcess;
class KShellProcess;
class KTextEdit;

namespace KJS {
    class Value;
}

namespace KJSEmbed {

class KJSEmbedPart;

/**
 * A TQWidget that provides a console for executing Javascript commands. Creating
 * a JS console is easy, as you can see below:
 *
 * <pre>
 *   KJS::Object global( new KJS::ObjectImp() );
 *   KJS::Interpreter *js = new KJS::Interpreter( global );
 *   KJSEmbed::JSConsoleWidget *win = new KJSEmbed::JSConsoleWidget( js );
 *   win->addBindings( js->globalExec(), global );
 * </pre>
 *
 * This example creates a console for a JS interpreter and adds a print function
 * to the interpreter.
 *
 * @version $Id$
 * @author Richard Moore, rich@kde.org
 */
class KJSEMBED_EXPORT JSConsoleWidget : public TQFrame
{
    TQ_OBJECT

public:
    JSConsoleWidget( KJSEmbedPart *js, TQWidget *parent=0, const char *name=0 );
    virtual ~JSConsoleWidget();

public slots:
    /** Returns the KJSEmbedPart the console is using. */
    KJSEmbed::KJSEmbedPart *jscript() const { return js; }

    /** Returns the message widget. */
    KTextEdit *messages() const { return log; }

    /** Returns the title widget. */
    TDEPopupTitle *title() const { return ttl; }

    /**
     * Returns the TQHBox used to layout the entry part of the console. This
     * can be used by clients to hide and show the interactive parts of the
     * console, or to add new buttons etc.
     */
    TQHBox *commandBox() const { return cmdBox; }

    /** Invokes the content of the command entry field. */
    void invoke();

    /** Invokes the specified command string. */
    virtual bool execute( const TQString &cmd );

    bool execute( const TQString &cmd, const KJS::Value &self );

    /** Prints the specified string to the console. */
    virtual void println( const TQString &text );

    /**
     * Prints the specified string to the console as a warning, the default
     * implementation prints the text in red.
     */
    virtual void warn( const TQString &text );

    /**
     * Runs the specified command using KShellProcess. The output of the
     * command is displayed in the console. Any output sent to stdout is
     * displayed as normal text, anything sent to stderr is displayed as
     * a warning. Once the command has finished the exit status is printed
     * if it is non-zero.
     *
     * Note that there is no extra layer of security to prevent this method
     * being called from a script because in general a script that can
     * access this object can also write content to a file then execute the
     * file. If you intend to run untrusted scripts, it is your responsibility
     * to ensure that you only expose safe objects and methods to the
     * interpreter.
     */
    virtual bool run( const TQString &shellCmd );

protected:
    /** Creates the console view. */
    void createView();

protected slots:
    /** Called when the process exits. */
    void childExited();

    /** Called when the process sends message to stdout. */
    void receivedStdOutput(TDEProcess *, char *, int);

    /** Called when the process sends message to stderr. */
    void receivedStdError(TDEProcess *, char *, int);

private:
    KJSEmbedPart *js;
    KShellProcess *proc;

    KTextEdit *log;
    TQHBox *cmdBox;
    KLineEdit *cmd;
    TQPushButton *go;
    TDEPopupTitle *ttl;

    class JSConsoleWidgetPrivate *d;
};

} // namespace KJSEmbed

#endif // KJSEMBEDJSCONSOLEWIDGET_H
