/*
 * Copyright (C) 2021 Red Hat Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "config.h"

#include "backends/native/meta-virtual-monitor-native.h"

#include "backends/native/meta-crtc-mode-virtual.h"
#include "backends/native/meta-crtc-virtual.h"
#include "backends/native/meta-output-virtual.h"

struct _MetaVirtualMonitorNative
{
  MetaVirtualMonitor parent;

  uint64_t id;
};

static uint64_t mode_id = 1;

G_DEFINE_TYPE (MetaVirtualMonitorNative, meta_virtual_monitor_native,
               META_TYPE_VIRTUAL_MONITOR)

static void
meta_virtual_monitor_native_set_modes (MetaVirtualMonitor *virtual_monitor,
                                       GList              *mode_infos)
{
  MetaOutput *output = meta_virtual_monitor_get_output (virtual_monitor);
  MetaCrtcMode **modes;
  GList *l;
  int n_modes = 0;

  modes = g_new0 (MetaCrtcMode *, g_list_length (mode_infos));
  for (l = mode_infos; l; l = l->next)
    {
      MetaVirtualModeInfo *mode_info = l->data;

      modes[n_modes++] =
        META_CRTC_MODE (meta_crtc_mode_virtual_new (mode_id++, mode_info));
    }

  meta_output_update_modes (output, modes[0], modes, n_modes);
}

uint64_t
meta_virtual_monitor_native_get_id (MetaVirtualMonitorNative *virtual_monitor_native)
{
  return virtual_monitor_native->id;
}

static MetaCrtcModeVirtual *
create_virtual_crtc_mode_cb (MetaVirtualModeInfo *mode_info)
{
  return meta_crtc_mode_virtual_new (mode_id++, mode_info);
}

MetaVirtualMonitorNative *
meta_virtual_monitor_native_new (MetaBackend                  *backend,
                                 uint64_t                      id,
                                 const MetaVirtualMonitorInfo *info)
{
  MetaVirtualMonitorNative *virtual_monitor_native;
  MetaCrtcVirtual *crtc_virtual;
  g_autolist (MetaCrtcModeVirtual) crtc_modes = NULL;
  MetaOutputVirtual *output_virtual;

  crtc_virtual = meta_crtc_virtual_new (backend, id);
  crtc_modes = g_list_copy_deep (info->mode_infos,
                                 (GCopyFunc) create_virtual_crtc_mode_cb,
                                 NULL);
  output_virtual = meta_output_virtual_new (id, info,
                                            crtc_virtual,
                                            crtc_modes);

  virtual_monitor_native = g_object_new (META_TYPE_VIRTUAL_MONITOR_NATIVE,
                                         "crtc", crtc_virtual,
                                         "output", output_virtual,
                                         NULL);
  virtual_monitor_native->id = id;

  return virtual_monitor_native;
}

static void
meta_virtual_monitor_native_init (MetaVirtualMonitorNative *virtual_monitor_native)
{
}

static void
meta_virtual_monitor_native_class_init (MetaVirtualMonitorNativeClass *klass)
{
  MetaVirtualMonitorClass *virtual_monitor_class = META_VIRTUAL_MONITOR_CLASS (klass);

  virtual_monitor_class->set_modes = meta_virtual_monitor_native_set_modes;
}
